/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

class Theme2DFile : Theme2D {
    private string[] filename_alts = {"red", "blue", "black", "white", "table|board"};
    private string[] filetypes = {".png", ".svg"};

    // Set true for old themes that doesn't support Layout
    private bool old_theme = true;

    public Theme2DFile () {
        // Themes found in $datadir/pasang-emas
        add_themes_in_dir (Reloc.theme_root_dir);
        // Themes found in $home/.pasang-emas
        add_themes_in_dir (Path.build_filename (Environment.get_home_dir (), Config.USER_THEME_ROOT_DIR, null));
    }

    /**
     * Get all directories inside the given one. Assume these to be themes.
     * Side effect: items
     */
    private void add_themes_in_dir (string parent_directory) {
        try {
            var theme_dir = Dir.open (parent_directory);
            while (true) {
                var dir = theme_dir.read_name ();
                if (dir == null) break;
                var path = Path.build_filename (parent_directory, dir, null);
                if (FileUtils.test (path, FileTest.IS_DIR) && image_files_found (path)) {
                    var item = ThemeItem () {name=dir, path=path, engine=this};
                    if (add_theme_info (path, ref item)) {;
                        items += item;
                    }
                }
            }
        }
        catch (Error e) {
            stderr.printf ("File error: %s\n", e.message);
        }
    }

    /**
     * Check if the directory contains all the required image files.
     */
    private bool image_files_found (string path) {
        foreach (var filename_alt in filename_alts) {
            string[] filenames = filename_alt.split ("|");
            // found = a file "<any recognized name><any recognized extension>" is found
            var found = false;
            foreach (var filename in filenames) {
                foreach (var filetype in filetypes) {
                    var filepath = Path.build_filename (path, filename + filetype, null);
                    if (FileUtils.test (filepath, FileTest.EXISTS)) {
                        found = true;
                        break;
                    }
                }
                if (found) break;
            }
            if (!found) return false;
        }
        return true;
    }

    /**
     * Read theme.prop info into ThemeItem.
     * Return false if the theme is malformed.
     */
    private bool add_theme_info (string path, ref ThemeItem item) {
        var filepath = Path.build_filename (path, "theme.prop", null);
        if (FileUtils.test (filepath, FileTest.EXISTS)) {
            try {
                var kf = new KeyFile ();
                kf.load_from_file (filepath, KeyFileFlags.NONE);
                get_kf_string (kf, "Info", "Name", ref item.name);
                get_kf_string (kf, "Info", "Artist", ref item.artist);
                get_kf_string (kf, "Info", "License", ref item.license);
                get_kf_string (kf, "Info", "Comment", ref item.comment);
            }
            catch (Error e) {
                stderr.printf ("Info error while reading %s: %s\n", filepath, e.message);
                return false;
            }
        }

        // Select the theme for the purpose of peeking at its detail
        if (select (path, 0, 0) == null) return false;

        // Set item.pixbuf
        var size = 48;
        var p = Piece.KAS_1;
        var frame_size = pixbufs[p].get_height () / num_rows_in_image[p];
        var num_cols = pixbufs[p].get_width () / frame_size;
        var y = layout.kas_frame[p] / num_cols * frame_size;
        var x = layout.kas_frame[p] % num_cols * frame_size;
        // Clip the required frame
        var pixbuf = new Gdk.Pixbuf.subpixbuf (pixbufs[p], x, y, frame_size, frame_size);
        // Scale.
        item.pixbuf = pixbuf.scale_simple(size, size, Gdk.InterpType.BILINEAR);
        return true;
    }

    protected Gdk.Pixbuf[] pixbufs = new Gdk.Pixbuf[Piece.COUNT + 1];  

    public override void free () {
        base.free ();
        for (int i=0; i < pixbufs.length; i++) pixbufs[i] = null;
    }

    /**
     * Load theme files.
     * If width == 0, then we are actually only peeking for the purpose
     * of populating the theme menu.
     * Side-effect: pixbufs[], also imges[] through resize
     */
    public override Theme? select (string path, int width, int height) {
        // Defaults
        time_span = 1.3;
        for (int i=0; i < Piece.COUNT; i++) num_rows_in_image[i] = 1;
        layout = default_layout;

        layout_changed = true;
        for (int i=0; i < filename_alts.length; i++) {
            cycles[i] = 0;
            pixbufs[i] = load_as_pixbuf (path, filename_alts[i]);
            if (pixbufs[i] == null) return null;
        }

        // Process optional theme.prop file
        old_theme = true;
        get_kf_layout (null);
        var filepath = Path.build_filename (path, "theme.prop", null);
        if (FileUtils.test (filepath, FileTest.EXISTS)) {
            try {
                var kf = new KeyFile ();
                kf.load_from_file (filepath, KeyFileFlags.NONE);
                get_kf_integer_list (kf, "View", "Repeat", ref cycles);
                if (kf.has_key ("View", "Span")) {
                    var val = kf.get_double ("View", "Span");
                    if (val > 1 && val <= 5) time_span = val;
                }
                get_kf_integer_list (kf, "View", "Rows", ref num_rows_in_image);
                if (kf.has_group ("Layout")) old_theme = false;
                get_kf_layout (kf);
            }
            catch (Error e) {
                stderr.printf ("File error while reading %s: %s\n", filepath, e.message);
            }
        }

        if (width != 0) {
            resize (width, height);
        }
        return this;
    }

    /**
     * Load <file name + any recognized extension> and return as a Pixbuf
     */
    private Gdk.Pixbuf? load_as_pixbuf (string path, string filename_alt) {
        try {
            string[] filenames = filename_alt.split ("|");
            foreach (var filename in filenames) {
                foreach (var filetype in filetypes) {
                    var filepath = Path.build_filename (path, filename + filetype, null);
                    if (FileUtils.test (filepath, FileTest.EXISTS)) {
                        return new Gdk.Pixbuf.from_file (filepath);
                    }
                }
            }
        }
        catch (Error e) {
            stderr.printf ("File error: %s\n", e.message);
            free ();
        }
        return null;
    }

    /**
     * Read the layout from the KeyFile. If kf is null or values are not available,
     * simply set them to some defaults.
     */
    private void get_kf_layout (KeyFile? kf) {
        try {
            var image_width = pixbufs[Piece.BOARD].get_width ();
            var image_height = pixbufs[Piece.BOARD].get_height ();

            // Interpret Layout-Table
            // Default: use the entire image
            int[] table = {0, 0, image_width, image_height};
            get_kf_integer_list (kf, "Layout", "Table", ref table);
            layout.filler_left = table[0];
            layout.filler_top = table[1];
            layout.filler_right = image_width - table[2];
            layout.filler_bottom = image_height - table[3];
            layout.width = table[2] - table[0] + 1;
            layout.height = table[3] - table[1] + 1;

            // Grid size. This is tentative.
            // It will finally depend on layout.board_size.
            var grid_size = (double) layout.height / BOARD_WIDTH;

            // Interpret Layout-Grid
            // Default: use the entire (always visible) table height for the board
            var corner1x = (int)(1.5 * grid_size) + layout.filler_left;
            var corner1y = (int)(1.5 * grid_size) + layout.filler_top;
            var corner2x = (int)(11.5 * grid_size) + layout.filler_left;
            int[] grid = {corner1x, corner1y, corner2x};
            get_kf_integer_list (kf, "Layout", "Grid", ref grid);
            layout.board_size = (grid[2] - grid[0] + 1) * BOARD_WIDTH / (BOARD_WIDTH - 3);
            grid_size = (double) layout.board_size / BOARD_WIDTH;
            layout.board_x = (int)(grid[0] - 1.5 * grid_size) - layout.filler_left;
            layout.board_y = (int)(grid[1] - 1.5 * grid_size) - layout.filler_top;

            // Interpret Layout-Wins
            // Default: to the right of the board
            var x = layout.board_x + (int)(1.1 * layout.board_size) + layout.filler_left;
            var y0 = layout.board_y + (int)(0.5 * grid_size) + layout.filler_top;
            var y1 = layout.board_y + (int)(12.5 * grid_size) + layout.filler_top;
            int[] pt = {x, y0, x, y1};
            get_kf_integer_list (kf, "Layout", "Wins", ref pt);
            layout.num_wins_x = {pt[0] - layout.filler_left,  pt[2] - layout.filler_left};
            layout.num_wins_y = {pt[1] - layout.filler_top,   pt[3] - layout.filler_top};

            // Interpret Layout-WinsFont
            // Default: 0.7 grid size, opaque black
            int[] font = {(int)(0.7 * grid_size), 0, 0, 0, 255};
            get_kf_integer_list (kf, "Layout", "WinsFont", ref font);
            layout.num_wins_font_size = font[0];
            layout.num_wins_color = Rgb (font[1] / 255.0, font[2] / 255.0, font[3] / 255.0, font[4] / 255.0);

            // Interpret Layout-Score
            // Default: displace vertically from Wins
            pt[1] += (int)(3 * grid_size);
            pt[3] -= (int)(3 * grid_size);
            get_kf_integer_list (kf, "Layout", "Score", ref pt);
            layout.score_x = {pt[0] - layout.filler_left,  pt[2] - layout.filler_left};
            layout.score_y = {pt[1] - layout.filler_top,   pt[3] - layout.filler_top};

            // Interpret Layout-WinsFont
            // Default: 0.5 grid size, opaque black (for both scores)
            font = {(int)(0.5 * grid_size), 0, 0, 0, 255,  0, 0, 0, 255};
            get_kf_integer_list (kf, "Layout", "ScoreFont", ref font);
            layout.score_font_size = font[0];
            layout.score_color = {Rgb (font[1] / 255.0, font[2] / 255.0, font[3] / 255.0, font[4] / 255.0),
                                  Rgb (font[5] / 255.0, font[6] / 255.0, font[7] / 255.0, font[8] / 255.0)};

            // Interpret Layout-Kas
            // Default: displace vertically from Score
            pt[1] += (int)(grid_size);
            pt[3] -= (int)(grid_size);
            get_kf_integer_list (kf, "Layout", "Kas", ref pt);
            layout.kas_x = {pt[0] - layout.filler_left,  pt[2] - layout.filler_left};
            layout.kas_y = {pt[1] - layout.filler_top,   pt[3] - layout.filler_top};

            // Interpret Layout-KasFrame
            // Default: 0 0
            int[] kas_frame = {0, 0};
            get_kf_integer_list (kf, "Layout", "KasFrame", ref kas_frame);
            layout.kas_frame = kas_frame;

            // Interpret Layout-Round
            // Default: between the 2 kas, format = 0
            pt = {(pt[0] + pt[2]) / 2, (pt[1] + pt[3]) / 2, 0};
            get_kf_integer_list (kf, "Layout", "Round", ref pt);
            layout.num_rounds_x = pt[0] - layout.filler_left;
            layout.num_rounds_y = pt[1] - layout.filler_top;
            layout.num_rounds_format = pt[2];

            // Interpret Layout-RoundFont
            // Default: 0.3 grid size, opaque black (for both scores)
            font = {(int)(0.3 * grid_size), 0, 0, 0, 255,  0, 0, 0, 255};
            get_kf_integer_list (kf, "Layout", "RoundFont", ref font);
            layout.num_rounds_font_size = font[0];
            layout.num_rounds_color = Rgb (font[1] / 255.0, font[2] / 255.0, font[3] / 255.0, font[4] / 255.0);

            // Correction for old themes
            if (old_theme) layout.width = 120 * layout.width / 100;
        }
        catch (Error e) {
            stderr.printf ("KeyFile error: %s\n", e.message);
        }
    }

    /**
     * Read the KeyFile if the given key is available. Otherwise leave the output untouched.
     */
    private void get_kf_string (KeyFile kf, string group, string key, ref string? output)
    throws KeyFileError {
        if (kf.has_group (group) && kf.has_key (group, key))
            output = kf.get_locale_string (group, key, null);
    }

    private void get_kf_integer_list (KeyFile? kf, string group, string key, ref int[] output)
    throws KeyFileError {
        if (kf != null && kf.has_group (group) && kf.has_key (group, key)) {
            var vals = kf.get_integer_list (group, key);
            var n = int.min (vals.length, output.length);
            for (int i=0; i < n; i++) output[i] = vals[i];
        }
    }

    /**
     * Scale pixbuf images previously loaded from files.
     * Use scale factor current_size / size of board image.
     * Side-effect: images[]
     */
    public override void resize_images () {
        for (int i=0; i < Piece.COUNT + 1; i++) {
            int width, height;
            if (i == Piece.BOARD) {
                width = old_theme ? table.image_height : table.image_width;
                height = table.image_height;
            }
            else {
                // Scale pieces so that the animation strip is evenly divisible by the zoomed frame size
                int num_cols = pixbufs[i].get_width () / (pixbufs[i].get_height () / num_rows_in_image[i]);
                height = (int) (pixbufs[i].get_height () * table.board_size / layout.board_size);
                width = (height / num_rows_in_image[i]) * num_cols;
            }
            // Use Gdk rather than Cairo to scale
            var pixbuf = pixbufs[i].scale_simple(width, height, Gdk.InterpType.BILINEAR);
            // Pre-convert Gdk format to Cairo format
            // I could have used create_from_png directly, but I want to support other file formats also
            images[i] = new Cairo.ImageSurface (Cairo.Format.ARGB32, width, height);
            var cr = new Cairo.Context (images[i]);
            Gdk.cairo_set_source_pixbuf (cr, pixbuf, 0.0, 0.0);
            cr.paint();
        }
    }
}//class
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
