/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2018  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

class ClientConnection {
    private static ClientConnection[] connections = {};
    private static long next_id = 0;

    public static void add_connection (SocketConnection connection) {
        var cc = new ClientConnection (connection, next_id++);
        // Recycle...
        for (int i=0; i < connections.length; i++) {
            if (connections[i].connection == null) {
                connections[i] = cc;
                cc.run.begin ();
                return;
            }
        }
        // ...or append
        connections += cc;
        cc.run.begin ();
    }

    private Cancellable cancellable;
    private SocketConnection connection;
    private DataInputStream input;
    private DataOutputStream output;
    private long id;

    /**
     * The server also keeps the state of the game. So the clients
     * cannot do any funny business.
     */
    private Game game = null;

    /**
     * Set to non-null upon a valid login
     */
    private string user_name = null;

    /**
     * Set to non-null when after a successful challenge
     */
    private ClientConnection opponent = null;

    private ClientConnection (SocketConnection con, long id) {
        connection = con;
        input = new DataInputStream (connection.input_stream);
        output = new DataOutputStream (connection.output_stream);
        this.id = id;
        cancellable = new Cancellable ();
    }

    private async void run () {
        message ("Listening to #%ld", id);
        try {
            string line;
            size_t length;
            while ((line = yield input.read_line_async (100, cancellable, out length)) != null) {
                message ("From #%ld: %s", id, line);
                interpret (sanitize (line));
                if (cancellable.is_cancelled ()) break;
            }
        }
        catch (Error e) {
            stderr.printf ("IO error in line #%ld: %s\n", id, e.message);
        }
        usher_out ();
        try {
            connection.close (null);
        }
        catch (Error e2) {
            stderr.printf ("Disconnection error in line #%ld: %s\n", id, e2.message);
        }
        cancellable.cancel ();
        connection = null;
        message ("Disconnecting #%ld", id.abs ());
    }

    private string sanitize (string s) {
        return s.strip().replace ("\n", "");
    }

    /**
     * true if con refers to another connection (not self)
     */
    private bool is_others (ClientConnection con) {
        return con.id != id && !con.cancellable.is_cancelled () && con.user_name != null;
    }

    /**
     * return "USER id status name"
     *   e.g. USER 6 w mat bon
     * where status is either w (waiting) or p (playing)
     */
    private string status () {
        return "USER %ld %c %s\n".printf (id, opponent == null ? 'w' : 'p', user_name);
    }

    /**
     * Send message to other connections other than self.
     */
    private void broadcast (string message) {
        foreach (var con in connections) {
            if (is_others (con)) con.send (message);
        }
    }

    /**
     * Errorless put_string.
     * Error is caught here. Reason: in a broadcast, a disconnected client shouldn't cause
     * the broadcast to be aborted and fail to reach other clients.
     */
    private void send (string message) {
        try {
            output.put_string (message, null);
        }
        catch (Error err) {
            stderr.printf ("Send error in line #%ld: %s\n", id, err.message);
        }
    }

    private void interpret (string line) {
        if (line == "") return;
        var cmd = line.split (" ", 2);
        var command = cmd[0];
        var arg = cmd.length == 2 ? cmd[1].strip () : "";
        switch (command) {
            case "LOGIN" : login (arg); break;
            case "LOGOUT" : logout (); break;
            case "GET-USER-LIST" : get_user_list (); break;
            case "CHALLENGE" : challenge (long.parse (arg)); break;
            case "LEAVE" : leave (); break;
            case "CHAT" : chat (arg); break;
            case "MOVE" : move (arg); break;
        }
    }

    private void login (string name) {
        if (name != "") {
            usher_out ();
            user_name = name;
            send ("LOGIN-OK %ld %s\n".printf (id, name));
            broadcast ("USER %ld w %s\n".printf (id, user_name));
        }
        else {
            send ("LOGIN-ERROR\n");
        }
    }

    private void logout () {
        usher_out ();
        cancellable.cancel ();
    }

    /**
     * A player is ushered out after 1. logout, 2. lost connection.
     */
    private void usher_out () {
        if (user_name == null) return;
        leave ();   // break any ongoing match
        broadcast ("EXIT %ld\n".printf (id));
        user_name = null;
    }

    private void get_user_list () {
        if (user_name == null) return;
        foreach (var con in connections) {
            if (is_others (con)) send (con.status ());
        }
    }

    /**
     * When a challenge is issued, the challengee must accept it (unless
     * if she is already in the middle of a game).
     */
    private void challenge (long opponent_id) {
        if (user_name == null) return;
        if (opponent != null) return;  // This may happen when the challenger has just been challenged
        foreach (var con in connections) {
            if (is_others (con) && con.id == opponent_id && con.opponent == null) {
                send ("PLAY-AGAINST %ld\n".printf (con.id));
                con.send ("PLAY-AGAINST %ld\n".printf (id));
                opponent = con;
                opponent.opponent = this;
                broadcast (status ());
                opponent.broadcast (opponent.status ());
                game = new Game ();
                opponent.game = game;
                break;
            }
        }
    }

    /**
     * The user cannot challenge another opponent until she LEAVE her current game.
     */
    private void leave () {
        if (user_name == null) return;
        if (opponent == null) return;
        opponent.opponent = null;
        opponent.send ("PLAY-END\n");
        opponent.broadcast (opponent.status ());
        game = null;
        opponent.game = null;
        opponent = null;
        send ("PLAY-END\n");
        broadcast (status ());
    }

    /**
     * Broadcast chit-chat message.
     */
    private void chat (string msg) {
        if (user_name == null) return;
        broadcast ("MESSAGE %ld %s\n".printf (id, msg));
    }

    /**
     * Relay move to the opponent.
     * Also, echo the move back.
     */
    private void move (string arg) {
        if (user_name == null) return;
        if (opponent == null) return;
        var arg_split = arg.split (" ", 2);
        if (arg_split.length != 2) {
            message ("Error: Malformed move");
            return;
        }
        var seq = int.parse (arg_split[0]);
        var move_notation = arg_split[1];
        if (seq == 0) {
            if (game.stage != Stage.NULL && game.stage != Stage.GAME_OVER || !Game.is_valid_pattern (move_notation)) {
                message ("Invalid move: %s", arg);
                return;
            }
            game.start (move_notation);
        }
        else if (seq == game.seq + 1) {
            var move = game.get_move (move_notation);
            if (move == null) {
                message ("Invalid move: %s", arg);
                return;
            }
            game.perform_from_notation (move_notation);
        }
        else {
            message ("Invalid move: %s", arg);
            return;
        }
        opponent.send ("MOVE %ld %s\n".printf (id, arg));
        send ("MOVE-POSTED %s\n".printf (arg));
    }
}//class ClientConnection

class Server {
    MainLoop loop;

    /**
     * Entry point for Server
     */
    public static void main (uint16 port) {
        stdout.printf ("%s\n", Config.PACKAGE_STRING);
        stdout.printf ("Connection port: %d\n", port);
        new Server (port);
    }

    public Server (uint16 port) {
        try {
            var service = new SocketService ();
            service.add_inet_port (port, null);
            service.incoming.connect (on_connection);
            service.start ();
            loop = new MainLoop (null, false);
            loop.run ();
        } catch (Error e) {
            error ("Socket error: %s\n", e.message);
        }
    }

    private bool on_connection (SocketConnection connection, Object? src) {
        ClientConnection.add_connection (connection);
        return true;
    }
}//class Server

}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
