/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

enum Player {
    AI,
    HUMAN,
    REMOTE
}  

struct Point {
    public int x;
    public int y;
    public Point.xy (int x, int y) {this.x = x; this.y = y;}
}

class GameView : Gtk.DrawingArea {
    /**
     * Emitted whenever a move has been completed.
     * move = null means "pass", i.e., let the other player make
     * the first move.
     */
    public signal void next_move_signal (Move? move);

    public GameSeries game {get; private set;}

    public ThemeSwitch theme_switch {get; private set; default = new ThemeSwitch ();}

    public Theme theme {get; private set; default = null;}

    /**
     * Position of kas being dragged. If not dragged, x = -1
     */
    public Point kas_point = Point ();

    /**
     * Move that matches the user (or simulated) gesture.
     */
    public Move? selected_move {get; private set; default = null;}

    /**
     * Message that depends on the state of the game.
     */
    public string? message {get; private set; default = null;}

    /**
     *  Set by set_turn()
     */
    private Player[] player_types = new Player[2];

    /**
     * Position of pointer relative to kas_point.
     */
    private Point drag_point = Point ();

    /**
     * When a gesture is completed (typically press, drag, release), the move
     * is not immediately performed. It is first "submitted", the pieces to be captured
     * are then visually "lifted". Only then will the game state be updated.
     */
    private bool submitting_move;

    private Gtk.GestureDrag gesture;
    private Gtk.EventControllerMotion event_controller;

    public GameView (GameSeries game) {
        this.game = game;
        set_turn (Player.AI, Player.HUMAN, 1);
        draw.connect ((cr) => {
            theme.draw (this, cr);
            return true;
        });
        size_allocate.connect (theme_switch.resize);
        theme_switch.changed.connect (() => {
            theme = theme_switch.theme;
            queue_draw ();
        });

        // Non-dragging motion
        add_events (Gdk.EventMask.POINTER_MOTION_MASK);
        event_controller = new Gtk.EventControllerMotion (this);
        event_controller.motion.connect ((x, y) => {
            if (input_allowed) hover (Point.xy ((int)x, (int)y));
        });

        // Drag gesture
        gesture = new Gtk.GestureDrag (this);
        gesture.set_propagation_phase (Gtk.PropagationPhase.BUBBLE);
        gesture.set_button (1);
        gesture.drag_begin.connect ((x, y) => {
            if (input_allowed) drag_start (Point.xy ((int)x, (int)y));
        });
        gesture.drag_update.connect ((x, y) => {
            double sx, sy;
            gesture.get_start_point (out sx, out sy);
            if (input_allowed) drag_update (Point.xy ((int)(sx + x), (int)(sy + y)));
        });
        gesture.drag_end.connect ((x, y) => {
            double sx, sy;
            gesture.get_start_point (out sx, out sy);
            if (input_allowed) drag_stop (Point.xy ((int)(sx + x), (int)(sy + y)));
        });

        theme = theme_switch.theme;
        init_state ();
    }

    private void init_state () {
        selected_move  = null;
        submitting_move = false;
        kas_point.x = -1;
        theme.drop_pieces ();
        theme.drop_kas (game);
        queue_draw ();
    }

    public void start_game (string starting_pattern) {
        game.start_recorded (starting_pattern);
        init_state ();
        hang_message ();
    } 

    /**
     * top_player: Player who "sits" at the top
     * bottom_player: Player who "sits" at the bottom
     * first_player: who plays first, top_player (0) or bottom_player (1)
     */
    public void set_turn (Player top_player, Player bottom_player, int first_player)
        requires (first_player == 0 || first_player == 1) {
        game.rotated = first_player == 1;
        player_types[first_player] = top_player;
        player_types[1 - first_player] = bottom_player;
        hang_message ();
    }

    /**
     * Set a suitable message to display depending on the
     * state of the game
     */
    public void hang_message () {
        if (game.score[0] == 0 && game.score[1] == 0) {
            if (player_types[1] == Player.AI && player_types[0] == Player.HUMAN)
                message = _("Press this side to let\nthe machine move first.");
            else if (player_types[0] == Player.REMOTE || player_types[1] == Player.REMOTE)
                message = _("Playing online");
            else
                message = null;
        }
        else if (game.stage == Stage.GAME_OVER) {
            if (player_types[0] == Player.HUMAN || player_types[1] == Player.HUMAN)
                message = _("Game over.\nClick for\nanother round.");
            else
                message = _("Game over");
        }
        else {
            message = null;
        }
        queue_draw ();
    }

    /**
     * Who is playing next turn: Player.HUMAN, or Player.AI, or ...?
     */
    public Player player_type {
        get {return player_types[game.player];}
    }

    /**
     * Who is playing next turn: the player sitting at the top (0) or the one at the bottom (1)?
     */
    public int player_side {
        get {return (!game.rotated ? game.player : 1 - game.player);}
    }

    public void queue_draw_piece (int position) {
        theme.queue_draw_piece (this, game.board[position], to_point (position));
    }

    private bool input_allowed {
        get {
            return player_type == Player.HUMAN && !submitting_move
                // Allow input when the game is over so that a null move can be emitted
                // to start the next round.
                || player_type == Player.REMOTE && game.stage == Stage.GAME_OVER;
        }
    }
     
    private void hover (Point point) {
        animate.begin ();
        // For a fresh board, in a 2-player face-to-face game, allow any side to play
        if (game.score[0] == 0 && game.score[1] == 0) {
            if (player_types[1] == Player.HUMAN && player_types[0] == Player.HUMAN) {
                var prev_rotated = game.rotated;
                game.rotated = theme.to_side (point) == 1;
                // The theme doesn't know about the change yet, so simply redraw the whole thing
                if (prev_rotated != game.rotated) queue_draw ();
            }
        }
        switch (game.stage) {
        case Stage.OPENING :
            theme.drop_pieces ();
            theme.pick_booties (game, find_move (point), false);
            break;
        case Stage.SUB_SELECT :
        case Stage.SUB_MOVE : 
            if (find_move (point) != null) {
                theme.drop_pieces ();
                theme.pick_booties (game, selected_move, false);
            }
            break;
        }
    }

    private void drag_update (Point point) {
        switch (game.stage) {
        case Stage.SELECT : 
            drag_candidate (point);
            break;
        case Stage.MOVE :
            drag_kas (point);
            break;
        }
    }

    private void drag_start (Point point) {
        hover (point);
        if (game.score[0] == 0 && game.score[1] == 0) {
            if (player_types[1] == Player.AI && player_types[0] == Player.HUMAN) {
                if (theme.to_side (point) == 0) {
                    // The user pressed the TOP side of the board.
                    // So, let the machine move first by signaling a null move.
                    next_move_signal (null);
                    return;
                }
            }
        }
        switch (game.stage) {
        case Stage.OPENING :
        case Stage.SUB_SELECT :
        case Stage.SUB_MOVE : 
            if (find_move (point) != null) {
                game.leave_one_move (selected_move);
                submit_selected_move ();
            }
            break;
        case Stage.SELECT : 
            pick_candidate (point);
            break;
        case Stage.MOVE :
            pick_kas (point);
            break;
        case Stage.GAME_OVER :
            next_move_signal (null);
            break;
        }
    }
     
    private void drag_stop (Point point) {
        switch (game.stage) {
        case Stage.SELECT :
            drop_candidate (point);
            break;
        case Stage.MOVE :
            drop_kas (point);
            break;
        }
    }

    /**
     * Find the move that captures the piece at the given screen location.
     * Side effect: set selected_move
     */
    private Move? find_move (Point point) {
        int position = to_position (point);
        foreach (Move move in game) {
            foreach (var pos in move.booties) {
                if (position == pos) return selected_move = move;
            }
        }
        return selected_move = null;
    }

    /**
     * Select a piece to be promoted to kas. The candidate must be within
     * the vicinity of the given point.
     * Side effect: selected_move, kas_point, drag_point
     */
    private void pick_candidate (Point point) {
        selected_move = null;
        int position = to_position (point);
        foreach (var move in game) {
            if (position == move.position) {
                selected_move = move;
                kas_point = to_point (position);
                drag_point = Point.xy (kas_point.x - point.x,  kas_point.y - point.y);
                theme.pick_kas (game);
                break;
            }
        }
    }

    /**
     * If there was a selected candidate, drag it. If it goes to the
     * empty column (which was captured during the opening move), highlight 
     * all its possible captures.
     * Side effect: kas_point
     * Return: true if the candidate is dragged home
     */
    private bool drag_candidate (Point point) {
        if (selected_move == null) return false;
        kas_point = Point.xy (point.x + drag_point.x, point.y + drag_point.y);
        if (kas_point.x == -1) kas_point.x = 0;  // Because -1 means "not dragged"
              
        // See if the kas is dragged to its proper place
        int pos = to_position (kas_point);
        int row = pos / BOARD_WIDTH;
        int col = pos % BOARD_WIDTH;
        int kas_row      = selected_move.position / BOARD_WIDTH;
        int empty_column = game.first_move[game.player];
        bool dragged_home = (row == kas_row  &&  col == empty_column);
        if (dragged_home) {
            // Show all possible captures
            foreach (Move move in game) {
                if (move.position == selected_move.position) theme.pick_booties (game, move, false);
            }
            // In case the kas is dropped, invalidate its drop square.
            queue_draw_piece (pos);
        }
        else {
            theme.drop_pieces ();
        }
        return dragged_home;
    }
   
    /**
     * Promote the selected candidate to kas, if it is already dragged home.
     * Side effect: selected_move null if kas dropped elsewhere
     */
    private void drop_candidate (Point point) {
        if (selected_move == null) return;
        if (drag_candidate (point)) {
            submit_selected_move ();
        }
        else {
            queue_draw_piece (selected_move.position);  // Redraw candidate on its original position
            selected_move = null;
        }
        theme.drop_kas (game);
    }

    /**
     * Pick the kas if it is being pointed at.
     * Side effect: kas_point, drag_point
     */
    private void pick_kas (Point point) {
        int pos = to_position (point);
        if (pos == game.kas_position[game.player]) {
            kas_point = to_point (pos);
            drag_point = Point.xy (kas_point.x - point.x, kas_point.y - point.y);
            theme.pick_kas (game);
        }
        else {
            kas_point.x = -1;
        }
    }

    /**
     * Drag the kas to follow the user gesture. The kas must be previously picked.
     * Side effect: selected_move
     */
    private void drag_kas (Point point) {
        if (kas_point.x == -1) return;
        kas_point = Point.xy (point.x + drag_point.x, point.y + drag_point.y);
        if (kas_point.x == -1) kas_point.x = 0;  // Because -1 means "not dragged"

        int pos = to_position (kas_point);
        selected_move = null;
        theme.drop_pieces ();
        // See if the kas position is listed in the move list
        foreach (Move move in game) {
            if (move.position == pos) {
                selected_move = move;
                // Show all the possible captures
                foreach (Move m in game) {
                    if (m.position == selected_move.position) theme.pick_booties (game, m, false);
                }
                break;
            }
        }
        theme.pick_kas (game);
    }
        
    /**
     * Drop the kas at the given location.
     */
    private void drop_kas (Point point) {
        if (kas_point.x == -1) return;
        drag_kas (point);   // Just to set selected_move
        if (selected_move != null) {
            submit_selected_move ();
        }
        else {
            kas_point.x = -1;
            queue_draw_piece (game.kas_position[game.player]);
        }
        theme.drop_kas (game);
    }    

    private Point to_point (int pos) {
        return theme.to_point (pos, game.rotated);
    }

    private int to_position (Point p) {
        // If player_type != HUMAN, we must be in the middle of simulating
        // a move (for the computer or for the remote opponent).
        // Hence, be precise in the translation from point to position.
        return theme.to_position (this.game, p, game.rotated, player_type != Player.HUMAN);
    }

    /**
     * To be called only when the user gesture has been completed.
     */
    private void submit_selected_move ()
        requires (selected_move != null) {
        if (game.count_submoves (selected_move) == 1) {
            theme.drop_pieces ();
            theme.pick_booties (game, selected_move, true);
        }
        if (kas_point.x != -1) {
            kas_point = to_point (to_position (kas_point));     // Snap kas
        }
        if (game.count_submoves (selected_move) > 1) {
            // Allow the human player to immediately make a sub move
            perform_selected_move ();
        }
        else {
            submit_selected_move_async.begin ();
        }
    }

    /**
     * Allow time for the captured pieces to be visually lifted before
     * actually performing the move.
     */
    private int submission_ticket;

    private async void submit_selected_move_async () {
        submitting_move = true;
        var my_submission_ticket = ++submission_ticket;
        var lifting_time = (int)(theme.time_span * 1000);
        yield Util.wait_async (lifting_time);
        if (submitting_move && my_submission_ticket == submission_ticket) {
            perform_selected_move ();
        }
        submitting_move = false;
    }

    /**
     * Actually perform the selected move by calling game.perform().
     */
    private void perform_selected_move () {
        var applied_move = selected_move;
        game.perform_recorded (selected_move, player_type != Player.HUMAN);
        theme.drop_kas (game);
        theme.drop_pieces ();
        hang_message ();
        queue_draw ();
        selected_move = null;
        kas_point.x = -1;
        show_sub_moves.begin ();
        next_move_signal (applied_move);
    }

    /**
     * Show all possible captures for SUB_SELECT or SUB_MOVE
     */
    private async void show_sub_moves () {
        int n = 0;
        while (!submitting_move && (game.stage == Stage.SUB_SELECT || game.stage == Stage.SUB_MOVE)) {
            var highlight_time = (int)(theme.time_span * 750);
            if (selected_move == null) {  // Else selected move is already highlighted
                int num_moves = game.numMoves ();
                assert (num_moves > 1);
                theme.drop_pieces ();
                theme.pick_booties (game, game.getMove (n), false);
                n = (n + 1) % num_moves;
            }
            yield Util.wait_async (highlight_time);
        }
    }

    /**
     * Simulate user gesture to effect the given move
     */
    public void simulate (Move move) {
        game.leave_one_move (move);
        int row = 0;
        int col = 0;
        switch (game.stage) {
        case Stage.OPENING :
            row = (game.player == 0) ? 1 : 7;
            simulate_async.begin (move.position + row * BOARD_WIDTH,
                                  move.position + row * BOARD_WIDTH);
            break;
        case Stage.SELECT :
            row = move.position / BOARD_WIDTH;
            col = game.first_move[game.player];
            simulate_async.begin (move.position,
                                  col + row * BOARD_WIDTH);
            break;
        case Stage.MOVE :
            simulate_async.begin (game.kas_position[game.player],
                                  move.position);
            break;
        }
    }

    /**
     * Simulate the drag gesture from press_pos to release_pos
     */
    int motion_ticket;

    private async void simulate_async (int press_pos, int release_pos)
        requires (player_type != Player.HUMAN) {
        var my_motion_ticket = ++motion_ticket;
        var timer = new Timer ();
        var press_point = to_point (press_pos);
        var release_point = to_point (release_pos);

        if (game.stage != Stage.OPENING) drag_start (press_point);
        else hover (press_point);

        var time_to_move = double.max (1.5, theme.time_span);
        while (true) {
            // Recalculate press_point and release_point in case
            // the theme has changed or resized.
            press_point = to_point (press_pos);
            release_point = to_point (release_pos);
            var lapse = timer.elapsed () / time_to_move;
            if (game.stage == Stage.OPENING) {
                if (lapse > 1) {
                    drag_start (press_point);
                    break;
                }
            }
            else {
                // Let the gesture linger a bit at the press point to complete
                // the animation of lifting the kas.
                var time_to_pick_kas = 0.5;
                // How far the piece has moved from press point (distance 0)
                // to release point (distance 1).
                var distance = (lapse - time_to_pick_kas).clamp (0, 1.0);
                drag_update (Point.xy (
                    (int)(press_point.x * (1 - distance) + release_point.x * distance),
                    (int)(press_point.y * (1 - distance) + release_point.y * distance)));
                if (distance == 1.0) {
                    drag_stop (release_point);
                    break;
                }
            }
            yield Util.wait_async (30);
            if (my_motion_ticket != motion_ticket) break;
        }
    }

    /**
     * Call this to prematurely stop simulating user gesture initiated by simulate()
     */
    public void stop_simulation () {
        motion_ticket++;
        init_state ();
    }

    /**
     * Animate until the pieces are static
     */
    bool animating = false;

    private async void animate () {
        if (animating) return;
        animating = true;
        var timer = new Timer ();
        var last_motion = timer.elapsed ();
        while (true) {
            if (theme.animate (this, timer.elapsed () / theme.time_span)) {
                last_motion = timer.elapsed ();
            }
            // Stop when 2 seconds elapsed without any motion
            else if (timer.elapsed () - last_motion > 2.0) {
                break;
            }
            yield Util.wait_async (30);
        };
        animating = false;
    }

    public override Gtk.SizeRequestMode get_request_mode () {
        return Gtk.SizeRequestMode.HEIGHT_FOR_WIDTH;
    }

    public override void get_preferred_width (out int minimum_width, out int natural_width)  {
        minimum_width = BOARD_WIDTH * 25;
        natural_width = BOARD_WIDTH * 40;
    }

    public override void get_preferred_height (out int minimum_height, out int natural_height)  {
        minimum_height = BOARD_WIDTH * 25;
        natural_height = BOARD_WIDTH * 40;
    }

    public override void get_preferred_height_for_width (int width, out int minimum_height, out int natural_height) {
        natural_height = width;
        minimum_height = BOARD_WIDTH * 25;
    }
}//class
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
