/*
 *   unidiags.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <wtt.h>
#include <mwmspcfc.h>
#include <mwmparsi.h>
#include <unidiags.h>
#include <unidiagx.h>

#include <math.h>

/*paulsch*/
/*#include <pbmplus.h>*/

WORD UniDiagsCarrierNegotiationResult(PMWM_DSPINFO pmwmDspInfo);

BOOL UniDiagsReadWord(PMWM_DSPINFO pmwmDspInfo, HMTASK hTask, char* pszLabel, WORD* pWord)
{
   ULONG  ulAddress;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsReadWord entry\n");  }


   if(!dspLabelToAddress(hTask, pszLabel, &ulAddress))
      if(!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                  pWord, 1,
                  DSP_MEMXFER_DATA_READ)) {
         //DPF("Label %s = %Xh", pszLabel, *pWord);
	  {MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsReadWord exit TRUE\n");  }
         return TRUE;
      }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsReadWord exit FALSE\n");  }


   return FALSE;
}


/*------------------------------------------------------------------
 *
 * Clear call status structure before next call.
 *
 *------------------------------------------------------------------*/
BOOL UniDiagsClearCallStatus()
{
  BOOL bRC = TRUE;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsClearCallStatus entry\n");  }


  bRC = mwmPoundUDClearCallStatus(FALSE);
  mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsClearCallStatus exit\n");  }

  return(bRC);

}

/*------------------------------------------------------------------------
 *
 *  Call is being terminated.  Get termination cause.
 *
 *-----------------------------------------------------------------------*/
BOOL UniDiagsFinalCallStatus(PMWM_DSPINFO pmwmDspInfo)
{
   UNIDIAG* pUniDiags;
   KEY6X* pKey6X;
   WORD disccode;
   WORD wData;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsFinalCallStatus entry\n");  }


   if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
      return FALSE;

   pKey6X = &pUniDiags->Key6X;

   pKey6X->Key60 = CauseUnidentified;
   pKey6X->IsValid |= BIT(0);

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "DISCCODE", &disccode))
   {
      switch (disccode)
      {
         case 0:
            pKey6X->Key60 = DteHangupCommand;
            break;
         case 1:
            pKey6X->Key60 = cct108turnedOff;
            break;
         case 2:
            pKey6X->Key60 = AnyKeyAbort;
            break;
         case 3:
         case 0x64:
            pKey6X->Key60 = CarrierLost;
            break;
         case 4:
            pKey6X->Key60 = CallSetupFailTimerExpired;
            break;
         case 5:
            pKey6X->Key60 = LongSpaceDisconnect;
            break;
         case 6:
            if(UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "CSRCODE", &wData))
               switch (wData)
               {
                  case 2:
                     pKey6X->Key60 = ReorderTone;
                     break;
                  case 3:
                     pKey6X->Key60 = EngagedTone;  // busy???
                     break;
               }
            break;
         case 7:
            pKey6X->Key60 = NoDialTone;
            break;
         case 8:
            pKey6X->Key60 = NoModulationinCommon;
            break;
         case 0x0A:
            pKey6X->Key60 = InactivityTimerExpired;
            break;
         case 0x0C:
            mwmPoundUDClearCallStatus(TRUE);
            pKey6X->Key60 = DteResetCommand;
            break;
         case 0x10:
         case 0x11:
         case 0x12:
            pKey6X->Key60 = NoErrorControlEstablished;
            break;
         case 0x13:
            if(UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "MNPCODE", &wData))
               switch (wData)
               {
                  case 0xFE00:
                     pKey6X->Key60 = NoErrorControlEstablished;
                     break;
                  case 0xFEFE:
                  case 0xFEB1:
                     pKey6X->Key60 = ProtocolViolation;
                     break;
                  case 0xFE01:
                  case 0xFE02:
                  case 0xFE03:
                     pKey6X->Key60 = NegotiationFailed;
                     break;
                  case 0xFE04:
                     pKey6X->Key60 = n400exceeded;
                     break;
               }
            break;
         case 0x14:
            if(UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "MNPCODE", &wData))
            {
               switch (wData)
               {
                  case 0xFEFE:
                  case 0xFEB1:
                     pKey6X->Key60 = ProtocolViolation;
                     break;
                  case 0xFE04:
                     pKey6X->Key60 = n400exceeded;
                     break;
                  case 0xFE07:
                     pKey6X->Key60 = DisconnectFrameReceived;
                     break;
                  case 0xFEB0:
                     pKey6X->Key60 = LossOfSynchronization;
                     break;
               }
               if (wData >= 0xFF00)
                  pKey6X->Key60 = DisconnectFrameReceived;
            }
            break;
         case 0x20:
         case 0x22:
            pKey6X->Key60 = NoErrorControlEstablished;
            break;
         case 0x21:
            if(UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "V42CODE", &wData))
               switch (wData)
               {
                  case 0x0003:
                     pKey6X->Key60 = NegotiationFailed;
                     break;
                  default:
                     pKey6X->Key60 = NoErrorControlEstablished;
                     break;
               }
            break;
         case 0x23:
            if(UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "V42CODE", &wData))
               switch (wData)
               {
                  case 0xF006:
                     pKey6X->Key60 = ProtocolViolation;
                     break;
                  case 0xF001:
                  case 0xF00C:
                     pKey6X->Key60 = NegotiationFailed;
                     break;
                  case 0xF003:
                     pKey6X->Key60 = n400exceeded;
                     break;
                  case 0xF008:
                     pKey6X->Key60 = SabmeFrameReceived;
                     break;
               }
            break;
         case 0x24:
            if(UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "V42CODE", &wData))
            {
               switch (wData)
               {
                  case 0xF006:
                     pKey6X->Key60 = ProtocolViolation;
                     break;
                  case 0xF003:
                     pKey6X->Key60 = n400exceeded;
                     break;
                  case 0xF002:
                     pKey6X->Key60 = DisconnectFrameReceived;
                     break;
                  case 0xF005:
                     pKey6X->Key60 = LossOfSynchronization;
                     break;
                  case 0xF00A:
                     pKey6X->Key60 = FrameReject;
                     break;
               }
               if (wData >= 0xFF00)
                  pKey6X->Key60 = DisconnectFrameReceived;
            }
            break;
         case 0x32:
           pKey6X->Key60 = NoErrorControlEstablished;
           break;
         case 0x40:
         case 0x41:
         case 0x42:
         case 0x43:
         case 0x44:
         case 0x45:
         case 0x46:
         case 0x47:
         case 0x48:
         case 0x49:
         case 0x4A:
         case 0x4B:
         case 0x4C:
         case 0x4D:
         case 0x4E:
         case 0x4F:
           pKey6X->Key60 = TrainingFailed;
           break;
         case 0x50:
         case 0x51:
         case 0x52:
         case 0x53:
         case 0x54:
         case 0x55:
         case 0x56:
         case 0x57:
         case 0x58:
         case 0x59:
         case 0x5A:
         case 0x5B:
         case 0x5C:
         case 0x5D:
         case 0x5E:
         case 0x5F:
           pKey6X->Key60 = GstnCleardownReceived;
           break;
         case 0x60:
         case 0x61:
         case 0x62:
         case 0x63:
         case 0x65:
         case 0x66:
         case 0x67:
         case 0x68:
         case 0x69:
         case 0x6A:
         case 0x6B:
         case 0x6C:
         case 0x6D:
         case 0x6E:
         case 0x6F:
           pKey6X->Key60 = RetrainFailed;
           break;

         case 0xFFFF:
            if(UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "CSRCODE", &wData))
               switch (wData)
               {
                  case 1:
                     pKey6X->Key60 = NoDialTone;
                     break;
                  case 2:
                     pKey6X->Key60 = ReorderTone;
                     break;
                  case 3:
                     pKey6X->Key60 = EngagedTone;  // busy???
                     break;
               }
            break;
      }
   }

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsFinalCallStatus exit\n");  }

   return TRUE;
}

/*------------------------------------------------------------------
 *
 * Save carrer events.
 * Call pump to get information.
 * for class 0, called on at command, and at call termination.
 *
 *------------------------------------------------------------------*/
BOOL UniDiagsSaveCarrierEvents(KEY3X* pKey3X, PMWM_DSPINFO pmwmDspInfo)
{

   WORD protocol;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveCarrierEvents entry\n");  }

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "PROT_COL", &protocol)) {
      if (protocol & MWM_STATUS_V34_CONNECT) {

        WORD retrains_requested = 0, retrains_granted = 0;
        WORD temp_count = 0;
        WORD bitratex, bitrater;

        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RETRNINITCNT", &retrains_requested)) {
          pKey3X->Key32 = retrains_requested;
          pKey3X->IsValid |= BIT(2);
        } /* endif */

        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RETRNRESPCNT", &retrains_granted)) {
          pKey3X->Key33 = retrains_granted;
          pKey3X->IsValid |= BIT(3);
        } /* endif */

        pKey3X->Key30 = retrains_requested + retrains_granted;
        pKey3X->IsValid |= BIT(0);

        pKey3X->Key31 = 0;

        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RENGSCNT", &temp_count)) {
          pKey3X->Key31 += temp_count;
          pKey3X->IsValid |= BIT(1);
        } /* endif */
        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RENGFCNT", &temp_count)) {
          pKey3X->Key31 += temp_count;
          pKey3X->IsValid |= BIT(1);
        } /* endif */
        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RENGRCNT", &temp_count)) {
          pKey3X->Key31 += temp_count;
          pKey3X->IsValid |= BIT(1);
        } /* endif */

        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "BITRATEX", &bitratex)) {
          pKey3X->Key34 = bitratex; //v34.BITRATEX // Initial transmit carrier data rate (0-64000)
          pKey3X->IsValid |= BIT(4);
        } /* endif */

        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "BITRATER", &bitrater)) {
          if (pmwmDspInfo->usPCMConnection && bitrater) {
            pKey3X->Key35 = mwmcmsgConvertX2ConnectSpeed(bitrater);
          } else {
            pKey3X->Key35 = bitrater; //v34.BITRATER // Initial transmit carrier data rate (0-64000)
          } /* endif */
          pKey3X->IsValid |= BIT(5);
        } /* endif */
      } /* endif */
   } /* endif */



	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveCarrierEvents exit\n");  }

   return TRUE;
}

void UniDiagsGetMNPEvents(KEY4X* pKey4X, KEY5X* pKey5X, PMWM_DSPINFO pmwmDspInfo)
{
   WORD  wData;
   ULONG ulAddress;
   DWORD dwData;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsGetMNPEvents entry\n");  }


   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtMNP, "N402", &wData))
   {
      pKey4X->Key41    = wData;
      pKey4X->IsValid |= BIT(1);
   }
   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtMNP, "LTTIMES", &wData))
   {
      pKey4X->Key42    = wData;
      pKey4X->IsValid |= BIT(2);
   }
   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtMNP, "NAKSSEEN", &wData))
   {
      pKey4X->Key43    = wData;
      pKey4X->IsValid |= BIT(3);
   }

   // Key5X.Key56;  // Transmit Frame count, if error control protocol running
   if (!dspLabelToAddress(pmwmDspInfo->hmtMNP, "FRMXMITC", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &dwData, 2, DSP_MEMXFER_DATA_READ)) {
        pKey5X->Key56 = dwData; // >> 16;
        //pKey5X->Key56 |= dwData << 16;
        pKey5X->IsValid |= BIT(6);
      } /* endif */
   }

   // Key5X.Key57;  // Received Frame count, if error control protocol running
   if (!dspLabelToAddress(pmwmDspInfo->hmtMNP, "FRMRECVC", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &dwData, 2, DSP_MEMXFER_DATA_READ)) {
        pKey5X->Key57 = dwData;// >> 16;
        //pKey5X->Key57 |= dwData << 16;
        pKey5X->IsValid |= BIT(7);
      } /* endif */
   }

   // Key5X.Key58;  // Transmit Frame error count, if error control protocol running
   if (!dspLabelToAddress(pmwmDspInfo->hmtMNP, "FRMXMITE", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &dwData, 2, DSP_MEMXFER_DATA_READ)) {
        pKey5X->Key58 = dwData; // >> 16;
        //pKey5X->Key58 |= dwData << 16;
        pKey5X->IsValid |= BIT(8);
      } /* endif */
   }

   // Key5X.Key59;  // Received Frame error count, if error control protocol running
   if (!dspLabelToAddress(pmwmDspInfo->hmtMNP, "FRMRECVE", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &dwData, 2, DSP_MEMXFER_DATA_READ)) {
        pKey5X->Key59 = dwData;// >> 16;
        //pKey5X->Key59 |= dwData << 16;
        pKey5X->IsValid |= BIT(9);
      } /* endif */
   }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsGetMNPEvents exit\n");  }

   return;
}


void UniDiagsGetV42Events(KEY4X* pKey4X, KEY5X* pKey5X, PMWM_DSPINFO pmwmDspInfo)
{
   WORD  wData;
   ULONG ulAddress;
   DWORD dwData;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsGetV42Events entry\n");  }

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtProtSupr, "N401X", &wData))
   {
      pKey4X->Key41    = wData;
      pKey4X->IsValid |= BIT(1);
   }
   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtProtSupr, "RETRANS", &wData))
   {
      pKey4X->Key42    = wData;
      pKey4X->IsValid |= BIT(2);
   }
   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV42LAPDLC, "REJSSEEN", &wData))
   {
      pKey4X->Key43    = wData;
      pKey4X->IsValid |= BIT(3);
   }

   // Key5X.Key56;  // Transmit Frame count, if error control protocol running
   if (!dspLabelToAddress(pmwmDspInfo->hmtV42LAPDLC, "FRMXMITC", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &dwData, 2, DSP_MEMXFER_DATA_READ)) {
        pKey5X->Key56 = dwData; // >> 16;
        //pKey5X->Key56 |= dwData << 16;
        pKey5X->IsValid |= BIT(6);
      } /* endif */
   }

   // Key5X.Key57;  // Received Frame count, if error control protocol running
   if (!dspLabelToAddress(pmwmDspInfo->hmtV42LAPDLC, "FRMRECVC", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &dwData, 2, DSP_MEMXFER_DATA_READ)) {
        pKey5X->Key57 = dwData;// >> 16;
        //pKey5X->Key57 |= dwData << 16;
        pKey5X->IsValid |= BIT(7);
      } /* endif */
   }

   // Key5X.Key58;  // Transmit Frame error count, if error control protocol running
   if (!dspLabelToAddress(pmwmDspInfo->hmtV42LAPDLC, "FRMXMITE", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &dwData, 2, DSP_MEMXFER_DATA_READ)) {
        pKey5X->Key58 = dwData; // >> 16;
        //pKey5X->Key58 |= dwData << 16;
        pKey5X->IsValid |= BIT(8);
      } /* endif */
   }

   // Key5X.Key59;  // Received Frame error count, if error control protocol running
   if (!dspLabelToAddress(pmwmDspInfo->hmtV42LAPDLC, "FRMRECVE", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &dwData, 2, DSP_MEMXFER_DATA_READ)) {
        pKey5X->Key59 = dwData;// >> 16;
        //pKey5X->Key59 |= dwData << 16;
        pKey5X->IsValid |= BIT(9);
      } /* endif */
   }


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsGetV42Events exit\n");  }

   return;
}


/*------------------------------------------------------------------
 *
 * Save protocol/compression events.
 * Call data link part to get information.
 * for class 0, called on at command, and at call termination.
 *
 *------------------------------------------------------------------*/
BOOL UniDiagsSaveProtocolCompressionEvents(KEY4X* pKey4X, KEY5X* pKey5X, PMWM_DSPINFO pmwmDspInfo)
{
   WORD protocol;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveProtocolCompressionEvents entry\n");  }


// Key4X.Key42;  // Error control link timeouts
// Key4X.Key43;  // Error control link NAKs

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "PROT_COL", &protocol)) {
      if (protocol & MWM_STATUS_V42_CONNECT) {
         UniDiagsGetV42Events(pKey4X, pKey5X, pmwmDspInfo);
      } else if (protocol & MWM_STATUS_MNP_CONNECT) {
         UniDiagsGetMNPEvents(pKey4X, pKey5X, pmwmDspInfo);
      }
   }

    { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveProtocolCompressionEvents exit\n");  }

   return TRUE;
}

/*------------------------------------------------------------------
 *
 * for class 0, called on at command, and at call termination.
 *
 *------------------------------------------------------------------*/
BOOL UniDiagsSaveDataFlowInfo(PMWM_DSPINFO pmwmDspInfo)
{
  UNIDIAG* pUniDiags;
  KEY5X*   pKey5X;
  WORD     wData;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveDataFlowInfo entry\n");  }


  if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
     return FALSE;

  pKey5X = &pUniDiags->Key5X;

// Key5X.Key50;  // Transmit flow control: 0 = off; 1 = DC1/DC3; 2 = V.24 ckt 106/133
// Key5X.Key51;  // Receive flow control: 0 = off; 1 = DC1/DC3; 2 = V.24 ckt 106/133
   if(UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "BS_Q", &wData))
   {
      pKey5X->IsValid |= (BIT(0) | BIT(1));

      switch (wData)
      {
         case 0:
            pKey5X->Key50 = pKey5X->Key51 = 0;
            break;
         case 1:
         case 3:
            pKey5X->Key50 = pKey5X->Key51 = 2;
            break;
         case 4:
         case 5:
            pKey5X->Key50 = pKey5X->Key51 = 1;
            break;
      }
   }

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveDataFlowInfo exit\n");  }

   return TRUE;
}


BOOL UniDiagsClearUARTCounters(PMWM_DSPINFO pmwmDspInfo)
{
  DWORD dwData = 0;
  ULONG ulAddress;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsClearUARTCounters entry\n");  }


  if (!dspLabelToAddress(pmwmDspInfo->hmtUART, "TXCHARS", &ulAddress)) {
     dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                    &dwData, 2, DSP_MEMXFER_DATA_WRITE);
  }

  if (!dspLabelToAddress(pmwmDspInfo->hmtUART, "RXCHARS", &ulAddress)) {
     dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                    &dwData, 2, DSP_MEMXFER_DATA_WRITE);
  }

  if (!dspLabelToAddress(pmwmDspInfo->hmtUART, "RXOVRUN", &ulAddress)) {
     dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                    &dwData, 2, DSP_MEMXFER_DATA_WRITE);
  }


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsClearUARTCounters exit\n");  }

  return TRUE;

}

BOOL UniDiagsSaveDataFlowEvents(KEY5X* pKey5X, PMWM_DSPINFO pmwmDspInfo)
{
  ULONG    ulAddress;
  DWORD    dwData;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveDataFlowEvents entry\n");  }


// Key5X.Key52;  // Transmit characters sent from DTE
  if (!dspLabelToAddress(pmwmDspInfo->hmtUART, "TXCHARS", &ulAddress)) {
     if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                         &dwData, 2, DSP_MEMXFER_DATA_READ)) {
       pKey5X->Key52 = dwData >> 16;
       pKey5X->Key52 |= dwData << 16;
       pKey5X->IsValid |= BIT(2);
     } /* endif */
  }

// Key5X.Key53;  // Received characters sent to DTE
  if (!dspLabelToAddress(pmwmDspInfo->hmtUART, "RXCHARS", &ulAddress)) {
     if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                         &dwData, 2, DSP_MEMXFER_DATA_READ)) {
       pKey5X->Key53 = dwData >> 16;
       pKey5X->Key53 |= dwData << 16;
       pKey5X->IsValid |= BIT(3);
     } /* endif */
  }

// Key5X.Key54;  // Transmit characters lost (data overrun errors from DTE)
  pKey5X->Key54 = 0;
  pKey5X->IsValid |= BIT(4);

// Key5X.Key55;  // Received characters lost (data overrun errors to DTE)
  if (!dspLabelToAddress(pmwmDspInfo->hmtUART, "RXOVRUN", &ulAddress)) {
     if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                         &dwData, 2, DSP_MEMXFER_DATA_READ)) {
       pKey5X->Key55 = dwData >> 16;
       pKey5X->Key55 |= dwData << 16;
       pKey5X->IsValid |= BIT(5);
     } /* endif */
  }


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveDataFlowEvents exit\n");  }

   return TRUE;
}

BOOL UniDiagsSaveOtherEvents(KEY6X* pKey6X)
{
// Key6X.Key61;  // Call Waiting event count

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveOtherEvents entry\n");  }


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveOtherEvents exit\n");  }

   return TRUE;
}

/*------------------------------------------------------------------
 *
 * Save status of items which can change over the course of a call.
 *
 *------------------------------------------------------------------*/
BOOL MWM_ENTRY UniDiagsUpdateCallStatus(PMWM_DSPINFO pmwmDspInfo)
{
   UNIDIAG* pUniDiags;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsUpdateCallStatus entry\n");  }


   if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
      return FALSE;

   UniDiagsSaveCarrierEvents(&pUniDiags->Key3X, pmwmDspInfo);
   UniDiagsSaveProtocolCompressionEvents(&pUniDiags->Key4X, &pUniDiags->Key5X, pmwmDspInfo);
   if (pmwmDspInfo->bCarrierDetected)
     UniDiagsSaveDataFlowEvents(&pUniDiags->Key5X, pmwmDspInfo);
   UniDiagsSaveOtherEvents(&pUniDiags->Key6X);

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsUpdateCallStatus exit\n");  }

   return TRUE;
}

/*------------------------------------------------------------------
 *
 *  The call setup result represents the initial state of the call.
 *  If a call has been attempted, modem control will have the initial
 *  state of the network, no dialtone, busy, etc.
 *
 *  If we get connected to the network, the next signal detected is
 *  obtained from the pumps.
 *
 *------------------------------------------------------------------*/
WORD UniDiagsCallSetupResult(PMWM_DSPINFO pmwmDspInfo)
{
   WORD res=0;

   WORD disccode;
   WORD usFCLASS;
   WORD RequestAnswer = 0;    //TODO

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsCallSetupResult entry\n");  }


   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "DISCCODE", &disccode) &&
       UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "FCLASS", &usFCLASS))
   {
      do
      {
         if ( disccode != 0xffff )         // if already disconnected
         {
            if ( disccode == 6 )           // check for basic errors
            {
               res = Busy_signal;
               break;                      // ----------------> leave
            }
       // These aren't in disccode right now
            //   res = Reorder_signal;
            //   res = No_dialtone;
         }

         switch (usFCLASS)
         {
            case 0:                                // called on dp_cycles event
               if (RequestAnswer)
                  res = Data_Answering_signal;
               else
                  res = Data_Calling_signal;
               break;
            case 1:
            case 2:
               if (RequestAnswer)
                  res = Fax_Answering_signal;
               else
                  res = Fax_Calling_signal;
               break;
            case 8:
               res = Voice;
               break;
            case 80:
               res = Voice;
               break;

            default:
               res = No_recognized_signal;
               break;
         }
      }
      while (0);           // loop used only for breakout
   }


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsCallSetupResult exit\n");  }

   return res;
}

WORD UniDiagsMultiMediaMode(PMWM_DSPINFO pmwmDspInfo)
{
   WORD res=0;
   WORD usFCLASS;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsMultiMediaMode entry\n");  }


   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "FCLASS", &usFCLASS))
   {
      switch (usFCLASS)
      {
         case 0:
            {
            WORD protocol;
            if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "PROT_COL", &protocol)) {
               if (protocol & MWM_STATUS_V80_CONNECT) {
                  res = Other_V_80;
               } else if (protocol & MWM_STATUS_H324_CONNECT) {
                  res = Video_telephony_H324;
               } else {
                  res = Data_Only;
               }
            }
            }
            break;
         case 1:
         case 2:
            res = FAX_Only;
            break;
         case 8:
            res = Voice_Only;
            break;
         case 80:
            res = VoiceView;
            break;
         default:
            res = Data_Only;
            break;
      }
   }


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsMultiMediaMode exit\n");  }

   return res;
}

/*------------------------------------------------------------------
 *
 * Save status of items at call setup time.
 * for class 0, called at DP_CYCLES event
 *
 *------------------------------------------------------------------*/
BOOL UniDiagsSaveSetupInfo(PMWM_DSPINFO pmwmDspInfo)
{
   UNIDIAG* pUniDiags;
   KEY0X* pKey0X;
   WORD   protocol;
   WORD   csrcode;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveSetupInfo entry\n");  }


   if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
      return FALSE;

   pKey0X = &pUniDiags->Key0X;


   //pKey0X->Key1 = UniDiagsCallSetupResult(pmwmDspInfo);  // Table 2 Call Setup Result code
   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "CSRCODE", &csrcode)) {
     pKey0X->Key1 = csrcode;  // Table 2 Call Setup Result code
     pKey0X->IsValid |= BIT(1);
   }

   pKey0X->Key2 = UniDiagsMultiMediaMode(pmwmDspInfo);   // Table 3 Multi-media mode
   pKey0X->IsValid |= BIT(2);

   // Table 4 DTE-DCE interface mode
   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "PROT_COL", &protocol)) {
      if (protocol & MWM_STATUS_V80_CONNECT)
        pKey0X->Key3 = V_80_transparent_synchronous;
      else
        pKey0X->Key3 = Async_data;

      pKey0X->IsValid |= BIT(3);
   } /* endif */

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveSetupInfo exit\n");  }

   return TRUE;
}

BOOL UniDiagsSaveCallSetupResultInfo(PMWM_DSPINFO pmwmDspInfo)
{
   UNIDIAG* pUniDiags;
   KEY0X* pKey0X;
   WORD   csrcode;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveCallSetupResultInfo entry\n");  }


   if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
      return FALSE;

   pKey0X = &pUniDiags->Key0X;

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "CSRCODE", &csrcode)) {
     pKey0X->Key1 = csrcode;  // Table 2 Call Setup Result code
     pKey0X->IsValid |= BIT(1);
   }

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveCallSetupResultInfo exit\n");  }

   return TRUE;
}

BOOL UniDiagsSaveOctets(PMWM_DSPINFO pmwmDspInfo)
{
  UNIDIAG* pUniDiags;
  KEY0X* pKey0X;
  ULONG  ulAddress;
  USHORT ausCMOctets[7];
  USHORT ausJMOctets[7];
  WORD   i;
  char   achTemp[3];

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveOctets entry\n");  }


  if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
     return FALSE;

  pKey0X = &pUniDiags->Key0X;

  // NOTE: For V.80 mode, there are special steps that should be taken.
  // Since the importance of this is not apparent, it is not yet been
  // implemented.
  // - The CM report as the calling modem is valid as implemented below.
  // - To report JM as the calling modem, we need to check if we
  //   are in V.80 mode, and check for valid +A8E values, then read
  //   the MSGA8MX string from modem control.
  // - To report CM as the answering modem, do the same steps as
  //   for reporting JM as the calling modem.
  // - To report JM as the answering modem, we need to convert the
  //   CM string extracted above, convert it to hex, then perform
  //   the ANDing with the bits in the 7 words beginning at CALLFN.

// pKey0X->Key4[12]; // V.8 CM octet string, same format as V.25ter Annex A, in quotes
// pKey0X->Key5[12]; // V.8 JM octet string, same format as V.25ter Annex A, in quotes
   if (pmwmDspInfo->usDialOrAnswerFlag == DIAL) {
     if (!dspLabelToAddress(pmwmDspInfo->hmtProtSupr, "CALLFN", &ulAddress)) {
        if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                            ausCMOctets, 7, DSP_MEMXFER_DATA_READ)) {
          memset(pKey0X->Key4, '\0', sizeof(pKey0X->Key4));
          strcpy( pKey0X->Key4, "\"");
          for (i=0; i<=6; i++) {
             if (!(ausCMOctets[i] & 0x8000)) {
               sprintf( achTemp, "%02X", ((ausCMOctets[i] & 0x1FE)>>1) );
               strcat( pKey0X->Key4, achTemp );
               pKey0X->IsValid |= BIT(4);
             } /* endif */
          } /* endfor */
          strcat( pKey0X->Key4, "\"");
        } /* endif */
     } /* endif */

     if (!dspLabelToAddress(pmwmDspInfo->hmtProtSupr, "REP_CF", &ulAddress)) {
        if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                            ausJMOctets, 7, DSP_MEMXFER_DATA_READ)) {
          memset(pKey0X->Key5, '\0', sizeof(pKey0X->Key5));
          strcpy( pKey0X->Key5, "\"");
          for (i=0; i<=6; i++) {
            if (!(ausJMOctets[i] & 0x8000)) {
              sprintf( achTemp, "%02X", ausJMOctets[i]);
              strcat( pKey0X->Key5, achTemp );
              pKey0X->IsValid |= BIT(5);
            } /* endif */
          } /* endfor */
          strcat( pKey0X->Key5, "\"");
        } /* endif */
     } /* endif */

   } else {
     if (!dspLabelToAddress(pmwmDspInfo->hmtProtSupr, "REP_CF", &ulAddress)) {
        if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                            ausCMOctets, 7, DSP_MEMXFER_DATA_READ)) {
          memset(pKey0X->Key4, '\0', sizeof(pKey0X->Key4));
          for (i=0; i<=6; i++) {
            if (!(ausCMOctets[i] & 0x8000)) {
              sprintf( achTemp, "%02X", ausCMOctets[i]);
              strcat( pKey0X->Key4, achTemp );
              pKey0X->IsValid |= BIT(4);
            } /* endif */
          } /* endfor */

          if (!dspLabelToAddress(pmwmDspInfo->hmtProtSupr, "CALLFN", &ulAddress)) {
             if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                                 ausJMOctets, 7, DSP_MEMXFER_DATA_READ)) {
               memset(pKey0X->Key5, '\0', sizeof(pKey0X->Key5));
               for (i=0; i<=6; i++) {
                  if (!(ausJMOctets[i] & 0x8000) && !(ausCMOctets[i] & 0x8000)) {
                    sprintf( achTemp, "%02X", ((ausJMOctets[i] & 0x1FE)>>1) & ausCMOctets[i] );
                    strcat( pKey0X->Key5, achTemp );
                    pKey0X->IsValid |= BIT(5);
                  } /* endif */
               } /* endfor */
             } /* endif */
          } /* endif */
        } /* endif */
     } /* endif */
   } /* endif */

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveOctets exit\n");  }

   return TRUE;

}

/*------------------------------------------------------------------
 *
 * Save status of line.
 * for class 0, called at LINESPEED event
 *
 *------------------------------------------------------------------*/
BOOL UniDiagsSaveLineInfo(PMWM_DSPINFO pmwmDspInfo)
{
   WORD rtdelay;
   UNIDIAG* pUniDiags;
   KEY1X* pKey1X;
   WORD carrier = UniDiagsCarrierNegotiationResult(pmwmDspInfo);

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveLineInfo entry\n");  }


   if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
      return FALSE;

   pKey1X = &pUniDiags->Key1X;

   if (carrier == X2 || carrier == V_90 || carrier == V_34) {

      // pKey1X->Key10;  // Received signal power level, in -dBm (0-43)
      USHORT usAndMask = 0x4000;
      float  fNextValue = 24;
      float  fTotal = 0;
      USHORT usValue;
      ULONG  ulRC = 0;
      USHORT usTransmitLevel;

      if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "LDINGEQZ", &usValue)) {
        while (usAndMask) {
          if ( usValue & usAndMask )
            fTotal += fNextValue;
          fNextValue /= 2;
          usAndMask = usAndMask >> 1;
        } /* endwhile */
        pKey1X->Key10 = (USHORT)fTotal - 3;
        pKey1X->IsValid |= BIT(0);

        // pKey1X->Key12;  // Estimated noise level, in -dBm (10-90)
        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "EQZSNR", &usValue)) {
          usValue = usValue >> 8;
          pKey1X->Key12 = usValue + pKey1X->Key10;
          pKey1X->IsValid |= BIT(2);
        } /* endif */

      } /* endif */

      // pKey1X->Key11;  // Transmit signal power level, in -dBm (0-17)
      ulRC = WTT_QueryInfo( NULL, WTBLOCK_PLATFORM_SPECIFIC, WTPARAM_MODEM_TX,
                            WTLENGTH_ONEPARAM, (short *)&usTransmitLevel, 0L);
      if (!ulRC) {
        pKey1X->Key11 = -usTransmitLevel;
        pKey1X->Key11 = (float)pKey1X->Key11/10+0.5;
        pKey1X->IsValid |= BIT(1);

        // pKey1X->Key14;  // Near echo loss, in units of dB
        usAndMask = 0x4000;
        fNextValue = 24;
        fTotal = 0;
        if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "LDINGECH", &usValue)) {
          while (usAndMask) {
            if ( usValue & usAndMask )
              fTotal += fNextValue;
            fNextValue /= 2;
            usAndMask = usAndMask >> 1;
          } /* endwhile */
          pKey1X->Key14 = (USHORT)fTotal - pKey1X->Key11 - 3;
          pKey1X->IsValid |= BIT(4);

          // pKey1X->Key15;  // Far echo loss, in units of dB
          if (pKey1X->IsValid & BIT(0)) {
            pKey1X->Key15 = pKey1X->Key14 + 2*(pKey1X->Key10 - 12) + 6;
            pKey1X->IsValid |= BIT(5);
          } /* endif */
        } /* endif */
      } /* endif */



      // pKey1X->Key13;  // Normalized Mean Squared error, 100 (0x64) = minimum inter-symbol
      {
        double mse;
        ULONG  ulAddress;
        DWORD  dwEQZE2LPF;
        USHORT usPADGAIN;
        USHORT usDMIN1, usDMIN2, usDMIN;
        USHORT usCUREEVNTS;
        USHORT usBITRATER;
        USHORT ausTableValues[] = {104, 128, 165, 231, 255};
        SHORT  sIndex;
        SHORT  sMaxTableEntries = sizeof(ausTableValues) / sizeof(USHORT); 

        if (carrier == X2 || carrier == V_90 ) {
          //if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "EQZE2LPF", &usEQZE2LPF)) {

            if (!dspLabelToAddress(pmwmDspInfo->hmtV34, "EQZE2LPF", &ulAddress)) {
               if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                                   &dwEQZE2LPF, 2, DSP_MEMXFER_DATA_READ)) {

                 if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "PADGAIN", &usPADGAIN)) {
                   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "DMIN1", &usDMIN1)) {
                     if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "DMIN2", &usDMIN2)) {

                       usDMIN = min(usDMIN1, usDMIN2);

                       if (usDMIN !=0 ) {
                         mse = 78.125 * pow((float)usPADGAIN/pow(2,15),2);
                         mse *= dwEQZE2LPF / pow(usDMIN,2);
                       } else {
                         mse = 0;
                       } /* endif */

                       pKey1X->Key13 = mse;

                       // Note: For whatever reason, it appears that this
                       // debug statement allows the correct value to be
                       // displayed in the 32-bit modem, so it should stay.
                       DPF("Key 13 = %d", pKey1X->Key13);

                       pKey1X->IsValid |= BIT(3);

                     } /* endif */
                   } /* endif */
                 } /* endif */
               } /* endif */
            } /* endif */
          //} /* endif */

        } else {
          if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "CUREEVNTS", &usCUREEVNTS)) {
            if (usCUREEVNTS == 0) {
              pKey1X->Key13 = ausTableValues[0];
              pKey1X->IsValid |= BIT(3);
            } else if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "BITRATER", &usBITRATER)) {
              sIndex = 0.5 + log10(0x1E6 * (float)usCUREEVNTS / (usBITRATER*3));
              if (sIndex < 0) {
                sIndex = 0;
              } else if (sIndex >= sMaxTableEntries) {
                sIndex = sMaxTableEntries - 1;
              } /* endif */
              pKey1X->Key13 = ausTableValues[sIndex];
              pKey1X->IsValid |= BIT(3);
            } /* endif */
          } /* endif */
        } /* endif */
      }



      // pKey1X->Key16   // Far echo delay, in units of ms
      // pKey1X->Key17;  // V34.RT_DEL // Round Trip delay, in units of ms
      if (carrier == X2 || carrier == V_90 ) {
         pKey1X->Key16 = 0;
         pKey1X->IsValid |= BIT(6);
      } else if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RT_DEL", &rtdelay)) {
         pKey1X->Key16 = (float)rtdelay/2400*1000;
         pKey1X->IsValid |= BIT(6);
      } /* endif */

      if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RT_DEL", &rtdelay)) {
         pKey1X->Key17 = (float)rtdelay/2400*1000;
         pKey1X->IsValid |= BIT(7);
      } /* endif */



   } /* endif */

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveLineInfo exit\n");  }

   return TRUE;
}

BOOL UniDiagsSaveInfoBits(PMWM_DSPINFO pmwmDspInfo)
{
   UNIDIAG* pUniDiags;
   KEY1X*   pKey1X;
   V34INFO  V34Info;
   DWORD   *pV34Info = (DWORD *)&V34Info;
   USHORT   usValue;
   ULONG    ulAddress;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveInfoBits entry\n");  }


   if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
      return FALSE;

   pKey1X = &pUniDiags->Key1X;

   // pKey1X->Key18;  // Table 5 V.34 INFO bit map
   memset( &V34Info, 0, sizeof(V34Info) );
   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "INFO0RXBFD", &usValue)) {
     V34Info.INFO0_bit20 = usValue >> 7;
   } /* endif */

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "INFO0TXBFD", &usValue)) {
     V34Info.INFO0_bit0 = usValue >> 7;
   } /* endif */

   if (!dspLabelToAddress(pmwmDspInfo->hmtV34, "INFO1CTXBFD", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress+18,
                          &usValue, 1,DSP_MEMXFER_DATA_READ)) {
        V34Info.INFOc_bits7988 = usValue;
      } /* endif */
   } /* endif */


   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "XSYMCODE", &usValue)) {
     if (!dspLabelToAddress(pmwmDspInfo->hmtV34, "INFO1CTXBFD", &ulAddress)) {
       if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress+(usValue+6),
                           &usValue, 1, DSP_MEMXFER_DATA_READ)) {
         V34Info.INFOc_bits2629 = usValue & 0x0F;
       } /* endif */
     } /* endif */
   } /* endif */

   if (!dspLabelToAddress(pmwmDspInfo->hmtV34, "INFO1ARXBFD", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress+6,
                          &usValue, 1,DSP_MEMXFER_DATA_READ)) {
        V34Info.INFOa_bits2629 = usValue & 0x0F;
      } /* endif */
   } /* endif */

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "EN_ASYM_RATES", &usValue)) {
     V34Info.MP_bit50 = usValue;
   } /* endif */

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "EN_ASYM_RATES_RM", &usValue)) {
     V34Info.MP_bit0 = usValue;
   } /* endif */

   if (!dspLabelToAddress(pmwmDspInfo->hmtV34, "INFO1ARXBFD", &ulAddress)) {
      if (!dspMemTransfer(pmwmDspInfo->hDSP, ulAddress+12,
                         &usValue, 1,DSP_MEMXFER_DATA_READ)) {
        V34Info.INFOa_bits4049 = usValue;
      } /* endif */
   } /* endif */

   pKey1X->Key18 = *pV34Info;
   pKey1X->IsValid |= BIT(8);

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveInfoBits exit\n");  }

   return TRUE;
}

WORD UniDiagsCarrierNegotiationResult(PMWM_DSPINFO pmwmDspInfo)
{
   WORD res = 0xff;
   WORD usFCLASS;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsCarrierNegotiationResult entry\n");  }


   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "FCLASS", &usFCLASS))
   {
      switch (usFCLASS)
      {
         case 0:
         {
            WORD protocol;
            if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "PROT_COL", &protocol)) {
               if (protocol & MWM_STATUS_V34_CONNECT) {
                  if (pmwmDspInfo->usPCMConnection == 1) {
                     res = X2;
                  } else if (pmwmDspInfo->usPCMConnection == 4) {
                     res = V_90;
                  } else {
                     res = V_34;
                  }
               } else if (protocol & MWM_STATUS_V32BIS_CONNECT) {
                  res = V_32bis;
               } else if (protocol & MWM_STATUS_V22BIS_CONNECT) {
                  if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V23) {
                     res = V_23_CC;
                  } else {
                     res = V_22bis;
                  }
               }
            }
         }
         break;
         case 1:
         case 2:
         {
            HMTASK hmtFAXPMP;
            WORD   wData;

            if(!dspNameToTaskHandle(pmwmDspInfo->hmodFAXPMP, "FAXPMP", &hmtFAXPMP)) {
               if (UniDiagsReadWord(pmwmDspInfo, hmtFAXPMP, "FCTLXRAT", &wData)) {
                  if (0xFFFF == wData) {
                     res = V_21;
                  } else if (0 == wData || 1 == wData) {
                     res = V_27ter;
                  } else if (2 == wData || 3 == wData) {
                     res = V_29_HD;
                  } else if (4 == wData || 5 == wData || 6 == wData || 7 == wData) {
                     res = V_17;
		  }
	       }
            }
         }
         break;
      }
   }

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsCarrierNegotiationResult exit\n");  }

   return res;
}

/*------------------------------------------------------------------
 *
 * Save status of carrier.
 * Call pump to get information.
 * for class 0, called at CARRIER_DETECT event
 *
 *------------------------------------------------------------------*/
BOOL UniDiagsSaveCarrierInfo(PMWM_DSPINFO pmwmDspInfo)
{
   UNIDIAG* pUniDiags;
   KEY2X* pKey2X;
   WORD xsymcode, rsymcode;
   WORD rcarrier;
   WORD carrier = UniDiagsCarrierNegotiationResult(pmwmDspInfo);
   USHORT ausSymbolRates[] = {2400, 2743, 2800, 3000, 3200, 3429, 8000};
   USHORT ausLowFreqs[] = {1600, 1646, 1680, 1800, 1829, 1959};
   USHORT ausHighFreqs[] = {1800, 1829, 1867, 2000, 1920, 1959};

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveCarrierInfo entry\n");  }


   if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
      return FALSE;

   pKey2X = &pUniDiags->Key2X;

   if (carrier != 0xff)
   {
      pKey2X->IsValid |= (BIT(0) | BIT(1));
      pKey2X->Key21 = carrier;     // Table 6 Receive Carrier Negotiation Result
      if (carrier == X2 || carrier == V_90)
         pKey2X->Key20 = V_34   ;  // Table 6 Transmit Carrier Negotiation Result
      else
         pKey2X->Key20 = carrier;
   }

   if (carrier == X2 || carrier == V_90 || carrier == V_34) {

     if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "XSYMCODE", &xsymcode)) {
       pKey2X->Key22  = ausSymbolRates[xsymcode/2]; //v34.XSYMCODE // Transmit Carrier symbol rate (0-8000)
       pKey2X->IsValid |= BIT(2);
     } /* endif */

     if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RSYMCODE", &rsymcode)) {
       pKey2X->Key23 = ausSymbolRates[rsymcode/2]; //v34.RSYMCODE // Receive Carrier symbol rate (0-8000)
       pKey2X->IsValid |= BIT(3);
     } /* endif */

     // pKey2X->Key24; //v34.TCARRIER // Transmit Carrier frequency (0-4000)
     // pKey2X->Key25; //v34.RCARRIER // Receive Carrier frequency  (0-4000)
     if (carrier == X2 || carrier == V_90) {
       pKey2X->Key24 = 1920;
       pKey2X->IsValid |= BIT(4);
       pKey2X->Key25 = 0;
       pKey2X->IsValid |= BIT(5);
     } else if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "RCARRIER", &rcarrier)) {
        if (rcarrier == 0) { // Low Carrier Frequency
          pKey2X->Key24 = ausLowFreqs[xsymcode/2];
          pKey2X->IsValid |= BIT(4);
          pKey2X->Key25 = ausLowFreqs[rsymcode/2];
          pKey2X->IsValid |= BIT(5);
        } else {             // High Carrier Frequency
          pKey2X->Key24 = ausHighFreqs[xsymcode/2];
          pKey2X->IsValid |= BIT(4);
          pKey2X->Key25 = ausHighFreqs[rsymcode/2];
          pKey2X->IsValid |= BIT(5);
        } /* endif */
     } /* endif */

   } /* endif */

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveCarrierInfo exit\n");  }

   return TRUE;
}

BOOL UniDiagsSaveConnectSpeed(PMWM_DSPINFO pmwmDspInfo)
{
  UNIDIAG* pUniDiags;
  KEY2X* pKey2X;
  WORD bitrater, bitratex;
  WORD carrier = UniDiagsCarrierNegotiationResult(pmwmDspInfo);


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveConnectSpeed entry\n");  }

  if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
     return FALSE;

  pKey2X = &pUniDiags->Key2X;

  if (carrier == X2 || carrier == V_90 || carrier == V_34) {
    if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "BITRATEX", &bitratex)) {
      pKey2X->Key26 = bitratex; //v34.BITRATEX // Initial transmit carrier data rate (0-64000)
      pKey2X->IsValid |= BIT(6);
    } /* endif */

    if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmtV34, "BITRATER", &bitrater)) {
      if (carrier == X2 || carrier == V_90) {
        pKey2X->Key27 = mwmcmsgConvertX2ConnectSpeed(bitrater);
      } else {
        pKey2X->Key27 = bitrater; //v34.BITRATER // Initial transmit carrier data rate (0-64000)
      } /* endif */
      pKey2X->IsValid |= BIT(7);
    } /* endif */
  } /* endif */

  mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveConnectSpeed exit\n");  }

  return TRUE;
}

/*------------------------------------------------------------------
 *
 * Save carrer events.
 * for class 0, called on PROTOCOL_SELECT event.
 *
 *------------------------------------------------------------------*/
BOOL UniDiagsSaveProtocolInfo(PMWM_DSPINFO pmwmDspInfo)
{
   UNIDIAG* pUniDiags;
   KEY4X* pKey4X;
   WORD protocol;
   WORD size;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveProtocolInfo entry\n");  }


   if (!mwmPoundUDGetCallStatusBuffer(&pUniDiags))
      return FALSE;

   pKey4X = &pUniDiags->Key4X;

   if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "PROT_COL", &protocol))
   {
      if (protocol & MWM_STATUS_V42_CONNECT)
          pKey4X->Key40 = V_42_LAPM;  // Table 7 Protocol Negotiation Result
      if (protocol & MWM_STATUS_MNP_CONNECT)
          pKey4X->Key40 = V_42_MNP;   // Table 7 Protocol Negotiation Result
      pKey4X->IsValid |= BIT(0);

/////   pKey4X->Key41;  // 3 digits Error Control frame size

      if (protocol & MWM_STATUS_V42BIS_CONNECT)
         pKey4X->Key44 = V_42bis;  // Table 8 Compression Negotiation Result
      if (protocol & MWM_STATUS_MNP5_CONNECT)
         pKey4X->Key44 = MNP5;     // Table 8 Compression Negotiation Result
      pKey4X->IsValid |= BIT(4);

      if (protocol & MWM_STATUS_V42BIS_CONNECT)
      {
         if (UniDiagsReadWord(pmwmDspInfo, pmwmDspInfo->hmctlTask, "NEGDICT", &size)) {
           pKey4X->Key45 = size;  // Compression dictionary size
           pKey4X->IsValid |= BIT(5);
         } /* endif */
      }
   }

   mwmParsePostMessage( WM_MWM_POUNDUD_UPDATE, 0, 0 );


	 { MW_SYSLOG_1(TRACE_MWMLW32,"unidiags::UniDiagsSaveProtocolInfo exit\n");  }

   return TRUE;
}

