/*
 *   dspactiv.c -- Activate module routines 
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include "dspmgr.h"

#include "dspactiv.h"                  // Include function prototypes
#include "dspispos.h"                  // CH01 Mwave/OS prototypes
#include "dspfree.h"                   // Include function prototypes
#include "dspgpc.h"                    // Include function prototypes
#include "dspitcb.h"                   // Include function prototypes
#include "dspstruc.h"                  // Include function prototypes

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ChangeModuleState                                       */
/*                                                                          */
/* FUNCTION: This routine will activate a module into one of three modes.   */
/*           INACTIVE<->STANDBY<->ACTIVE.  It will also free the temporary  */
/*           data structure that were used for the module load.             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD   prmod - Ptr to module to activate.                        */
/*        USHORT  usFlags - Flag indicates mode to active module to         */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*        - GPC connection list is trashed                                  */
/*        - Task temporary structure is trashed                             */
/*        - Segment temporary structure is trashed                          */
/*        - fixup table is trashed                                          */
/*        - discardable string table is trashed                             */
/*        - Debugging dumps occur here                                      */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ChangeModuleState(PRMOD prmod,USHORT usFlags)

{
   RC         ulRC;
   PRTSK      prtsk;   /* CH02 */

   if (prmod == NULL)
      return (DSP_INV_HANDLE);  /* CH01 */

   if (prmod->MOD_achValid != MODVALIDATE)
      return (DSP_INV_HANDLE);  /* CH01 */

   prtsk = prmod->MOD_prtskTail;
   if (prtsk) { /* Assumes module does not span DSPs */
      if ((ulRC=CheckDSPStatus(prtsk->TSK_prdsp)))
         return (ulRC);  /* CH02 Check for card unplugged */
   } /* endif */

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspactiv::ChangeModuleState entry prmod %x usFlags %x\n",(int)prmod,usFlags);

   /*************************************************************************/
   /* The general flow is INACTIVE<->STANDBY<->ACTIVE.  You                 */
   /* have to go through each step when processing a module.                */
   /* So first establish what mode the module is in and then                */
   /* determine where you want to go.                                       */
   /* Note that when a module goes into ACTIVE mode the                     */
   /* temporary structures are freed since they are no                      */
   /* longer needed.                                                        */
   /*************************************************************************/
   /*************************************************************************/
   /* Module is in INACTIVE MODE.                                           */
   /*************************************************************************/

   if ((prmod->MOD_usflgState&MOD_STATEMASK) == MOD_INACTIVE) {

      switch (usFlags&DSP_ACTIVATE_MASK) {

         case (DSP_ACTIVATE_INACTIVE) :
            if ((ulRC = ActivateModuleUtil(prmod, usFlags)) == DSP_NOERROR)
               prmod->MOD_usflgState = MOD_INACTIVE;
            break;

         case (DSP_ACTIVATE_STANDBY) :
            if ((ulRC = StandbyModule(prmod)) == DSP_NOERROR)
               prmod->MOD_usflgState = MOD_STANDBY;
            break;

         case (DSP_ACTIVATE_ACTIVE) :  /* put in STANDBY, then ACTIVE       */
            if ((ulRC = StandbyModule(prmod)) == DSP_NOERROR)
               prmod->MOD_usflgState = MOD_STANDBY;

            if (ulRC == DSP_NOERROR)
               if ((ulRC = ActivateModuleUtil(prmod, usFlags)) == DSP_NOERROR)
                  prmod->MOD_usflgState = MOD_ACTIVE;

            if (ulRC == DSP_NOERROR)
               ulRC = FreeModuleTemps(prmod);

            break;

         default  :
            return (DSP_INV_FLAG);
      }                                /* endswitch                         */
   }                                   /* Endif MOD_INACTIVE                */

   /*************************************************************************/
   /* Module is in STANDBY MODE.                                            */
   /*************************************************************************/

   else if ((prmod->MOD_usflgState&MOD_STATEMASK) == MOD_STANDBY) {

      switch (usFlags&DSP_ACTIVATE_MASK) {

         case (DSP_ACTIVATE_INACTIVE) :
            if ((ulRC = ActivateModuleUtil(prmod, usFlags)) == DSP_NOERROR)
               prmod->MOD_usflgState = MOD_INACTIVE;
            break;

         case (DSP_ACTIVATE_STANDBY) :
            if ((ulRC = ActivateModuleUtil(prmod, usFlags)) == DSP_NOERROR)
               prmod->MOD_usflgState = MOD_STANDBY;
            break;

         case (DSP_ACTIVATE_ACTIVE) :
            if ((ulRC = ActivateModuleUtil(prmod, usFlags)) == DSP_NOERROR)
               prmod->MOD_usflgState = MOD_ACTIVE;
            if (ulRC == DSP_NOERROR)
               ulRC = FreeModuleTemps(prmod);
            break;

         default  :
            return (DSP_INV_FLAG);
      }                                /* endswitch                         */
   }                                   /* Endif MOD_STANDBY                 */

   /*************************************************************************/
   /* Module is in ACTIVE MODE.                                             */
   /*************************************************************************/

   else if ((prmod->MOD_usflgState&MOD_STATEMASK) == MOD_ACTIVE) {

      switch (usFlags&DSP_ACTIVATE_MASK) {

         case (DSP_ACTIVATE_INACTIVE) :    /* First in Standby, then        */
                                           /* Inactive                      */
            ulRC = StandbyModule(prmod);
            prmod->MOD_usflgState = MOD_STANDBY;

            ulRC = ActivateModuleUtil(prmod, usFlags);
            prmod->MOD_usflgState = MOD_INACTIVE;
            break;

         case (DSP_ACTIVATE_STANDBY) :
            if ((ulRC = StandbyModule(prmod)) == DSP_NOERROR)
               prmod->MOD_usflgState = MOD_STANDBY;
            break;

         case (DSP_ACTIVATE_ACTIVE) :

   /* Reset tasks to Active Entry Points                                    */

            if ((ulRC = ActivateModuleUtil(prmod, usFlags)) == DSP_NOERROR)
               prmod->MOD_usflgState = MOD_ACTIVE;
            break;

         default  :
            return (DSP_INV_FLAG);
      }                                 /* endswitch                        */
   }                                    /* Endif MOD_ACTIVE                 */
   else
      return (DSP_INV_FLAG);


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspactiv::ChangeModuleState exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ActivateModuleUtil                                      */
/*                                                                          */
/* FUNCTION: This routine will do the Activate processing for               */
/*           Module.  It simply, traverses the Task list and processes      */
/*           the tasks.                                                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD   prmod   - Ptr to Module to activate.                      */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ActivateModuleUtil(PRMOD prmod,USHORT usFlags)

{
   PRTSK      prtskHead;
   PRTSK      prtsk;
   PRTSK      prtskNext;
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */

   /*************************************************************************/
   /* Traverse the task list for the module and Process each task.          */
   /* NOTE that a module cannot be activated if it has a non-null           */
   /* connection list.                                                      */
   /*************************************************************************/
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspactiv::ActivateModuleUtil entry prmod %x usFlags %x\n",(int)prmod,usFlags);

   if ((usFlags&DSP_ACTIVATE_MASK) == DSP_ACTIVATE_ACTIVE) {
      if (prmod->MOD_prgpcclTail != NULL) {
         ulRC = DSP_GPC_NOTCONNECTED;
         return (ulRC);
      }
      if (prmod->MOD_pritcbclTail != NULL) {
         ulRC = DSP_ITCB_NOTCONNECTED;
         return (ulRC);
      }
   }
   if (prmod->MOD_prtskTail != NULL) {
      prtskHead = prmod->MOD_prtskTail; /* set task head/tail               */
      prtskNext = prtskHead->TSK_prtskNext; /* Start at Head of List        */
      do {
         prtsk = prtskNext;            /* Next Task in List                 */
         prtskNext = prtsk->TSK_prtskNext; /* Save next current             */
         ulRC = ChangeTaskState(prtsk, usFlags);
      }  while (prtsk != prtskHead && ulRC == DSP_NOERROR); // Back at head yet or error ?
   }                                   /* endif                             */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspactiv::ActivateModuleUtil exit ulRC %lx\n",ulRC);

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ChangeTaskState                                         */
/*                                                                          */
/* FUNCTION: This routine will activate the task of the module into one of  */
/*           the possible activation mode. It will also free the temporary  */
/*           data structure that were used for the module load.             */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK   prtsk   - Ptr to task to activate.                        */
/*        USHORT  usFlags - Flag indicates mode to active module to         */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ChangeTaskState(PRTSK prtsk,USHORT usFlags)

{
   RC         ulRC;

   if (prtsk == NULL)
      return (DSP_INV_HANDLE);  /* CH01 */

   if (prtsk->TSK_achValid != TSKVALIDATE)
      return (DSP_INV_HANDLE);  /* CH01 */

   ulRC = CheckDSPStatus(prtsk->TSK_prdsp);       /* CH02 */
   if (ulRC != DSP_NOERROR)  /* Dont continue if DSP is not available! */
      return (ulRC);

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspactiv::ChangeTaskState entry prtsk %x usFlags %x\n",(int)prtsk,usFlags);

   /*************************************************************************/
   /* The general flow is INACTIVE<->STANDBY<->ACTIVE.  You                 */
   /* have to go through each step when processing a task.                  */
   /* So first establish what mode the task   is in and then                */
   /* determine where you want to go.                                       */
   /*************************************************************************/
   /*************************************************************************/
   /* Task is in INACTIVE MODE.                                             */
   /*************************************************************************/

   if ((prtsk->TSK_usflgState&TSK_STATEMASK) == TSK_INACTIVE) {

      switch (usFlags&DSP_ACTIVATE_MASK) {

         case (DSP_ACTIVATE_INACTIVE) :
            if ((ulRC = ActivateTaskInactive(prtsk, usFlags)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_INACTIVE;
            break;

         case (DSP_ACTIVATE_STANDBY) :
            if ((ulRC = StandbyTask(prtsk)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_STANDBY;
            break;

         case (DSP_ACTIVATE_ACTIVE) :

            if ((ulRC = StandbyTask(prtsk)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_STANDBY;

            if (ulRC == DSP_NOERROR)
               if ((ulRC = ChangeActivationState(prtsk, usFlags))==DSP_NOERROR)
                  prtsk->TSK_usflgState = TSK_ACTIVE;

            break;

         default  :
            return (DSP_INV_FLAG);
      }                                /* endswitch                         */
   }                                   /* Endif TSK_INACTIVE                */

   /*************************************************************************/
   /* Task is in STANDBY MODE.                                              */
   /*************************************************************************/

   else if ((prtsk->TSK_usflgState&TSK_STATEMASK) == TSK_STANDBY) {

      switch (usFlags&DSP_ACTIVATE_MASK) {

         case (DSP_ACTIVATE_INACTIVE) :
            if ((ulRC = ActivateTaskInactive(prtsk, usFlags)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_INACTIVE;
            break;

         case (DSP_ACTIVATE_STANDBY) :
            if ((ulRC = ChangeActivationState(prtsk, usFlags)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_STANDBY;
            break;

         case (DSP_ACTIVATE_ACTIVE) :
            if ((ulRC = ChangeActivationState(prtsk, usFlags)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_ACTIVE;
            break;

         default  :
            return (DSP_INV_FLAG);
      }                                /* endswitch                         */
   }                                   /* Endif TSK_STANDBY                 */

   /*************************************************************************/
   /* Task is in ACTIVE MODE.                                               */
   /*************************************************************************/

   else if ((prtsk->TSK_usflgState&TSK_STATEMASK) == TSK_ACTIVE) {

      switch (usFlags&DSP_ACTIVATE_MASK) {

         case (DSP_ACTIVATE_INACTIVE) :

            if ((ulRC = ChangeActivationState(prtsk,
               DSP_ACTIVATE_STANDBY)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_STANDBY;

            if (ulRC == DSP_NOERROR)
               if ((ulRC = ActivateTaskInactive(prtsk, usFlags)) == DSP_NOERROR)
                  prtsk->TSK_usflgState = TSK_INACTIVE;
            break;

         case (DSP_ACTIVATE_STANDBY) :
            if ((ulRC = ChangeActivationState(prtsk, usFlags)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_STANDBY;
            break;

         case (DSP_ACTIVATE_ACTIVE) :
            if ((ulRC = ChangeActivationState(prtsk, usFlags)) == DSP_NOERROR)
               prtsk->TSK_usflgState = TSK_ACTIVE;
            break;

         default  :
            return (DSP_INV_FLAG);
      }                                /* endswitch                         */
   }                                   /* Endif TSK_ACTIVE                  */
   else
      return (DSP_INV_FLAG);

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspactiv::ChangeTaskState exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ActivateTaskInactive                                    */
/*                                                                          */
/* FUNCTION: This routine will activate the task of the module the INACTIVE */
/*           state.  At this time this includes:                            */
/*           a) Disconnecting GPCs                                          */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK   prtsk   - Ptr to task to activate.                        */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: DisconnectGPC() will put tasks in STANDBY mode if they are */
/*               not already.                                               */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC ActivateTaskInactive(PRTSK prtsk,USHORT usFlags)

{
   PRGPC      prgpc;
   PRITCB     pritcb;
   PRITCB     pritcbHead;
   PRITCB     pritcbSecList;
   PRITCB     pritcbSecListHead;
   PRITCB     pritcbSec_next;
   RC         ulRC = DSP_NOERROR;      /* Assume DSP_NOERROR                */
   USHORT     usInhibit;

   if (prtsk == NULL)
      return (DSP_INV_PARAMETER);

   /*************************************************************************/
   /* For each GPC in the task, Go through it's connection                  */
   /* list, deleting each connection encountered                            */
   /*                                                                       */
   /* Note: for Owner GPCs ...UserListNext points to the                    */
   /*       tail of the list.                                               */
   /* Note: for User  GPCs ...UserListNext points to the                    */
   /*       Next User Attached to the Owner.                                */
   /*************************************************************************/

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspactiv::ActivateTaskInactive entry prtsk %x usFlags %x\n",(int)prtsk,usFlags);
   if (prtsk->TSK_prgpcTail != NULL)   /* does task have GPCs?              */
      {

      /**********************************************************************/
      /* This search must check the tail first because of the potential for */
      /* VGPC shadows.  VGPC shadows are always placed after the GPC root.  */
      /* If the search starts with the GPC pointed to by the tail then it   */
      /* could be a VGPC shadow and this is very bad.                       */
      /**********************************************************************/

      prgpc = prtsk->TSK_prgpcTail;    /* Check tail first                  */
      do {                             /* Traverse each GPC                 */
         if ((prgpc->GPC_usflgKind&GPC_KINDMASK) == GPC_USER) {
            if (prgpc->GPC_usflgKind != GPC_VIRTUSER) {/* process real user */
               if (prgpc->GPC_prgpcOwner != NULL) {

                  if (fDSPINIT_FLAG)
                     usInhibit = (DONT_FREE_OWNER_BIOS | DONT_FREE_USER_BIOS);
                  else
                     usInhibit = DONT_FREE_USER_BIOS;
                  ulRC = DisconnectGPC
                         (prgpc->GPC_prgpcOwner->GPC_prseg->SEG_prtsk,
                          prgpc->GPC_prgpcOwner->GPC_pszName, prtsk,
                          prgpc->GPC_pszName, usInhibit);

                  /* IF the the User Task was freeed, our task is gone      */
                  /* nothing left to walk the Task/CB is gone               */

                  if ((ulRC==DSP_USER_TASK_FREE)||(ulRC==DSP_OW_US_TASK_FREE))
                     return (DSP_NOERROR);

                  /* Don't Tell anyone we freed the task                    */

                  if (ulRC == DSP_OWNER_TASK_FREE)
                     ulRC = DSP_NOERROR;
               }
            }                          /* endof process real user gpc       */
            else {

            /****************************************************************/
            /* Process virtual user GPC:                                    */
            /*   Traverse list of VGPC shadows until the root               */
            /*   GPC prvgpcTail is NULL.  VGPC shadows are now              */
            /*   linked into the list of GPCs as well as the list           */
            /*   formed from prvgpcTail as follows:                         */
            /*                                                              */
            /*   Ŀ  Ŀ  Ŀ  Ŀ  Ŀ                   */
            /*   gpc->vgpc->shvgpc->shvgpc->gpc...                */
            /*        root->shadow<-shadow                        */
            /*         A                     */
            /*                                                   */
            /*                                                              */
            /*    where gpc    = normal gpc                                 */
            /*          vgpc   = root of vgpc                               */
            /*          shvgpc = shadow of root vgpc                        */
            /*                                                              */
            /*    Top pointers are regular gpc list pointers and            */
            /*    bottom pointers are VGPC list pointers kept in            */
            /*    prvgpcTail element of RGPC structure.                     */
            /****************************************************************/

               while (prgpc->GPC_prvgpcTail != NULL && ulRC == DSP_NOERROR) {
                  if (fDSPINIT_FLAG)
                     usInhibit = (DONT_FREE_OWNER_BIOS|
                        DONT_FREE_USER_BIOS);
                  else
                     usInhibit = DONT_FREE_USER_BIOS;
                  ulRC = DisconnectGPC(
                    prgpc->GPC_prvgpcTail->GPC_prgpcOwner->GPC_prseg->SEG_prtsk,
                    prgpc->GPC_prvgpcTail->GPC_prgpcOwner->GPC_pszName, prtsk,
                    prgpc->GPC_pszName, usInhibit);

                  /* IF the the User Task was freeed, our task is gone      */
                  /* nothing left to walk the Task/CB is gone               */

                  if ((ulRC==DSP_USER_TASK_FREE)||(ulRC==DSP_OW_US_TASK_FREE))
                     return (DSP_NOERROR);

                  /* Don't Tell anyone we freed the task                    */

                  if (ulRC == DSP_OWNER_TASK_FREE)
                     ulRC = DSP_NOERROR;

               }
            }                          /* endof process virtual user gpc    */
         }
         else {

            /****************************************************************/
            /* process owner GPC delete tail of owner's user list until     */
            /* user list is empty                                           */
            /****************************************************************/

            while (prgpc->GPC_prgpcUserListNext != NULL && ulRC == DSP_NOERROR
               ) {

               if (fDSPINIT_FLAG)
                  usInhibit = (DONT_FREE_OWNER_BIOS | DONT_FREE_USER_BIOS);
               else
                  usInhibit = DONT_FREE_OWNER_BIOS;
               ulRC = DisconnectGPC(prtsk, prgpc->GPC_pszName,
                        prgpc->GPC_prgpcUserListNext->GPC_prseg->SEG_prtsk,
                        prgpc->GPC_prgpcUserListNext->GPC_pszName, usInhibit);


               /* IF the the Owner Task was freeed, our task is gone        */
               /* nothing left to walk the Task/CB is gone                  */

               if ((ulRC==DSP_OWNER_TASK_FREE) || (ulRC==DSP_OW_US_TASK_FREE))
                  return (DSP_NOERROR);

                 /* Don't Tell anyone we freed the task                     */

               if (ulRC == DSP_USER_TASK_FREE)
                  ulRC = DSP_NOERROR;

            }
         }

         prgpc = prgpc->GPC_prgpcNext;

      }  while ((prgpc != prtsk->TSK_prgpcTail) && (ulRC == DSP_NOERROR));

   }                                   /* end if(gpcTail != NULL)           */

   /*************************************************************************/
   /* The same for ITCBs                                                    */
   /*************************************************************************/

   if ((ulRC == DSP_NOERROR) && (prtsk->TSK_pritcbTail != NULL)) {
      pritcb = pritcbHead = prtsk->TSK_pritcbTail;
      do {                             /* Traverse each ITCB                */
         pritcb = pritcb->ITCB_pritcbNext;

         if ((pritcb->ITCB_usflgKind&ITCB_KINDMASK) == ITCB_SECONDARY) {

            if (pritcb->ITCB_pritcbPrimary != NULL) {

               if (fDSPINIT_FLAG)
                  usInhibit = (DONT_FREE_OWNER_BIOS | DONT_FREE_USER_BIOS);
               else
                  usInhibit = DONT_FREE_USER_BIOS;
               ulRC = DisconnectITCB
                  (pritcb->ITCB_pritcbPrimary->ITCB_prseg->SEG_prtsk,
                                       /* prPrimary                         */
                  pritcb->ITCB_pritcbPrimary->ITCB_pszName, prtsk,
                                       /* prSecondary                       */
                  pritcb->ITCB_pszName, usInhibit);

               /* IF the the User Task was freeed, our task is gone         */
               /* nothing left to walk the Task/CB is gone                  */

               if ((ulRC==DSP_USER_TASK_FREE) || (ulRC==DSP_OW_US_TASK_FREE))
                  return (DSP_NOERROR);

               /* Don't Tell anyone we freed the task                       */

               if (ulRC == DSP_OWNER_TASK_FREE)
                  ulRC = DSP_NOERROR;

            }                          /* endif                             */
         }
         else {                        /* Primary ITCB                      */

            /****************************************************************/
            /* Start at the HEAD! of the List and traverse until            */
            /* the entire list has been disconnected.                       */
            /* Note that pritcb points to the Primary and is always         */
            /* the same.                                                    */
            /****************************************************************/

            if (pritcb->ITCB_pritcbSecListNext != NULL) {
               pritcbSecListHead =
                  pritcb->ITCB_pritcbSecListNext->ITCB_pritcbSecListNext;
               pritcbSec_next = pritcbSecListHead->ITCB_pritcbSecListNext;

               do {

                 /* Get Next ITCB this one may get deleted                  */


                  pritcbSecList = pritcbSec_next;
                  pritcbSec_next = pritcbSecList->ITCB_pritcbSecListNext;

                 /* >>>---> We are Owner                                    */

                  if (fDSPINIT_FLAG)
                     usInhibit = (DONT_FREE_OWNER_BIOS | DONT_FREE_USER_BIOS);
                  else
                     usInhibit = DONT_FREE_OWNER_BIOS;
                  ulRC = DisconnectITCB(prtsk,
                     pritcb->ITCB_pszName,
                     pritcbSecList->ITCB_prseg->SEG_prtsk, /* prUser        */
                     pritcbSecList->ITCB_pszName, usInhibit);

                 /* IF the the Owner Task was freeed, our task is gone      */
                 /* nothing left to walk the Task/CB is gone                */

                  if ((ulRC==DSP_OWNER_TASK_FREE)||(ulRC==DSP_OW_US_TASK_FREE))
                     return (DSP_NOERROR);

                 /* Don't Tell anyone we freed the task                     */

                  if (ulRC == DSP_USER_TASK_FREE)
                     ulRC = DSP_NOERROR;

               }  while ((pritcbSecList != pritcbSecListHead) && (ulRC ==
                  DSP_NOERROR));       /* enddo                             */
            }                          /* endif                             */
         }                             /* endif for ITCB is Primary or      */
                                       /* Secondary                         */
      }  while ((pritcb != pritcbHead) && (ulRC == DSP_NOERROR));/* enddo   */
   }                                   /* endif for pritcbTail != NULL      */

   /*************************************************************************/
   /* Remove the task from MWaveOS.                                         */
   /* If this is the last task, then remove the FM from MWaveOS.            */
   /* If the task is INACTIVE, it is not in the MWaveOS so don't remove it. */
   /*************************************************************************/

   if ((ulRC == DSP_NOERROR) &&
       (prtsk->TSK_usflgState != TSK_INACTIVE) && (!fDSPINIT_FLAG)) {
      ulRC = RemoveTaskFromOS(prtsk);
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspactiv::ActivateTaskInactive exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FreeModuleTemps                                         */
/*                                                                          */
/* FUNCTION: deallocate all temporary structures for this module. It has    */
/*           loaded and fixed-up and in fact is running on the DSP. This    */
/*           is basically a cleanup step. (Free step)                       */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD   prmod - Ptr to module to cleanup.                         */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*        - Task temporary structure is trashed                             */
/*        - Segment temporary structure is trashed                          */
/*        - fixup table is trashed                                          */
/*        - discardable string table is trashed                             */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC FreeModuleTemps(PRMOD prmod)

{
   PRTSK      prtsk;                   /* Current task node pointer         */
   PRTSK      prtskHead;               /* Head of task list                 */

   /*************************************************************************/
   /* Traverse the Module (each task) starting at the head                  */
   /* and free each task temporary structure                                */
   /* Free the Data Segments, then the Instruction Segs.                    */
   /*************************************************************************/

   prtskHead = prmod->MOD_prtskTail->TSK_prtskNext; // set task head
   prtsk = prtskHead;                  // Start at head of task list

   do {
      if (prtsk->TSK_prsegDTail != NULL) {
         if (FreeSegTemps(prtsk->TSK_prsegDTail->SEG_prsegNext))
            return (DSP_INTERNAL_ERROR);
      }                                /* endif                             */
      if (prtsk->TSK_prsegITail != NULL) {
         if (FreeSegTemps(prtsk->TSK_prsegITail->SEG_prsegNext))
            return (DSP_INTERNAL_ERROR);
      }                                /* endif                             */
      if (prtsk->TSK_prtsktmp != NULL)
         if (RemoveRTasktmp(prtsk))    // free the temp structure
            return (DSP_INTERNAL_ERROR);

      prtsk = prtsk->TSK_prtskNext;    // go to next in list
   }  while (prtsk != prtskHead);      // Back at head yet ?

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FreeSegTemps                                            */
/*                                                                          */
/* FUNCTION: deallocate all temporary structures for a segment linked-list. */
/*           Traverse the Task (each segment)  starting at the head and     */
/*           free each seg temporary structure                              */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRSEG  prsegHead - Ptr to head of segment list                    */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*        - Segment temporary structure is trashed                          */
/*        - fixup table is trashed                                          */
/*        - discardable string table is trashed                             */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC FreeSegTemps(PRSEG prsegHead)

{
   PRSEG      prseg;                   // Current seg node pointer

   prseg = prsegHead;                  // Start at head of seg list

   do {
      if (prseg->SEG_prsegtmp != NULL)
         if (RemoveRSegmenttmp(prseg)) // free the temp structure
            return (DSP_INTERNAL_ERROR);
      prseg = prseg->SEG_prsegNext;    // go to next in list
   }  while (prseg != prsegHead);      // Back at head yet ?

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: StandbyModule                                           */
/*                                                                          */
/* FUNCTION: This routine will activate the tasks of the module into standby*/
/*           mode and issue the GPC connections.                            */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD   prmod - Ptr to module to activate.                        */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC StandbyModule(PRMOD prmod)

{
   RC         ulRc = DSP_NOERROR;      // Error return code
   PRTSK      prtsk;                   // Current task node pointer

   /*************************************************************************/
   /* TRAVERSE THE MODULE TO INSERT TASK TCBs INTO THE ISPOS FMCB LIST      */
   /* in standby mode.                                                      */
   /*************************************************************************/
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspactiv::StandbyModule entry prmod %x\n",(int)prmod);
   if ((prtsk = prmod->MOD_prtskTail) != NULL) {

      /* TRAVERSE THE MODULE                                                */

      do {

         prtsk = prtsk->TSK_prtskNext; /* ON 1ST TIME, MOVES TO HEAD OF LIST*/
         ulRc = ChangeTaskState(prtsk, DSP_ACTIVATE_STANDBY);

      }  while ((prtsk != prmod->MOD_prtskTail) && (ulRc == DSP_NOERROR));

      /**********************************************************************/
      /* Check if there are any GPCCL control blocks attached.              */
      /* Make sure that there is a "NO ERROR" return code so far.           */
      /* If so, Process the GPC connection list.                            */
      /* Any error in connecting GPCs yields NOTCONNECTED message           */
      /**********************************************************************/

      if (ulRc == DSP_NOERROR)
         if (ProcessGPCCL(prmod) != DSP_NOERROR)
            ulRc = DSP_GPC_NOTCONNECTED;

      /**********************************************************************/
      /* Check if there are any ITCBCL control blocks attached.             */
      /* Make sure that there is a "NO ERROR" return code so far.           */
      /* If so, Process the ITCB connection list.                           */
      /* Any error in connecting ITCB yields NOTCONNECTED message           */
      /**********************************************************************/

      if (ulRc == DSP_NOERROR)
         if (ProcessITCBCL(prmod) != DSP_NOERROR)
            ulRc = DSP_ITCB_NOTCONNECTED;
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspactiv::StandbyModule exit ulRc %lx\n",ulRc);
   return (ulRc);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: StandbyTask                                             */
/*                                                                          */
/* FUNCTION: This routine will activate the task  of a   module into standby*/
/*           mode.                                                          */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK   prtsk - Ptr to task   to activate.                        */
/*                                                                          */
/* OUTPUT:                                                                  */
/*    DSP_NOERROR    - No errors occured                                    */
/*       TDB         - errors                                               */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC StandbyTask(PRTSK prtsk)

{
   RC         ulRc = DSP_NOERROR;      // Error return code
   PRMEM      prmemFMCB;

   /*************************************************************************/
   /* INSERT TASK TCBs INTO THE ISPOS FMCB LIST                             */
   /*************************************************************************/

   if (prtsk == NULL)
      return  DSP_NOERROR;

   /* IF THIS IS AN OS, IT HAS ALREADY BEEN PROCESSED.  DON'T DO IT AGAIN   */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspactiv::StandbyTask entry prtsk %x\n",(int)prtsk);

   if (((prtsk->TSK_usflgType&TSK_TYPEMASK) != TSK_OS) &&
      ((prtsk->TSK_usflgState == TSK_INACTIVE) || (prtsk->TSK_usflgState ==
      TSK_ACTIVE))) {

         /* INSERT THE TASK INTO THE FM LIST IN STANDBY MODE                */

      if (prtsk->TSK_prfm == NULL)     /* FM for NRT tasks is part of ISPOS */
         prmemFMCB = NULL;
      else
         prmemFMCB = prtsk->TSK_prfm->FM_prmemFMCB;

      if ((ulRc = InsertTaskInOS(prtsk, prmemFMCB,
         prtsk->TSK_prsegDTail->SEG_prsegNext->SEG_prmem)) != DSP_NOERROR)
         return  ulRc;
   }
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspactiv::StandbyTask exit ulRc %lx\n",ulRc);
   return (ulRc);
}
