/*
 *   mwblapi.h
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
/*  Mwave BlackList LIB (Worldtrade Call Restrictions)*/
/**/
/*     This LIB enforces country specified calling restrictions to limit*/
/*     the number and frequency that unsuccessful calls can be made.*/
/*     There are two call restrictions.  The first one is a delay that*/
/*     must be met before redialing a number that was previously*/
/*     unsuccessfully called.  The second call restriction prevents*/
/*     calling a number until a user intervention occurs.  This is*/
/*     known as a BlackList.*/
/**/
/*     Call restrictions are only for automatically dialing equipment.*/
/*     No enforcement is required for manualy dialed (or user dialed)*/
/*     calls.  However, all calls need to be registered with this*/
/*     LIB so that when a successful connection is made, further*/
/*     auto-dialing restrictions can be removed.*/
/**/
/*     Devices such as modem and fax need to use the BL_APIversion,*/
/*     BL_RegisterDevice, BL_Reset, BL_RequestDialPermission, and*/
/*     BL_DialResults API's.  The remaining API's support the BlackList*/
/*     viewing utility.*/
/**/
/*     A distinction needs to be made between device types in the system*/
/*     for tracking unsuccessful calls.  A failure of a modem to*/
/*     connect does not imply that a fax machine could not connect.*/
/*     Therefore, a class-0 modem needs to register (BL_RegisterDevice)*/
/*     separate from a class-1 fax, etc.  In the case of a multi-line*/
/*     system, ideally, the devices for each line will register*/
/*     separate.  For an example, suppose you have two lines each set up for*/
/*     fax with one line plugged into a telephone line and the other one*/
/*     unconnected.  If the two fax lines are registered separately then*/
/*     failures on the unconnected line will not affect the connected fax*/
/*     machine.*/
/**/
/**/
/**/

#if !defined(MWBLAPI_H)
#define MWBLAPI_H 1



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*---- includes ----------------------------------------------------------*/
/**/
#include <mwave.h>  /* picks up ULONG, USHORT, HDSP definitions*/
                    /* and for DOS, APIENTRY*/


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*----helpful defines----------------------------------------------------------*/
/**/

/* Note:  The API's in this file require defines such as HDSP, ULONG,*/
/*        and USHORT that are in the Mwave.h file.*/


#if !defined(FALSE)
  #define FALSE 0
#endif

#if !defined(TRUE)
  #define TRUE  1
#endif





#define BL_API_VERSION            1 /* This defines what information the API*/
                                    /* is returning.  This is used to make sure*/
                                    /* the LIB is in sync with code that uses*/
                                    /* it if the API ever changes.*/

#define BL_VIEWER_VERSION         2 /* This defines what level Blacklist Viewer is*/
                                    /* needed.  This value needs to change if*/
                                    /* the information passed to the Blacklist*/
                                    /* Viewer changes.*/

                                    /* This is assumed to be the same size as USHORT*/
#define TWO_BYTES                 2 /* which is two bytes.  The structure of the WT Tables*/
                                    /* requires two byte entries and this program assumes*/
                                    /* that the definition of unsigned short will not change*/
                                    /* in future compilers.*/

#define MAX_BL_DIAL_STRING      100 /* This defines the maximum size of the phone*/
                                    /* number string.  This should be an integer*/
                                    /* number of ULONG's.*/

#define MAX_DEVICE_NAME_SIZE    128 /* This defines the maximum size of the*/
                                    /* device name string.*/



/*----Device type defines------------------------------------------------------*/
#define DEVICE_MODEM     1 /* This defines that the device is a data modem.*/
#define DEVICE_FAX       2 /* This defines that the device is a fax modem.*/

/*----The type of call---------------------------------------------------------*/
#define MANUAL_DIAL              0  /* The call is under direct user control*/
#define AUTOMATIC_DIAL           1  /* The call is an autodialed, auto-repeat*/
                                    /* dialed call.*/


/*---- The standard phone number ----------------------------------------------*/
#define STANDARD_NUMBER  "0000000000"  /* This defines a dummy phone number to*/
                                       /* be used if unsuccessful counts apply*/
                                       /* to the total number of attempts instead*/
                                       /* of individual numbers.*/

/*----Data types---------------------------------------------------------------*/
/**/

/* The BlackList number list item structure.  An array of these structures*/
/* is filled in by the BL_GetNumberList API.*/

struct BL_NumberListItem
{
  char   szPhoneNumber[MAX_BL_DIAL_STRING];    /* A call restricted phone number*/

  ULONG bBlacklisted;                          /* Defines if the phone number is*/
                                               /* blacklisted or delayed.*/

  ULONG ulDelay;                               /* The amount of the delay in seconds*/

  ULONG ulUnsuccessfulCount;                   /* The unsuccessful count for the number*/

  ULONG  ulReserved;                           /* A reserved field for future expansion*/

};



/* The blacklist settings currently in effect.*/


struct BL_Settings
{
  /* This is the country number as defined in the mwwttbl.h file.*/
  ULONG ulCountryNumber;

  /* This specifies what the device name is that was registered in the*/
  /* BL_RegisterDevice call.*/
  char   szDeviceName[MAX_DEVICE_NAME_SIZE];

  /* This specifies what the device type is as was registered in the*/
  /* BL_RegisterDevice call.*/
  ULONG ulDeviceType;

  /* This specifies the maximum number of phone numbers tracked per device.*/
  /* This is used with the BL_GetNumberList function.*/
  ULONG ulMaxPhoneNumbersTracked;

  /* This is set if there are blacklist settings in the current WT Table.*/
  BOOL bBlacklistingEnabled;

  /* This settings says that special code is being used for the country*/
  /* so the following settings may not mean anything.*/
  BOOL bUseSpecialAlgorithm;

  BOOL bUseTimeWindowing;

  BOOL bUserAbortIsUnsuccessful;

  BOOL bBlacklistAfterFirstSeries;

  BOOL bBlacklistAfterSecondSeries;

  BOOL bDoNotDistiguishPhoneNumbers;

  ULONG ulAllCallDelay;

  ULONG ulFirstRetryDelay;

  ULONG ulUnsuccessfulRetryDelay;

  ULONG ulMaxUnsuccessfulCount;

  ULONG ulTimeWindow;

  ULONG ulSeriesToSeriesDelay;

};


/*---Return Codes----------------------------------------------------------------*/

  /* This is returned by functions if no error was detected.*/
  #define BL_SUCCESSFUL                                        0L /* must remain zero*/

  /* This is returned if no device is registered with the specified handle.*/
  #define BLERR_DEVICE_HANDLE_NOT_IN_USE                       1L

  /* This is returned if the device handle is out of valid range.*/
  #define BLERR_INVALID_DEVICE_HANDLE                          2L

  /* This is returned if the dial string is too large*/
  #define BLERR_DIAL_STRING_TOO_LARGE                          3L

  /* This is returned if the phone number sent to request permission to dial*/
  /* has no digits in it.*/
  #define BLERR_PHONE_NUMBER_IS_BLANK                          4L

  /* This is returned if the requested phone number has previously been requested*/
  /* and results have not yet been registered.*/
  #define BLERR_PHONE_NUMBER_IS_IN_USE                         5L

  /* This is returned if there is an internal program failure.*/
  #define BLERR_INTERNAL_ERROR                                 6L

  /* This is returned if the WTT settings specified that special code*/
  /* should be available for the country but nothing is coded in.*/
  #define BLERR_NO_SPECIAL_CODE_AVAILABLE                      7L

  /* This is returned if a dial handle is requested for a device and*/
  /* the device currently has a checked out dial handle but has not*/
  /* call BL_DialResults.*/
  #define BLERR_DIAL_HANDLE_IN_USE                             8L

  /* This is returned if a call is made to BL_DialResults and the*/
  /* dial handle does not match the expected dial handle.*/
  #define BLERR_INCORRECT_DIAL_HANDLE                          9L

  /* This is returned with a call to BL_GetNumberList if it is called with*/
  /* a memory size too small to hold all the possible numbers.  The size*/
  /* expected is returned in BL_Settings.ulMaxPhoneNumbersTracked when*/
  /* BL_GetInfo is called.*/
  #define BLERR_NUMBER_LIST_TOO_SMALL                         10L

  /* This is returned if the specified dial type is incorrect*/
  #define BLERR_INCORRECT_DIAL_TYPE                           11L

  /* This is returned if a call to the WTT_CheckCountryNumber function failed*/
  #define BLERR_WTT_LIB_FAILED                                12L


/*------ Initialization problems -----------------------------------------------*/

  /* This is returned any time a function is called and the blacklist driver*/
  /* has not successful initalized.*/
  #define BLERR_INIT_FAILED                                   50L

  /* This is returned if a failure occurs while reading settings from the*/
  /* WT Table.*/
  #define BLERR_WTT_READ_FAILED                               51L

  /* This is returned if the device array is not initalized for a certain*/
  /* device index.*/
  #define BLERR_DEVICE_NOT_INITIALIZED                        52L

  /* This is returned for a time window setting if the max unsuccessful count*/
  /* in the WT Table is set too large.  Time windowing requires a time stamp*/
  /* array that has a size fixed by MAX_TIME_STAMP_COUNT.*/
  #define BLERR_WTT_MAX_COUNT_TOO_LARGE                       53L

/*------ Registration problems -------------------------------------------------*/

  /* This is returned when there are no device handles available.*/
  #define BLERR_NO_AVAIL_DEVICE_HANDLES                       60L

  /* This is returned if the device name string used in registration*/
  /* is too large.*/
  #define BLERR_DEVICE_NAME_TOO_LARGE                         61L

  /* This is returned if an invalid device type is specified while registering.*/
  #define BLERR_INVALID_DEVICE_TYPE                           62L

  /* This is returned when registration for a device has failed.  This is*/
  /* used in class to class communications.  It is more likely for code*/
  /* calling the blacklist driver to receive BLERR_NO_AVAIL_DEVICE_HANDLES*/
  /* instead of this error.*/
  #define BLERR_DEVICE_REGISTRATION_FAILED                    63L


/* end of error return code list*/
/*^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^*/



/*^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^*/
/*---Dial Permission-----------------------------------------------------------*/

  /* This is returned if permission to dial is granted.*/
  #define BL_DIAL_GRANTED                                      1L

  /* This is returned if the number requested is blacklisted*/
  #define BL_NUMBER_BLACKLISTED                                2L

  /* This is returned if the number requested is delayed*/
  #define BL_NUMBER_DELAYED                                    3L


/*^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^*/
/*---Dial Status---------------------------------------------------------------*/
/**/
/* These are the status codes that need to returned to the BL_Results function*/
/* when the status of a call is known.*/
/**/
/**/
  /* This is the status if the call was connected successfully.  In the case*/
  /* of modem and fax, the call is successful if the pumps can make a connection.*/
  #define SUCCESSFUL                                           1L

  /* Unsuccessful Conditions:*/

  /* This is the status if the number could not be dialed because of lack of*/
  /* dial tone when dial tone was expected.*/
  #define NO_DIAL_TONE                                         2L

  /* This is the status if the number dialed was busy or if the network was busy.*/
  #define LINE_BUSY                                            3L

  /* This is the status if the user aborts the call (manual intervention).*/
  #define USER_ABORT                                           4L

  /* This is the default unsuccessful status if the above conditons do not*/
  /* happen.*/
  #define NO_CONNECT                                           5L




/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  BL_APIversion*/
/**/
/*     This function returns the version number of API functions as defined*/
/*     by BL_API_VERSION above.  This will force the LIB and users of the*/
/*     LIB to be in sync and prevent build problems.  Users of the BlackList*/
/*     LIB need to call this function and compare the result to*/
/*     BL_API_VERSION.*/
/**/
/*     Params:   none*/
/**/
/*     Returns:  BL_API_VERSION defined above*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY BL_APIversion (void);



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  BL_RegisterDevice*/
/**/
/*     This function needs to be called first to register a device.  Each*/
/*     device such as modem, fax needs to be registered.  The*/
/*     handle returned is used to identify the memory structure used while*/
/*     working with unsuccessful number status.  Also, some countries make*/
/*     distictions in the call restrictions for certain devices.  In this*/
/*     case there is typically a more severe restriction on fax machines*/
/*     then on modems.*/
/**/
/**/
/**/
/*     Params:   ulDeviceType      - The device type defined above.*/
/**/
/*               szDeviceName      - The name that the device is known by*/
/*                                   in the system such as in the registery.*/
/*                                   This will be used by the blacklist viewer*/
/*                                   in identifying status for a particular*/
/*                                   device/line.*/
/**/
/*               lpulDeviceHandle  - Pointer to where the ulDeviceHandle needs*/
/*                                   to be returned.  This is the device*/
/*                                   registration number.*/
/**/
/*               ulReserved        - A reserved location for future API expansion.*/
/**/
/**/
/*     Returns:  ULONG Successful:  0*/
/*                     Error:       BLERR_.... from the list above*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY BL_RegisterDevice  (ULONG ulDeviceType,
                                   char FAR *lpszDeviceName,
                                   ULONG FAR *lpulDeviceHandle,
                                   ULONG ulReserved);




/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  BL_Reset*/
/**/
/*     This function resets the blacklist LIB and reloads the WT settings.*/
/**/
/**/
/*     Params:   ulReserved - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG Successful:  0*/
/*                     Error:       BLERR_.... from the list above*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY BL_Reset (ULONG ulReserved);


/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  BL_ResetDevice*/
/**/
/*     This function resets call restrictions for a device, it does not*/
/*     reset the high level information about the device such as device*/
/*     type and name.*/
/**/
/**/
/*     Params:   ulDeviceHandle - The registered device handle.*/
/**/
/*               ulReserved - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG Successful:  0*/
/*                     Error:       BLERR_.... from the list above*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY BL_ResetDevice (ULONG ulDeviceHandle, ULONG ulReserved);



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  BL_RequestDialPermission*/
/**/
/*     This function is used to request permission to dial.  Permission is*/
/*     granted if BL_DIAL_GRANTED is returned.  In this case, a valid*/
/*     ulDialHandle will be returned which needs to be used in reporting*/
/*     BL_DialResults.*/
/**/
/*     In the case of a manual dial, permission will always be granted.*/
/*     This is used to obtain the ulDialHandle which can then be used*/
/*     to clean calling restrictions for the number if the manual dial*/
/*     succeeded.*/
/**/
/**/
/*     Params:   ulDeviceHandle  - The registered device handle.*/
/*               ulDialType      - The type of dial:  MANUAL_DIAL or AUTOMATIC_DIAL.*/
/*               lpszPhoneNumber - A pointer to the phone number that dialing is requested for.*/
/*               lpulDialStatus  - A pointer to return the status of the request:*/
/*                                 BL_DIAL_GRANTED       - OK to dial*/
/*                                 BL_NUMBER_BLACKLISTED - permission denied*/
/*                                 BL_NUMBER_DELAYED     - permission denied for*/
/*                                                         lpulDelayTime more seconds*/
/**/
/*               lpulDialHandle  - A pointer to the returned dial handle.*/
/*               lpulDelayTime   - A pointer to the returned delay time in seconds.*/
/*               ulReserved      - A reserved location for future API expansion.*/
/**/
/**/
/*     Returns:  ULONG Successful:  0*/
/*                     Error:       BLERR_.... from the list above*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY BL_RequestDialPermission (ULONG ulDeviceHandle,
                                         ULONG ulDialType,
                                         char  FAR *lpszPhoneNumber,
                                         ULONG FAR *lpulDialStatus,
                                         ULONG FAR *lpulDialHandle,
                                         ULONG FAR *lpulDelayTime,
                                         ULONG ulReserved);




/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/**/
/*  BL_DialResults*/
/**/
/*     This function is used to handle the results of a dial from the list*/
/*     above.  For manual dials, unsuccessful results do not change any*/
/*     phone number status, however, a successful result will clear call*/
/*     restrictions for auto-dialing.*/
/**/
/**/
/**/
/*     Params:   ulDeviceHandle - The registered device handle.*/
/*               ulDialHandle   - The dial handle from BL_RequestDialPermission.*/
/*               ulDialStatus   - The dial status defined above as:*/
/*                                   SUCCESSFUL*/
/*                                   NO_DIAL_TONE*/
/*                                   LINE_BUSY*/
/*                                   USER_ABORT*/
/*                                   NO_CONNECT*/
/*               ulReserved     - A reserved location for future API expansion.*/
/**/
/*     Returns:  ULONG Successful:  0*/
/*                     Error:       BLERR_.... from the list above*/
/**/
/*-----------------------------------------------------------------------------*/
ULONG APIENTRY BL_DialResults (ULONG ulDeviceHandle, ULONG ulDialHandle,
                               ULONG ulDialStatus, ULONG ulReserved);







/*-----------------------------------------------------------------------------*/
#endif /* if !defined (MWBLAPI_H)*/

