/*
 * screen.c --- screen package.
 *
 * Copyright (c) 1997-2002 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2002-05-25 21:14:02 Pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"

#ifdef HAVE_LIBNCURSES

#include	"screen.h"
#include	"langues.h"
#include	"hash.h"
#include	"pcpjtool.h"

#include	<string.h>	/* for strlen(...) */
#include	<ctype.h>	/* for tolower(...) */
#include	<stdlib.h>	/* For exit(...) */

/*----------------------------------------------------------*/
#define		LINE_SIZE		64
#define		NB_LINES		9

#define		STOP_X			8
#define		STOP_Y			4
/*----------------------------------------------------------*/
typedef	char		line_t[ LINE_SIZE ];
static line_t		Lines[ NB_LINES ];
static int		CurLine;
static int		CurPosition;
static int		FirstLine;
static const char*	StrCoupsRestants;

int	ScreenKeyboardCounter = 0 ;
/*----------------------------------------------------------*/

#ifdef __DJGPP__

/*
 * Attribut a : les 8 bits les moins significatifs donnent la couleur
 *		du caractere. Les bits 8-15 donnent la couleur du fond.
 *
 *	0 : noir	1 : bleu fonce		2 : vert
 *	3 : cyan	4 : rouge		5 : magenta
 *	6 : brun	7 : blanc		8 : gris
 *	9 : bleu	10: vert clair		11: cyan clair
 *	etc...
 */

typedef	unsigned char	attribut_t;

#define		ATTR_WHITE_ON_BLACK	0x07
#define		ATTR_RED_ON_BLACK	0x04

#define		ATTR_P_WHITE_ON_WHITE	0x7b
#define		ATTR_P_WHITE_ON_BLACK	0x6b
#define		ATTR_P_BLACK_ON_WHITE	0x70
#define		ATTR_P_BLACK_ON_BLACK	0x60

#else /* not __DJGPP__ */

int		Screen_Key;	/* Key typed by the user to enter
				 * function ScreenKeyBoard */

typedef unsigned long	attribut_t;

#define		ATTR_WHITE_ON_BLACK	COLOR_PAIR(0)

#define		ATTR_P_WHITE_ON_WHITE	COLOR_PAIR(1)
#define		ATTR_P_WHITE_ON_BLACK	COLOR_PAIR(2)
#define		ATTR_P_BLACK_ON_WHITE	COLOR_PAIR(3)
#define		ATTR_P_BLACK_ON_BLACK	COLOR_PAIR(4)

#define		ATTR_RED_ON_BLACK	COLOR_PAIR(5)

#include	<curses.h>

#define	ScreenClear()			erase()
#define	ScreenPutChar(c, a, x, y)	mvaddch(y, x, a | c)
#define	ScreenPutString(s, a, x, y)	{attrset(a); mvaddstr(y, x, s);}
#define	ScreenSetCursor(x, y)		move(y, x)

#endif /* not __DJGPP__ */

static void
screenPosition(
    const square_t*	board,
    const piece_type_t*	types,
    int			number_of_half_moves)
{
    int			i, j;
    piece_index_t	piece_index;
    int			nbBlanc, nbNoir;
    char		s[60];
    attribut_t		attr;
    const char* const	initialePieces = langGetString( eStrInitialOfPieces );

    nbBlanc = nbNoir = 0;
    for ( i = 7; i >= 0; i-- )
    {
	for ( j = 0; j < 8; j++ )
	{
	    piece_index = board[ i * 16 + j ];
	    if ( piece_index != 0 )
	    {
		if ( is_white( piece_index ) )
		{
		    nbBlanc++;
		    if ( ( i + j ) & 1 )	attr = ATTR_P_WHITE_ON_WHITE;
		    else			attr = ATTR_P_WHITE_ON_BLACK;
		}
		else
		{
		    nbNoir++;
		    if ( ( i + j ) & 1 )	attr = ATTR_P_BLACK_ON_WHITE;
		    else			attr = ATTR_P_BLACK_ON_BLACK;
		}
		ScreenPutChar( initialePieces[ types[ piece_index ] ],
			       attr, j, 7 - i );
	    }
	    else	/* case vide */
	    {
		if ( ( i + j ) & 1 )	attr = ATTR_P_WHITE_ON_WHITE;
		else			attr = ATTR_P_WHITE_ON_BLACK;
		ScreenPutChar( ' ', attr, j, 7 - i );
	    }
	}
    }

    sprintf( s, "%s %s", PACKAGE, VERSION );
    if ( strlen( s ) > 9 )
    {
	s[ 9 ] = 0 ;
    }
    ScreenPutString( s, ATTR_WHITE_ON_BLACK, 8, 0 );
    sprintf( s, "%s %s", PACKAGE, VERSION );
    if ( strlen( s ) > 9 )
    {
	strcpy( s, &s[ 9 ] );
	ScreenPutString( s, ATTR_WHITE_ON_BLACK, 9, 1 );
    }
    sprintf(s, "%d %s", number_of_half_moves, langGetString(eStrHalfMoves));
    ScreenPutString(s, ATTR_WHITE_ON_BLACK, 0, 8);
    sprintf(s, "%2d + %d", nbBlanc, nbNoir);
    ScreenPutString(s, ATTR_WHITE_ON_BLACK, 9, 7);

#ifndef __DJGPP__
    refresh();
#endif /* not __DJGPP__ */

    for ( i = 0; i < NB_LINES; i++ )
    {
	Lines[ i ][ LINE_SIZE - 1 ] = '\0';
    }
}

void
screenWriteAssoc(const piece_t* p)
{
    int		i;
    char	s[60];
    int		x, y;

    getOneAssociation(p, s);
    for (i=strlen(s); i<40; i++)	s[i] = ' ';
    s[i] = '\0';

    if (p->index < INDEX_ROI_NOIR)	x = 0;
    else				x = 40;
	
    y = 9 + (p->index & 0x0F);
	
    ScreenPutString(s, ATTR_WHITE_ON_BLACK, x, y);
}

void
screenInit(
    const square_t*	board,
    const piece_type_t*	types,
    unsigned int	number_of_half_moves,
    const piece_t*	pieces,
    unsigned int	nb_pieces)
{
    int		i;

#ifndef	__DJGPP__
    initscr();

    start_color();

    cbreak();
    noecho();
    nodelay(stdscr, TRUE);
    nonl();
    keypad(stdscr, TRUE);
    intrflush(stdscr, FALSE);

    /* COLOR_YELLOW is used because it gives BROWN on i386 !
     * See documentation in curs_color(3X).
     */
    init_pair(1, COLOR_CYAN,  COLOR_WHITE);	/* White on white square */
    init_pair(2, COLOR_CYAN,  COLOR_YELLOW);	/* White on black square */
    init_pair(3, COLOR_BLACK, COLOR_WHITE);	/* Black on white square */
    init_pair(4, COLOR_BLACK, COLOR_YELLOW);	/* Black on black square */
    init_pair(5, COLOR_RED,   COLOR_BLACK);	/* Red on black background */

    /* Set invisible cursor : if this doesn't work : the result of the function
       will be ERR, we just ignore it. */
    curs_set( 0 );
#endif /* not __DJGPP__ */

    ScreenClear();
    ScreenSetCursor( 0, 0 );
    screenPosition( board, types, number_of_half_moves );
    for ( i = 0; i < nb_pieces; i++ )
    {
	screenWriteAssoc( &pieces[ i ] );
    }
    StrCoupsRestants = langGetString( eStrCoupsRestants );

    screenCoup( 0, 0, NULL );	/* Affiche "Recherche position" + refresh */
}

void
screen(
    const piece_t*	pieces,
    unsigned int	nb_pieces,
    unsigned long	nb_position,
    unsigned int	white_spare_moves,
    unsigned int	black_spare_moves)
{
    int			i;
    unsigned char	s[ 20 ];

    for( i = 0; i < nb_pieces; i++ )
    {
	screenWriteAssoc( &pieces[ i ] );
    }

    sprintf( s, "%ld", nb_position );
    ScreenPutString( s, ATTR_WHITE_ON_BLACK, 9, 2 );

    sprintf( s, StrCoupsRestants, white_spare_moves, black_spare_moves );
    if ( strlen( s ) > 9 )	s[ 9 ] = '\0';
    ScreenPutString( s, ATTR_WHITE_ON_BLACK, 8, 3 );

#ifndef __DJGPP__
    refresh();
#endif /* not __DJGPP__ */
}

void
screenCoup(int i, int nb, const UCHAR* coup)
{
    char	s[60];

    if (i == 0)
    {
	ScreenPutString(langGetString(eStrRecherche),
			ATTR_WHITE_ON_BLACK, 8, 5);
	ScreenPutString(langGetString(eStrPosition),
			ATTR_WHITE_ON_BLACK, 8, 6);
    }
    else
    {
	sprintf(s, " %2d / %d  ", i, nb);
	s[9] = '\0';		/* Il peut y avoir 1 caracteres de trop. */
	ScreenPutString(s, ATTR_WHITE_ON_BLACK, 8, 5);
	(void)move_to_string(coup, &s[1]);
	s[0] = ' ';
	s[9] = '\0';		/* Le coup a 9 caracteres donc 1 de trop */
	ScreenPutString(s, ATTR_WHITE_ON_BLACK, 8, 6);
    }

#ifndef __DJGPP__
    refresh();
#endif /* not __DJGPP__ */
}

/*------------------------------------------------------------------*/

static void
screenPrintLines()
{
    int		i;

    for (i=0; i<NB_LINES; i++)
    {
	ScreenPutString(Lines[FirstLine], ATTR_WHITE_ON_BLACK, 17, i);
	FirstLine++;
	if (FirstLine == NB_LINES)	FirstLine = 0;
    }

#ifndef __DJGPP__
    refresh();
#endif /* not __DJGPP__ */
}

static void
screenIncLine()
{
    CurPosition = 0;
    CurLine++;
    if (CurLine == NB_LINES)	CurLine = 0;
    if (CurLine == FirstLine)
    {
	FirstLine++;
	if (FirstLine == NB_LINES)	FirstLine = 0;
    }
}

/*------------------------------------------------------------------*/

void
screenPrintListeCoups(
    unsigned long	nb_solution,
    unsigned long	see_solution_number,
    unsigned long	nb_position,
    const UCHAR*	moves,
    unsigned int	nb_moves)
{
    int		i, j;
    char	s[ 60 ];

    if (CurPosition)	screenIncLine();

    sprintf( Lines[ CurLine ], "%s %ld (%s %ld)",
	     langGetString( eStrSolution ), nb_solution ,
	     langGetString( eStrPosition ), nb_position );
    for ( i = strlen( Lines[ CurLine ] ); i < LINE_SIZE - 1; i++ )
    {
	Lines[ CurLine ][ i ] = ' ';
    }

    screenIncLine();
	
    j = 0;
    for ( i = 0; i < nb_moves; i++ )
    {
	if ((i & 1) == 0)
	{
	    sprintf( s, "%2d.", ++j );
	    strcpy( &Lines[ CurLine ][ CurPosition ], s );
	    CurPosition += 3;
	}

	(void)move_to_string( &moves[ i * 6 ], s );

	strcpy( &Lines[CurLine ][ CurPosition ], s );
	CurPosition += strlen( s );

	if ( CurPosition >= LINE_SIZE - 1)	screenIncLine();
    }

    if ( see_solution_number != 0 )
    {
	if (CurPosition > LINE_SIZE - 12)
	{
	    strcpy( &Lines[ CurLine ][ CurPosition ], "...     ");
	    screenIncLine();
	    sprintf( Lines[ CurLine ], "%s %ld",
		     langGetString( eStrSeeSolution ), see_solution_number );
	    CurPosition = 15;
	}
	else
	{
	    sprintf( &Lines[ CurLine ][ CurPosition ], "... %s %ld",
		    langGetString( eStrSeeSolution ), see_solution_number );
	}
    }

    for ( i = strlen( Lines[ CurLine ] ); i < LINE_SIZE - 1; i++ )
    {
	Lines[ CurLine ][ i ] = ' ';
    }

    screenPrintLines();
}

void
screenEnd(const char* begin,
	  const char* end,
	  const char* stopped,
	  const char* time)
{
    int		i;
    char*	s = "                                                                                ";
    char*	s2 = "         ";

    ScreenSetCursor(9, 0);

    for (i=0; i<16; i++)
    {
	ScreenPutString(s, ATTR_WHITE_ON_BLACK, 0, 9+i);
    }

    ScreenPutString(s2, ATTR_WHITE_ON_BLACK, 8, 5);
    ScreenPutString(s2, ATTR_WHITE_ON_BLACK, 8, 6);

    ScreenPutString(begin,   ATTR_WHITE_ON_BLACK, 0, 10);
    ScreenPutString(end,     ATTR_WHITE_ON_BLACK, 0, 11);
    ScreenPutString(stopped, ATTR_WHITE_ON_BLACK, 0, 12);
    ScreenPutString(time,    ATTR_WHITE_ON_BLACK, 0, 13);

#ifndef __DJGPP__
    refresh();
    curs_set( 1 );		/* Reset cursor back to normal */
    endwin();
#else /* not __DJGPP__ */
    ScreenSetCursor(14, 0);
#endif /* __DJGPP__ */

}

bool_t
screenKeyboard(
    bool_t		skipOK,
    const UCHAR*	moves,
    unsigned int	nb_moves,
    unsigned long	nb_positions )
{
    int			key;
    const char* const	stopTxt = langGetString(eStrStop);
    const char* const	skipTxt = langGetString(eStrSkip);
    const char*		texte;
    const char* const	empty = "         ";
    bool_t		skip;

#ifdef __DJGPP__
    key = getkey();
    while (kbhit())	getkey();	/* Flush input */
#else /* __DJGPP__ */
    key = Screen_Key;
    while (getch() != ERR);		/* Flush input */
#endif /* not __DJGPP__ */

    key = tolower(key);

    if (key == ' ')
    {
	/* Print the current moves in short format on the screen */
	static	char	change_indicator = ' ';
	char		string[ LINE_SIZE ];
	char		one_move[ 10 ];
	int		i;

	string[0] = '\0';
	for ( i = 0; i < nb_moves &&  i < 15; i++ )
	{
	    strcat( string, move_to_short_string( &moves[ i * 6], one_move ) );
	}
	for (; i < 15; i++ )
	{
	    strcat( string, "    " );
	}
	ScreenPutString( string, ATTR_WHITE_ON_BLACK, 17, 7 );

	string[ 0 ] = '\0';
	for (; i < nb_moves && i < 30; i++ )
	{
	    strcat( string, move_to_short_string( &moves[ i * 6 ], one_move ) );
	}
	for (; i < 30; i++ )
	{
	    strcat( string, "    " );
	}

	change_indicator = ' ' + '*' - change_indicator;
	i = strlen( string );
	string[ i ] = change_indicator;
	string[ i + 1 ] = '\0';

	ScreenPutString( string, ATTR_WHITE_ON_BLACK, 17, 8 );

	return FALSE;
    }

    if ( skipOK && key == 's' )
    {
	texte = skipTxt;
	skip = TRUE;
    }
    else
    {
	texte = stopTxt;
	skip = FALSE;
    }

    ScreenPutString( texte, ATTR_RED_ON_BLACK, STOP_X, STOP_Y );

#ifdef	__DJGPP__
    key = getkey();
#else /* __DJGPP__ */
    refresh();
    while ((key = getch()) == ERR);
#endif	/* not __DJGPP__ */

    key = tolower( key );

    ScreenPutString( empty, ATTR_WHITE_ON_BLACK, STOP_X, STOP_Y );

#ifndef __DJGPP__
    refresh();
#endif	/* not __DJGPP__ */

    if ( key == langGetChar( eCharYes ) )
    {
	if ( skip )
	{
	    return TRUE ;
	}
	else
	{
	    mainEnd( eStrStoppedByUser, nb_positions );
	    if ( skipOK )	hashFree();
	    exit( EXIT_SUCCESS );
	}
    }

    return FALSE;
}

#endif /* HAVE_LIBNCURSES */
