/*
 * hash.c --- hash package.
 *
 * Copyright (c) 1997, 98, 99, 2000, 2001 by Pascal Wassong All Rights Reserved.
 *
 * Time-stamp: <2001-12-29 14:41:39 Pascal>
 *
 * This file is part of Natch.
 *
 * Natch is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Natch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include	"common.h"

#define	EXTERN_HASH	/* Only defined here */
#include	"hash.h"

#include	"myHashTable.h"
#include	"pcpj.h"

#include	<string.h>			/* For memcpy(...) */
#include	<time.h>			/* for clock() */

static struct hash_table_t*	MyHashTable;
#define		PROMOTION_OR_CASTLING_BIT		0x80;

/*  #define HASH_STATS */
#ifdef HASH_STATS
static unsigned long	NumberOfTests ;
static unsigned long	NumberOfPositiveTests ;
#endif /* HASH_STATS */

static	unsigned int	Max_moves_to_insert ;

static unsigned long	Nb_Indexes ;
static unsigned long	Max_memory_size ;

/* Count the number of positions in the hashtable for each game length.
 * This allows compression of the hashtable in one pass.
 */
#define	Max_number_of_moves	256
static unsigned long		Number_of_Positions[ Max_number_of_moves ];

static unsigned char*
hashKey( const exploration_t* datas )
{
    static unsigned char	CurrentKey[ LG_KEY ];
    static unsigned char*	PositionKey = &CurrentKey[ 1 ];

    unsigned int	i = 1 ;
    const piece_t*	p = &( datas->pieces[ INDEX_ROI_BLANC ] );

    CurrentKey[ 0 ] = CoupsNb ;
    memcpy( PositionKey,
	    &(datas->pieces_squares[ INDEX_ROI_BLANC ]),
	    Hash_Key_Length - 1 );

    while ( i < Hash_Key_Length )
    {
	if ( ( p->casePromotion != CASE_PAS_DE_PROMOTION &&
	       datas->pieces_types[ INDEX_ROI_BLANC + i - 1 ] != PION ) ||
	     ( datas->pieces_types[ INDEX_ROI_BLANC + i - 1 ] == ROI &&
	       p->castling != NO_CASTLING ) )
	{
	    CurrentKey[ i ] |= PROMOTION_OR_CASTLING_BIT ;
	}
	i++;
	p++;
    }

    return CurrentKey ;
} /* hashKey */

bool_t
hashTest(
    const exploration_t*	datas      ,
    unsigned long*		hash_value ,
    unsigned int*		solution   )
{
    unsigned char*	key = hashKey( datas );
    hash_element_t*	he ;

    *hash_value = myHash_hash_value( key );
    he = dhtLookupElement( MyHashTable, *hash_value, key );

/*      clock_t		ticks = clock(); */

#ifdef HASH_STATS
    NumberOfTests++;
#endif

    if (he != dhtNilElement)
    {
#ifdef HASH_STATS
	NumberOfPositiveTests++;
#endif

	*solution = (unsigned int)(he->data);

/*  	MainTicks2 += clock() - ticks; */
	return TRUE;
    }
    else
    {
	*solution = 0;

/*  	MainTicks2 += clock() - ticks; */
	return FALSE;
    }
} /* hashTest */

static void
hashCompress()
{
    /* Remove 1/16 = ~ 6% */
    unsigned long	toDelete = ( dhtKeyCount( MyHashTable ) >> 4 ) + 1 ;

    unsigned char	min_val  = Max_moves_to_insert ;
    unsigned long	total    = 0 ;
    hash_element_t*	he       = dhtGetFirstElement( MyHashTable );

    if ( toDelete >= dhtKeyCount( MyHashTable ) )
    {
	toDelete = dhtKeyCount( MyHashTable );
	/* this is pathologic: it may only occur, when we are so
	 * low on memory, that only one or no position can be stored.
	 */
    }

    while ( total < toDelete )
    {
	min_val--;
#ifdef HASH_STATS
	fprintf( MainFD, "Remove %ld positions of length %d\n",
		 Number_of_Positions[ min_val ], min_val );
#endif
	total += Number_of_Positions[ min_val ];
	Number_of_Positions[ min_val ] = 0 ;
    }
#ifdef HASH_STATS
    fprintf( MainFD, "Must remove min_val=%d, nb=%ld\n", min_val, total );
#endif

    while ( he )
    {
	unsigned char	number = ((unsigned char*)(he->key))[ 0 ];
	if (number >= min_val)
	{
	    myHash_remove_current_element( MyHashTable );
	}
	he = dhtGetNextElement(MyHashTable);
    }
} /* hashCompress */

void
hashInsert(
    const exploration_t*	datas      ,
    unsigned long		hash_value ,
    unsigned int		solution   )
{
    unsigned char*	key ;
/*      clock_t		ticks = clock(); */

    if ( CoupsNb < 3 || CoupsNb >= Max_moves_to_insert )	return;

    if ( dhtKeyCount( MyHashTable ) >= myHash_nb_max_elements( MyHashTable ) )
    {
	hashCompress();
    }

    key = hashKey( datas );

    dhtEnterElement( MyHashTable, hash_value, key, solution );

    Number_of_Positions[ key[ 0 ] ]++;
/*      MainTicks2 += clock() - ticks; */

} /* hashInsert */

void
hashSetSize( unsigned int n )
{
    int		number_max_indexes ;
    Max_memory_size = n * 1024 * 1024 ;
    number_max_indexes = ( Max_memory_size / 4 ) * 0.15 ;
    Nb_Indexes = 0xFFFFFFFF ;
    while ( Nb_Indexes > number_max_indexes )
    {
	Nb_Indexes >>= 1 ;
    }
    Nb_Indexes++;
} /* hashSetSize */

void
hashInit( unsigned int	single_moves ,
	  unsigned int	lg_key       )
{
    unsigned int	i ;

    Hash_Key_Length = lg_key + 1 ;

    if ( Nb_Indexes == 0 )	return;

    if ( single_moves > 11 )
    {
	Max_moves_to_insert = single_moves - 6 ;
    }
    else
    {
	Max_moves_to_insert = single_moves ;
    }

    MyHashTable = dhtCreate( Hash_Key_Length, Nb_Indexes, Max_memory_size );
    if ( MyHashTable == NilHashTable )
    {
	fprintf(MainFD, "Error while creating MyHashTable\n");
    }

    for ( i = 0 ; i < Max_number_of_moves ; i++ )
    {
	Number_of_Positions[ i ] = 0 ;
    }

#ifdef HASH_STATS
    NumberOfTests = NumberOfPositiveTests = 0 ;
#endif
} /* hashInit */

void
hashFree()
{
#ifdef HASH_STATS
    fprintf( MainFD, "Hit ratio : %ld / %ld = %ld%%\n",
	     NumberOfPositiveTests, NumberOfTests,
	     NumberOfTests ? ( NumberOfPositiveTests * 100 ) / NumberOfTests : 0 );
#endif /* HASH_STATS */

    if ( MyHashTable )	dhtDestroy( MyHashTable );
} /* hashFree */
