// Copyright 2018 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "google/cloud/future.h"
#include "google/cloud/internal/throw_delegate.h"
#include "google/cloud/testing_util/chrono_literals.h"
#include "google/cloud/testing_util/expect_future_error.h"
#include <gmock/gmock.h>
#include <functional>

namespace google {
namespace cloud {
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_BEGIN
namespace {
using ::testing::HasSubstr;
using ::google::cloud::testing_util::chrono_literals::operator"" _ms;  // NOLINT
using ::google::cloud::testing_util::ExpectFutureError;

TEST(FutureTestInt, ThenSimple) {
  promise<int> p;
  future<int> fut = p.get_future();
  EXPECT_TRUE(fut.valid());

  bool called = false;
  future<int> next = fut.then([&called](future<int> r) {
    called = true;
    return 2 * r.get();
  });
  EXPECT_FALSE(fut.valid());
  EXPECT_TRUE(next.valid());
  EXPECT_FALSE(called);

  p.set_value(42);
  EXPECT_TRUE(called);
  EXPECT_TRUE(next.valid());
  ASSERT_EQ(std::future_status::ready, next.wait_for(0_ms));

  EXPECT_EQ(84, next.get());
  EXPECT_FALSE(next.valid());
}

// With exceptions disabled this test is not very useful. Yes,
// `internal::ThrowRuntimeError()` is called and that terminates the
// application. We knew that already.
#if GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
TEST(FutureTestInt, ThenException) {
  promise<int> p;
  future<int> fut = p.get_future();
  EXPECT_TRUE(fut.valid());

  bool called = false;
  future<int> next = fut.then([&called](future<int> r) {
    called = true;
    int value = r.get();
    if (value == 42) {
      internal::ThrowRuntimeError("test message");
    }
    return 2 * value;
  });
  EXPECT_FALSE(fut.valid());
  EXPECT_TRUE(next.valid());
  EXPECT_FALSE(called);

  p.set_value(42);
  EXPECT_TRUE(called);
  EXPECT_TRUE(next.valid());
  ASSERT_EQ(std::future_status::ready, next.wait_for(0_ms));

  EXPECT_THROW(
      try { next.get(); } catch (std::runtime_error const& ex) {
        EXPECT_THAT(ex.what(), HasSubstr("test message"));
        throw;
      },
      std::runtime_error);
  EXPECT_FALSE(next.valid());
}
#endif  // GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS

TEST(FutureTestInt, ThenUnwrap) {
  promise<int> p;
  future<int> fut = p.get_future();
  EXPECT_TRUE(fut.valid());

  promise<std::string> pp;
  bool called = false;
  auto cont = [&pp, &called](future<int>) {
    called = true;
    return pp.get_future();
  };
  future<std::string> next = fut.then(std::move(cont));
  EXPECT_FALSE(fut.valid());
  EXPECT_TRUE(next.valid());
  EXPECT_FALSE(next.is_ready());

  p.set_value(42);
  EXPECT_TRUE(called);
  EXPECT_FALSE(next.is_ready());

  pp.set_value("value=42");
  EXPECT_TRUE(next.is_ready());
  EXPECT_EQ("value=42", next.get());
  EXPECT_FALSE(next.valid());
}

TEST(FutureTestInt, ThenMoveOnlyCallable) {
  class MoveOnlyDoubler {
   public:
    explicit MoveOnlyDoubler(bool& called) : called_(&called) {}

    MoveOnlyDoubler(MoveOnlyDoubler&&) = default;
    MoveOnlyDoubler& operator=(MoveOnlyDoubler&&) = default;

    MoveOnlyDoubler(MoveOnlyDoubler const&) = delete;
    MoveOnlyDoubler& operator=(MoveOnlyDoubler const&) = delete;

    int operator()(future<int> f) {
      *called_ = true;
      return 2 * f.get();
    }

   private:
    bool* called_;
  };

  promise<int> p;
  future<int> fut = p.get_future();
  EXPECT_TRUE(fut.valid());

  bool called = false;
  MoveOnlyDoubler cb{called};
  future<int> next = fut.then(std::move(cb));
  EXPECT_FALSE(fut.valid());
  EXPECT_TRUE(next.valid());
  EXPECT_FALSE(called);

  p.set_value(42);
  EXPECT_TRUE(called);
  EXPECT_TRUE(next.valid());
  ASSERT_EQ(std::future_status::ready, next.wait_for(0_ms));

  EXPECT_EQ(2 * 42, next.get());
  EXPECT_FALSE(next.valid());
}

TEST(FutureTestInt, ThenByCopy) {
  promise<int> p;
  future<int> fut = p.get_future();
  EXPECT_TRUE(fut.valid());

  bool called = false;
  auto doubler = [&called](future<int> r) {
    called = true;
    return 2 * r.get();
  };
  future<int> next = fut.then(doubler);
  EXPECT_FALSE(fut.valid());
  EXPECT_TRUE(next.valid());
  EXPECT_FALSE(called);

  p.set_value(42);
  EXPECT_TRUE(called);
  EXPECT_TRUE(next.valid());
  ASSERT_EQ(std::future_status::ready, next.wait_for(0_ms));

  EXPECT_EQ(2 * 42, next.get());
  EXPECT_FALSE(next.valid());
}

/// @test Verify the behavior around cancellation.
TEST(FutureTestInt, CancelThroughContinuation) {
  bool cancelled = false;
  promise<int> p0([&cancelled] { cancelled = true; });
  auto f0 = p0.get_future();
  auto f1 = f0.then([](future<int> f) { return f.get() * 2; });
  EXPECT_TRUE(f1.cancel());
  EXPECT_TRUE(cancelled);
  p0.set_value(42);
  EXPECT_EQ(84, f1.get());
}

TEST(FutureTestInt, CancelThroughUnwrappingContinuation) {
  bool cancelled = false;
  promise<int> p0([&cancelled] { cancelled = true; });
  auto f0 = p0.get_future();
  auto f1 = f0.then(
      [](auto g) { return g.then([](auto h) { return h.get() * 2; }); });
  EXPECT_TRUE(f1.cancel());
  EXPECT_TRUE(cancelled);
  p0.set_value(42);
  EXPECT_EQ(f1.get(), 84);
}

TEST(FutureTestInt, CancelThroughConverted) {
  bool cancelled = false;
  promise<int> p0([&cancelled] { cancelled = true; });
  future<std::int64_t> f1(
      p0.get_future().then([](auto h) { return h.get() * 2; }));
  EXPECT_TRUE(f1.cancel());
  EXPECT_TRUE(cancelled);
  p0.set_value(42);
  EXPECT_EQ(f1.get(), 84);
}

/// @test Verify that `.then()` can return its argument.
TEST(FutureTestInt, ThenReturnsArgument) {
  promise<int> p;
  int counter = 0;
  auto f = p.get_future().then([&counter](auto g) {
    ++counter;
    return g;
  });
  p.set_value(42);
  EXPECT_TRUE(f.is_ready());
  EXPECT_EQ(counter, 1);
  EXPECT_EQ(f.get(), 42);
}

/// @test Verify that `.then()` can return its argument.
TEST(FutureTestInt, ThenAttachesContinuationToArgument) {
  promise<int> p;
  auto f = p.get_future().then(
      [](auto g) { return g.then([](auto h) { return h.get() * 2; }); });
  p.set_value(42);
  EXPECT_TRUE(f.is_ready());
  EXPECT_EQ(f.get(), 84);
}

/// @test Verify that `.then()` continuations are notified on abandoned futures.
TEST(FutureTestInt, AbandonNotifiesContinuation) {
  future<int> f;
  {
    promise<int> p;
    f = p.get_future().then([](auto g) {
#if GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
      EXPECT_THROW(
          try { g.get(); } catch (std::future_error const& ex) {
            EXPECT_EQ(std::future_errc::broken_promise, ex.code());
            throw;
          },
          std::future_error);
#else
      EXPECT_DEATH_IF_SUPPORTED(g.get(), "exceptions are disabled");
#endif  // GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
      return 42;
    });
  }
  EXPECT_EQ(f.get(), 42);
}

// The following tests reference the technical specification:
//   http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2015/p0159r0.html
// The test names match the section and paragraph from the TS.

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_2_a) {
  // future<T> should have an unwrapping constructor.
  promise<future<int>> p;
  future<future<int>> f = p.get_future();

  future<int> unwrapped(std::move(f));
  EXPECT_FALSE(noexcept(future<int>(p.get_future())));
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_3_a) {
  // A future<T> created via the unwrapping constructor becomes satisfied
  // when both become satisfied.
  promise<future<int>> p;

  future<int> unwrapped(p.get_future());
  EXPECT_TRUE(unwrapped.valid());
  EXPECT_FALSE(unwrapped.is_ready());

  promise<int> p2;
  p.set_value(p2.get_future());
  EXPECT_FALSE(unwrapped.is_ready());

  p2.set_value(42);
  EXPECT_TRUE(unwrapped.is_ready());
  EXPECT_EQ(42, unwrapped.get());
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_3_b) {
  // A future<T> created via the unwrapping constructor becomes satisfied
  // when the wrapped future is satisfied by an exception.
  promise<future<int>> p;

  future<int> unwrapped(p.get_future());
  EXPECT_TRUE(unwrapped.valid());
  EXPECT_FALSE(unwrapped.is_ready());

  p.set_exception(std::make_exception_ptr(std::runtime_error("test message")));
  EXPECT_TRUE(unwrapped.is_ready());
#if GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
  EXPECT_THROW(
      try { unwrapped.get(); } catch (std::runtime_error const& ex) {
        EXPECT_THAT(ex.what(), HasSubstr("test message"));
        throw;
      },
      std::runtime_error);
#else
  EXPECT_DEATH_IF_SUPPORTED(
      unwrapped.get(),
      "future<T>::get\\(\\) had an exception but exceptions are disabled");
#endif  // GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_3_c) {
  // A future<T> created via the unwrapping constructor becomes satisfied
  // when the inner future is satisfied by an exception.
  promise<future<int>> p;

  future<int> unwrapped(p.get_future());
  EXPECT_TRUE(unwrapped.valid());
  EXPECT_FALSE(unwrapped.is_ready());

  promise<int> p2;
  p.set_value(p2.get_future());
  EXPECT_FALSE(unwrapped.is_ready());

  p2.set_exception(std::make_exception_ptr(std::runtime_error("test message")));
  EXPECT_TRUE(unwrapped.is_ready());
#if GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
  EXPECT_THROW(
      try { unwrapped.get(); } catch (std::runtime_error const& ex) {
        EXPECT_THAT(ex.what(), HasSubstr("test message"));
        throw;
      },
      std::runtime_error);
#else
  EXPECT_DEATH_IF_SUPPORTED(
      unwrapped.get(),
      "future<T>::get\\(\\) had an exception but exceptions are disabled");
#endif  // GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_3_d) {
  // A future<T> created via the unwrapping constructor becomes satisfied
  // when the inner future is invalid.
  promise<future<int>> p;

  future<int> unwrapped(p.get_future());
  EXPECT_TRUE(unwrapped.valid());
  EXPECT_FALSE(unwrapped.is_ready());

  promise<int> p2;
  p.set_value(future<int>{});
  EXPECT_TRUE(unwrapped.is_ready());
#if GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
  EXPECT_THROW(
      try { unwrapped.get(); } catch (std::future_error const& ex) {
        EXPECT_EQ(std::future_errc::broken_promise, ex.code());
        throw;
      },
      std::future_error);
#else
  EXPECT_DEATH_IF_SUPPORTED(
      unwrapped.get(),
      "future<T>::get\\(\\) had an exception but exceptions are disabled");
#endif  // GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_4) {
  // future<T> should leaves the source invalid.
  promise<future<int>> p;
  future<future<int>> f = p.get_future();

  future<int> unwrapped(std::move(f));
  EXPECT_TRUE(unwrapped.valid());
  EXPECT_FALSE(f.valid());  // NOLINT(bugprone-use-after-move)
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_5) {
  // future<int>::then() is a template member function that takes callables.
  future<int> f;
  auto callable = [](future<int> f) -> int { return 2 * f.get(); };
  EXPECT_TRUE((std::is_same<future<int>, decltype(f.then(callable))>::value));

  auto void_callable = [](future<int> f) { f.get(); };
  EXPECT_TRUE(
      (std::is_same<future<void>, decltype(f.then(void_callable))>::value));

  auto long_callable =
      [](future<int> f) -> long {  // NOLINT(google-runtime-int)
    return 2 * f.get();
  };
  EXPECT_TRUE(
      (std::is_same<future<long>, decltype(f.then(long_callable))>::value));

  auto string_callable = [](future<int>) -> std::string { return "42"; };
  EXPECT_TRUE((std::is_same<future<std::string>,
                            decltype(f.then(string_callable))>::value));
}

// Use SFINAE to test if future<int>::then() accepts a T parameter.
template <typename T>
auto test_then(int) -> decltype(std::declval<future<int>>().then(T()),
                                std::true_type{}) {
  return std::true_type{};
}

template <typename T>
auto test_then(long) -> std::false_type {  // NOLINT(google-runtime-int)
  return std::false_type{};
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_7) {
  // future<int>::then() requires callables that take future<int> as a
  // parameter.
  future<int> f;
  using valid_callable_type = std::function<void(future<int>)>;
  using invalid_callable_type = std::function<void(int)>;

  EXPECT_TRUE(decltype(test_then<valid_callable_type>(0))::value);
  EXPECT_FALSE(decltype(test_then<invalid_callable_type>(0))::value);

  EXPECT_TRUE(decltype(test_then<std::function<int(future<int>)>>(0))::value);
  EXPECT_FALSE(decltype(test_then<std::function<int()>>(0))::value);
  EXPECT_TRUE(
      decltype(test_then<std::function<std::string(future<int>)>>(0))::value);
  EXPECT_FALSE(decltype(test_then<std::function<std::string()>>(0))::value);
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_8_a) {
  // future<int>::then() creates a future with a valid shared state.
  promise<int> p;
  future<int> f = p.get_future();

  future<void> next = f.then([&](future<int>) {});
  EXPECT_TRUE(next.valid());
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_8_b) {
  // future<int>::then() calls the functor when the future becomes ready.
  promise<int> p;
  future<int> f = p.get_future();

  bool called = false;
  future<void> next = f.then([&](future<int>) { called = true; });
  EXPECT_TRUE(next.valid());
  EXPECT_FALSE(called);

  p.set_value(42);
  EXPECT_TRUE(called);
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_8_c) {
  // future<int>::then() calls the functor if the future was ready.
  promise<int> p;
  future<int> f = p.get_future();

  p.set_value(42);
  bool called = false;
  future<void> next = f.then([&](future<int>) { called = true; });
  EXPECT_TRUE(next.valid());
  EXPECT_TRUE(called);
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_8_d) {
  // future<int>::then() propagates the value from the functor to the returned
  // future.
  promise<int> p;
  future<int> f = p.get_future();

  future<int> next = f.then([&](future<int> r) { return 2 * r.get(); });
  EXPECT_TRUE(next.valid());
  p.set_value(42);
  ASSERT_EQ(std::future_status::ready, next.wait_for(0_ms));
  EXPECT_EQ(84, next.get());
}

#if GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_8_e) {
  // future<int>::then() propagates exceptions raised by the functort to the
  // returned future.
  promise<int> p;
  future<int> f = p.get_future();

  future<void> next = f.then([&](future<int>) {
    internal::ThrowRuntimeError("test exception in functor");
  });
  EXPECT_TRUE(next.valid());
  p.set_value(42);
  ASSERT_EQ(std::future_status::ready, next.wait_for(0_ms));
  EXPECT_THROW(
      try { next.get(); } catch (std::runtime_error const& ex) {
        EXPECT_THAT(ex.what(), HasSubstr("test exception in functor"));
        throw;
      },
      std::runtime_error);
  EXPECT_FALSE(next.valid());
}
#endif  // GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_9_a) {
  // future<int>::then() returns a functor containing the type of the value
  // returned by the functor.
  promise<int> p;
  future<int> f = p.get_future();

  auto returns_void = [](future<int>) -> void {};
  EXPECT_TRUE(
      (std::is_same<future<void>, decltype(f.then(returns_void))>::value));

  auto returns_int = [](future<int>) -> int { return 42; };
  EXPECT_TRUE(
      (std::is_same<future<int>, decltype(f.then(returns_int))>::value));

  auto returns_string = [](future<int>) -> std::string { return "42"; };
  EXPECT_TRUE((std::is_same<future<std::string>,
                            decltype(f.then(returns_string))>::value));
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_9_b) {
  // future<int>::then() implicitly unwraps the future type when a functor
  // returns a future<>.
  promise<int> p;
  future<int> f = p.get_future();

  auto returns_void = [](future<int>) -> future<void> {
    return promise<void>().get_future();
  };
  EXPECT_TRUE(
      (std::is_same<future<void>, decltype(f.then(returns_void))>::value));

  auto returns_int = [](future<int>) -> future<int> {
    return promise<int>().get_future();
  };
  EXPECT_TRUE(
      (std::is_same<future<int>, decltype(f.then(returns_int))>::value));

  // The spec says the returned type must be future<R2> *exactly*, references do
  // not count.
  promise<int> p_int;
  future<int> f_int = p_int.get_future();
  auto returns_int_ref = [&f_int](future<int>) -> future<int>& {
    return f_int;
  };
  EXPECT_FALSE(
      (std::is_same<future<int>, decltype(f.then(returns_int_ref))>::value));
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_9_c) {
  // future<int>::then() implicitly unwrapping captures the returned value.
  promise<int> p;
  future<int> f = p.get_future();

  promise<int> p2;
  bool called = false;
  future<int> r = f.then([&p2, &called](future<int> f) {
    called = true;
    EXPECT_EQ(7, f.get());
    return p2.get_future();
  });
  EXPECT_TRUE(r.valid());
  EXPECT_FALSE(r.is_ready());
  EXPECT_FALSE(f.valid());

  p.set_value(7);
  EXPECT_TRUE(called);
  EXPECT_FALSE(r.is_ready());

  p2.set_value(42);
  EXPECT_TRUE(r.is_ready());
  EXPECT_EQ(42, r.get());
}

#if GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS
/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_9_d) {
  // future<int>::then() implicitly unwrapping captures exceptions.
  promise<int> p;
  future<int> f = p.get_future();

  promise<int> p2;
  bool called = false;
  future<int> r = f.then([&p2, &called](future<int> f) {
    called = true;
    f.get();
    return p2.get_future();
  });
  EXPECT_TRUE(r.valid());
  EXPECT_FALSE(r.is_ready());
  EXPECT_FALSE(f.valid());

  p.set_exception(std::make_exception_ptr(std::runtime_error("test message")));
  EXPECT_TRUE(called);
  EXPECT_TRUE(r.is_ready());
  EXPECT_THROW(
      try { r.get(); } catch (std::runtime_error const& ex) {
        EXPECT_THAT(ex.what(), HasSubstr("test message"));
        throw;
      },
      std::runtime_error);
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_9_e) {
  // future<int>::then() implicitly unwrapping raises on invalid future
  // returned by continuation.
  promise<int> p;
  future<int> f = p.get_future();

  bool called = false;
  future<int> r = f.then([&called](future<int> f) {
    called = true;
    f.get();
    return future<int>{};
  });
  EXPECT_TRUE(r.valid());
  EXPECT_FALSE(r.is_ready());
  EXPECT_FALSE(f.valid());

  p.set_value(7);
  EXPECT_TRUE(called);
  EXPECT_TRUE(r.is_ready());

  EXPECT_THROW(
      try { r.get(); } catch (std::future_error const& ex) {
        EXPECT_EQ(std::future_errc::broken_promise, ex.code());
        throw;
      },
      std::future_error);
}
#endif  // GOOGLE_CLOUD_CPP_HAVE_EXCEPTIONS

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_10) {
  // future<int>::then() invalidates the source future.
  promise<int> p;
  future<int> f = p.get_future();
  future<int> r = f.then([](future<int> f) { return 2 * f.get(); });
  EXPECT_TRUE(r.valid());
  EXPECT_FALSE(r.is_ready());
  EXPECT_FALSE(f.valid());

  p.set_value(42);
  EXPECT_TRUE(r.is_ready());
  EXPECT_EQ(2 * 42, r.get());
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_11_a) {
  // future<int>::is_ready() returns false for futures that are not ready.
  promise<int> p;
  future<int> const f = p.get_future();
  EXPECT_FALSE(f.is_ready());
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_11_b) {
  // future<int>::is_ready() returns true for futures that are ready.
  promise<int> p;
  future<int> const f = p.get_future();
  p.set_value(42);
  EXPECT_TRUE(f.is_ready());
}

/// @test Verify conformance with section 2.3 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestInt, conform_2_3_11_c) {
  // future<int>::is_ready() raises for futures that are not valid.
  future<int> const f;
  ExpectFutureError([&] { f.is_ready(); }, std::future_errc::no_state);
}

/// @test Verify conformance with section 2.10 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestString, conform_2_10_4_2_a) {
  // When T is a simple value type we get back T.
  future<std::string> f = make_ready_future(std::string("42"));
  EXPECT_TRUE(f.valid());
  ASSERT_EQ(std::future_status::ready, f.wait_for(0_ms));
  EXPECT_EQ("42", f.get());
}

/// @test Verify conformance with section 2.10 of the Concurrency TS.
// NOLINTNEXTLINE(google-readability-avoid-underscore-in-googletest-name)
TEST(FutureTestString, conform_2_10_4_2_b) {
  // When T is a reference we get std::decay<T>::type.
  std::string value("42");
  std::string& sref = value;
  future<std::string> f = make_ready_future(sref);
  EXPECT_TRUE(f.valid());
  ASSERT_EQ(std::future_status::ready, f.wait_for(0_ms));
  EXPECT_EQ("42", f.get());
}

class MockFunctor {
 public:
  MockFunctor() = default;
  MockFunctor(MockFunctor&& other) noexcept : moved_from_(other.moved_from_) {
    other.moved_from_ = true;
  }
  MockFunctor(MockFunctor const& other) = default;

  void operator()(future<int>) {}

  bool moved_from_{false};
};

TEST(FutureTestInt, RValueThenFunctorIsMoved) {
  promise<int> promise;
  future<int> fut = promise.get_future();
  MockFunctor fun;
  fut.then(std::move(fun));
  promise.set_value(1);
  EXPECT_TRUE(fun.moved_from_);  // NOLINT(bugprone-use-after-move)
}

TEST(FutureTestInt, LValueThenFunctorIsCopied) {
  promise<int> promise;
  future<int> fut = promise.get_future();
  MockFunctor fun;
  fut.then(fun);
  promise.set_value(1);
  EXPECT_FALSE(fun.moved_from_);
}

class MockUnwrapFunctor {
 public:
  MockUnwrapFunctor() = default;
  MockUnwrapFunctor(MockUnwrapFunctor&& other) noexcept
      : moved_from_(other.moved_from_) {
    other.moved_from_ = true;
  }
  MockUnwrapFunctor(MockUnwrapFunctor const& other) = default;

  future<void> operator()(future<int>) { return make_ready_future(); }

  bool moved_from_{false};
};

TEST(FutureTestInt, RValueThenUnwrapFunctorIsMoved) {
  promise<int> promise;
  future<int> fut = promise.get_future();
  MockUnwrapFunctor fun;
  fut.then(std::move(fun));
  promise.set_value(1);
  EXPECT_TRUE(fun.moved_from_);  // NOLINT(bugprone-use-after-move)
}

TEST(FutureTestInt, LValueThenUnwrapFunctorIsCopied) {
  promise<int> promise;
  future<int> fut = promise.get_future();
  MockUnwrapFunctor fun;
  fut.then(fun);
  promise.set_value(1);
  EXPECT_FALSE(fun.moved_from_);
}

}  // namespace
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_END
}  // namespace cloud
}  // namespace google
