// Copyright 2017 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_BIGTABLE_ROW_H
#define GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_BIGTABLE_ROW_H

#include "google/cloud/bigtable/cell.h"
#include "google/cloud/bigtable/row_key.h"
#include "google/cloud/bigtable/version.h"
#include <vector>

namespace google {
namespace cloud {
namespace bigtable {
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_BEGIN
/**
 * The in-memory representation of a Bigtable row.
 *
 * Notice that a row returned by the Bigtable Client may have been filtered by
 * any filtering expressions provided by the application, and may not contain
 * all the data available.
 */
class Row {
 public:
  /// Create a row from a list of cells.
  template <typename T>
  Row(T&& row_key, std::vector<Cell> cells)
      : row_key_(std::forward<T>(row_key)), cells_(std::move(cells)) {}

  /// Return the row key. The returned value is not valid
  /// after this object is deleted.
  RowKeyType const& row_key() const { return row_key_; }

  /// Return all cells.
  std::vector<Cell> const& cells() const& { return cells_; }
  /// Return all cells.
  std::vector<Cell>&& cells() && { return std::move(cells_); }

 private:
  RowKeyType row_key_;
  std::vector<Cell> cells_;
};

GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_END
}  // namespace bigtable
}  // namespace cloud
}  // namespace google

#endif  // GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_BIGTABLE_ROW_H
