/*
 *  JBIG image compression library for Canon LIPSLX/UFR2/NCAP Printer.
 *  Copyright CANON INC. 2018
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <dlfcn.h>
#include <string.h>
#include <stdbool.h>
#include <signal.h>
#include <errno.h>
#include "jbig.h"
#include "cnjbig.h"

#define DEF_ERR 1
#define DEF_NO_ERR 0

typedef	struct _JBIGLIBINFO {
	void *hLib;
	void (*fn_jbg_enc_init)(struct jbg_enc_state *s, unsigned long x, unsigned long y, int planes, unsigned char **p, void (*data_out)(unsigned char *start, size_t len, void *file), void *file);
	void (*fn_jbg_enc_options)(struct jbg_enc_state *s, int order, int options, unsigned long l0, int mx, int my);
	void (*fn_jbg_enc_out)(struct jbg_enc_state *s);
	void (*fn_jbg_enc_free)(struct jbg_enc_state *s);
} JBIGLIBINFO, *LPJBIGLIBINFO;

#define	JBG_ENC_INIT		"jbg_enc_init"
#define	JBG_ENC_OPTIONS		"jbg_enc_options"
#define	JBG_ENC_OUT			"jbg_enc_out"
#define	JBG_ENC_FREE		"jbg_enc_free"

bool g_sigterm_received = false;

static void sigterm_handler(int sigcode)
{
	g_sigterm_received = true;
}

static JBIGLIBINFO *zLoadJBIGLibrary()
{
	void *hLib = NULL;
	JBIGLIBINFO *pLibInfo = NULL;
	char *pLibjbigNames[] = {
		"libjbig.so",
		"libjbig.so.2.1",
		"libjbig.so.2.0",
		"libjbig.so.0",
		NULL
	};
	int idx;

	for(idx = 0; pLibjbigNames[idx] != NULL; idx++){
		hLib = dlopen( pLibjbigNames[idx], RTLD_NOW );
		if(hLib != NULL){
			break;
		}
	}

	if( hLib != NULL ) {
		if ( ( pLibInfo = calloc( sizeof( JBIGLIBINFO ), 1 ) ) != NULL ) {
			pLibInfo->hLib = hLib;
			pLibInfo->fn_jbg_enc_init       = dlsym( hLib, JBG_ENC_INIT );
			pLibInfo->fn_jbg_enc_options    = dlsym( hLib, JBG_ENC_OPTIONS );
			pLibInfo->fn_jbg_enc_out        = dlsym( hLib, JBG_ENC_OUT );
			pLibInfo->fn_jbg_enc_free       = dlsym( hLib, JBG_ENC_FREE );
			if( ( pLibInfo->fn_jbg_enc_init == NULL )
			 || ( pLibInfo->fn_jbg_enc_options == NULL )
			 || ( pLibInfo->fn_jbg_enc_out == NULL )
			 || ( pLibInfo->fn_jbg_enc_free == NULL )
			) {
				goto LoadJBIGLibrary_error;
			}
			return pLibInfo;
		}
	}

LoadJBIGLibrary_error:
	if ( hLib != NULL ) {
		dlclose( hLib );
		hLib = NULL;
	}
	if ( pLibInfo != NULL ) {
		free( pLibInfo );
		pLibInfo = NULL;
	}
	return NULL;
}

static void zFreeJBIGLibrary(
	JBIGLIBINFO *pLibInfo
)
{
	if(pLibInfo)
	{
		if(pLibInfo->hLib)
		{
			dlclose( pLibInfo->hLib );
		}

		free( pLibInfo );
	}
}

static int zWritePipe(int fd, char *data, int write_size)
{
	int size = write_size;
	ssize_t tmp_size = 0;

	while(size){
		data += tmp_size;
		tmp_size = write(fd, data, size);

		if(tmp_size == DEF_ERR){
			return DEF_ERR;
		}
		size -= tmp_size;
	}
	return DEF_NO_ERR;

}

static int zReadPipe(int fd, char *data, int read_size)
{
	int size = read_size;
	ssize_t tmp_size = 0;
	int cnt = 0;
	fd_set selmask;

	while(size){
		data += tmp_size;

		FD_ZERO( &selmask );
		FD_SET( fd, &selmask );

		if( select( fd + 1, &selmask, NULL, NULL, NULL ) <= 0 ) {
			return DEF_ERR;
		}
		else {
			tmp_size = read(fd, data, size);

			if(tmp_size == DEF_ERR){
				return DEF_ERR;
			}
			if(tmp_size == 0){
				if(cnt == 10){
					return DEF_ERR;
				}
				cnt++;
			}
			size -= tmp_size;
		}
	}

	return DEF_NO_ERR;
}

static int zJbigReadDataFromDriver( int fd, void **pInputData, CNJBIGDATAHEADER *pDataHeader )
{
	int err = DEF_NO_ERR;

	if( ( pInputData == NULL ) || ( pDataHeader == NULL ) ) {
		return DEF_ERR;
	}

	*pInputData = NULL;
	memset(pDataHeader, 0, sizeof(CNJBIGDATAHEADER));

	if( zReadPipe( fd, (char*)pDataHeader, sizeof(CNJBIGDATAHEADER) ) < 0 ) {
		err = DEF_ERR;
	}
	else {
		int nInputSize = pDataHeader->x * pDataHeader->y / 8;
		if( nInputSize > 0 )
		{
			*pInputData = malloc( nInputSize );
			if( *pInputData == NULL ) {
				err = DEF_ERR;
			}
			else {
				if( zReadPipe( fd, *pInputData, nInputSize ) < 0 ) {
					err = DEF_ERR;
				}
				if(err != DEF_NO_ERR) {
					free( *pInputData );
					*pInputData = NULL;
				}
			}
		}
		else {
			*pInputData = NULL;
		}
	}

	return err;
}

static int zJbigWriteDataToDriver( int fd, void *pOutputData, int lOutputSize )
{
	int err = DEF_NO_ERR;

	if( ( lOutputSize > 0 ) && ( pOutputData == NULL ) ) {
		return DEF_ERR;
	}

	if( zWritePipe( fd, (char*)&lOutputSize, sizeof(int) ) < 0 ) {
		err = DEF_ERR;
	}
	else {
		if( lOutputSize > 0 )
		{
			if( zWritePipe( fd, pOutputData, lOutputSize ) < 0 ) {
				err = DEF_ERR;
			}
		}

	}

	return err;
}

typedef struct jbig_encode_buf_t{
	u_char	*buf;
	long	len;
} JbigEncodeBuf;

static void
jbig_dataout_callback(unsigned char *start, size_t len, void *file)
{
	JbigEncodeBuf *encodeBuf = (JbigEncodeBuf *)file;
	if ((encodeBuf == NULL) || (encodeBuf->buf == NULL)) {
		return;
	}
	if(len > 0) {
		memcpy(encodeBuf->buf + encodeBuf->len, start, len);
		encodeBuf->len += len;
	}
	return;
}

int main(int argc, const char * argv[]) {

	int err = DEF_NO_ERR;
	JBIGLIBINFO *pLibInfo = NULL;
	int in_fd = STDIN_FILENO;
	int out_fd = STDOUT_FILENO;
	void *pInputData = NULL;
	CNJBIGDATAHEADER dataHeader = {0};
	struct jbg_enc_state jbg_state;
	u_char *pCompBuf = NULL;
	JbigEncodeBuf encodeBuf;
	struct sigaction sigact;

	memset(&sigact, 0, sizeof(sigact));
	sigact.sa_handler = sigterm_handler;
	sigaction(SIGTERM, &sigact, NULL);

	pLibInfo = zLoadJBIGLibrary();
	if(pLibInfo)
	{
		while(g_sigterm_received == false)
		{
			err = zJbigReadDataFromDriver(in_fd, &pInputData, &dataHeader);
			if(err != DEF_NO_ERR) {
				break;
			}
			else {
				if((dataHeader.x == 0) || (dataHeader.y == 0)) {
					break;
				}

				if(pCompBuf == NULL) {
					pCompBuf = malloc( dataHeader.x * dataHeader.y / 8 * 2 );
					if(pCompBuf == NULL) {
						err = DEF_ERR;
						break;
					}
				}

				encodeBuf.len = 0;
				encodeBuf.buf = pCompBuf;

				pLibInfo->fn_jbg_enc_init(&jbg_state, dataHeader.x, dataHeader.y, 1, (unsigned char **)&pInputData, jbig_dataout_callback, &encodeBuf);
				pLibInfo->fn_jbg_enc_options(&jbg_state, 0, JBG_LRLTWO | JBG_TPBON, dataHeader.l0, 0, 0);
				pLibInfo->fn_jbg_enc_out(&jbg_state);
				pLibInfo->fn_jbg_enc_free(&jbg_state);

				err = zJbigWriteDataToDriver(out_fd, pCompBuf, (int)encodeBuf.len);
				if(err != DEF_NO_ERR) {
					break;
				}

				free(pInputData);
				pInputData = NULL;
			}
		}

		zFreeJBIGLibrary(pLibInfo);
		pLibInfo = NULL;
	}

	if(pCompBuf) {
		free(pCompBuf);
		pCompBuf = NULL;
	}

    return 0;
}
