import { PartialJSONObject, ReadonlyJSONObject } from '@lumino/coreutils';
import { Poll } from '@lumino/polling';
import { ISignal } from '@lumino/signaling';
import { ServerConnection } from '../serverconnection';
import { BaseManager, IManager as IBaseManager } from '../basemanager';
/**
 * The user API service manager.
 */
export declare class UserManager extends BaseManager implements User.IManager {
    private _isReady;
    private _ready;
    private _pollSpecs;
    private _identity;
    private _permissions;
    private _userChanged;
    private _connectionFailure;
    /**
     * Create a new user manager.
     */
    constructor(options?: UserManager.IOptions);
    /**
     * The server settings for the manager.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the manager is ready.
     */
    get isReady(): boolean;
    /**
     * A promise that fulfills when the manager is ready.
     */
    get ready(): Promise<void>;
    /**
     * Get the most recently fetched identity.
     */
    get identity(): User.IIdentity | null;
    /**
     * Get the most recently fetched permissions.
     */
    get permissions(): ReadonlyJSONObject | null;
    /**
     * A signal emitted when the user changes.
     */
    get userChanged(): ISignal<this, User.IUser>;
    /**
     * A signal emitted when there is a connection failure.
     */
    get connectionFailure(): ISignal<this, Error>;
    /**
     * Dispose of the resources used by the manager.
     */
    dispose(): void;
    /**
     * Force a refresh of the specs from the server.
     *
     * @returns A promise that resolves when the specs are fetched.
     *
     * #### Notes
     * This is intended to be called only in response to a user action,
     * since the manager maintains its internal state.
     */
    refreshUser(): Promise<void>;
    /**
     * Execute a request to the server to poll the user and update state.
     */
    protected requestUser(): Promise<void>;
}
/**
 * A namespace for `UserManager` statics.
 */
export declare namespace UserManager {
    /**
     * The instantiation options for a user manager.
     */
    interface IOptions extends BaseManager.IOptions {
        /**
         * When the manager stops polling the API. Defaults to `when-hidden`.
         */
        standby?: Poll.Standby | (() => boolean | Poll.Standby);
    }
}
/**
 * A namespace for user API interfaces.
 */
export declare namespace User {
    /**
     * The interface describing a user identity.
     */
    interface IUser {
        readonly identity: IIdentity;
        readonly permissions: PartialJSONObject;
    }
    /**
     * The interface describing a user identity.
     */
    interface IIdentity extends PartialJSONObject {
        /**
         * User's unique identifier.
         */
        readonly username: string;
        /**
         * User's full name.
         */
        readonly name: string;
        /**
         * Shorter version of the name for displaying it on the UI.
         */
        readonly display_name: string;
        /**
         * User's name initials.
         */
        readonly initials: string;
        /**
         * User's cursor color and icon color if avatar_url is undefined
         * (there is no image).
         */
        readonly color: string;
        /**
         * User's avatar url.
         * The url to the user's image for the icon.
         */
        readonly avatar_url?: string;
    }
    /**
     * Object which manages user's identity.
     *
     * #### Notes
     * The manager is responsible for maintaining the state of the user.
     */
    interface IManager extends IBaseManager {
        /**
         * A signal emitted when the user changes.
         */
        userChanged: ISignal<this, User.IUser>;
        /**
         * User's identity.
         *
         * #### Notes
         * The value will be null until the manager is ready.
         */
        readonly identity: User.IIdentity | null;
        /**
         * User's permissions.
         *
         * #### Notes
         * The value will be null until the manager is ready.
         */
        readonly permissions: ReadonlyJSONObject | null;
        /**
         * Force a refresh of user's identity from the server.
         *
         * @returns A promise that resolves when the identity is fetched.
         *
         * #### Notes
         * This is intended to be called only in response to a user action,
         * since the manager maintains its internal state.
         */
        refreshUser(): Promise<void>;
    }
}
