"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.getKernelModel = exports.shutdownKernel = exports.interruptKernel = exports.restartKernel = exports.startNew = exports.listRunning = exports.KERNEL_SERVICE_URL = void 0;
const serverconnection_1 = require("../serverconnection");
const coreutils_1 = require("@jupyterlab/coreutils");
const validate_1 = require("./validate");
/**
 * The url for the kernel service.
 */
exports.KERNEL_SERVICE_URL = 'api/kernels';
/**
 * Fetch the running kernels.
 *
 * @param settings - The optional server settings.
 *
 * @returns A promise that resolves with the list of running kernels.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
async function listRunning(settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.KERNEL_SERVICE_URL);
    const response = await serverconnection_1.ServerConnection.makeRequest(url, {}, settings);
    if (response.status !== 200) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    validate_1.validateModels(data);
    return data;
}
exports.listRunning = listRunning;
/**
 * Start a new kernel.
 *
 * @param options - The options used to create the kernel.
 *
 * @returns A promise that resolves with a kernel connection object.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
async function startNew(options = {}, settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.KERNEL_SERVICE_URL);
    const init = {
        method: 'POST',
        body: JSON.stringify(options)
    };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status !== 201) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    validate_1.validateModel(data);
    return data;
}
exports.startNew = startNew;
/**
 * Restart a kernel.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response (and thus after a restart) and rejected otherwise.
 */
async function restartKernel(id, settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.KERNEL_SERVICE_URL, encodeURIComponent(id), 'restart');
    const init = { method: 'POST' };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status !== 200) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    validate_1.validateModel(data);
}
exports.restartKernel = restartKernel;
/**
 * Interrupt a kernel.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
async function interruptKernel(id, settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.KERNEL_SERVICE_URL, encodeURIComponent(id), 'interrupt');
    const init = { method: 'POST' };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status !== 204) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
}
exports.interruptKernel = interruptKernel;
/**
 * Shut down a kernel.
 *
 * @param id - The id of the running kernel.
 *
 * @param settings - The server settings for the request.
 *
 * @returns A promise that resolves when the kernel is shut down.
 *
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
async function shutdownKernel(id, settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.KERNEL_SERVICE_URL, encodeURIComponent(id));
    const init = { method: 'DELETE' };
    const response = await serverconnection_1.ServerConnection.makeRequest(url, init, settings);
    if (response.status === 404) {
        const msg = `The kernel "${id}" does not exist on the server`;
        console.warn(msg);
    }
    else if (response.status !== 204) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
}
exports.shutdownKernel = shutdownKernel;
/**
 * Get a full kernel model from the server by kernel id string.
 *
 * #### Notes
 * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/kernels) and validates the response model.
 *
 * The promise is fulfilled on a valid response and rejected otherwise.
 */
async function getKernelModel(id, settings = serverconnection_1.ServerConnection.makeSettings()) {
    const url = coreutils_1.URLExt.join(settings.baseUrl, exports.KERNEL_SERVICE_URL, encodeURIComponent(id));
    const response = await serverconnection_1.ServerConnection.makeRequest(url, {}, settings);
    if (response.status === 404) {
        return undefined;
    }
    else if (response.status !== 200) {
        const err = await serverconnection_1.ServerConnection.ResponseError.create(response);
        throw err;
    }
    const data = await response.json();
    validate_1.validateModel(data);
    return data;
}
exports.getKernelModel = getKernelModel;
//# sourceMappingURL=restapi.js.map