//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// INCL++ intra-nuclear cascade model
// Alain Boudard, CEA-Saclay, France
// Joseph Cugnon, University of Liege, Belgium
// Jean-Christophe David, CEA-Saclay, France
// Pekka Kaitaniemi, CEA-Saclay, France, and Helsinki Institute of Physics, Finland
// Sylvie Leray, CEA-Saclay, France
// Davide Mancusi, CEA-Saclay, France
//
#define INCLXX_IN_GEANT4_MODE 1

#include "globals.hh"

#include "G4INCLNKbElasticChannel.hh"
#include "G4INCLKinematicsUtils.hh"
#include "G4INCLBinaryCollisionAvatar.hh"
#include "G4INCLRandom.hh"
#include "G4INCLGlobals.hh"
#include "G4INCLLogger.hh"
#include <algorithm>
#include "G4INCLPhaseSpaceGenerator.hh"

namespace G4INCL {
	
	NKbElasticChannel::NKbElasticChannel(Particle *p1, Particle *p2)
		: particle1(p1), particle2(p2)
		{}
	
	NKbElasticChannel::~NKbElasticChannel(){}
	
	void NKbElasticChannel::fillFinalState(FinalState *fs){
		
		Particle *nucleon;
		Particle *kaon;
		
		if(particle1->isNucleon()){
			nucleon = particle1;
			kaon = particle2;
		}
		else{
			nucleon = particle2;
			kaon = particle1;
		}
		
		ThreeVector mom_kaon = KaonMomentum(kaon,nucleon); // Caution: angular distribution comes from K-p which is "Lambda resonant"

		kaon->setMomentum(mom_kaon);
		nucleon->setMomentum(-mom_kaon);
		
		nucleon->adjustEnergyFromMomentum();
		kaon->adjustEnergyFromMomentum();
		
		fs->addModifiedParticle(nucleon);
		fs->addModifiedParticle(kaon);
				
	}
	
	ThreeVector NKbElasticChannel::KaonMomentum(Particle const * const kaon, Particle const * const nucleon){
		
		const G4double norm = KinematicsUtils::momentumInCM(kaon,nucleon);
		const G4double pLab = KinematicsUtils::momentumInLab(kaon,nucleon);
		
		if(pLab < 225.) return Random::normVector()*norm; // isotropic
		
		G4double cos_theta = 1.;
		G4double sin_theta = 0.;
		const G4double cos_phi = std::cos(Random::shoot()*Math::twoPi);
		const G4double sin_phi = std::sqrt(1-cos_phi*cos_phi);
		
		const G4double x = kaon->getMomentum().getX();
		const G4double y = kaon->getMomentum().getY();
		const G4double z = kaon->getMomentum().getZ();
		
		const G4double r = std::sqrt(x*x+y*y+z*z);
		const G4double rho = std::sqrt(x*x+y*y);
		
		if(pLab >= 2375.){
			const G4double b = 12. * pLab/2375.; // correspond to the forward slope description at 2375 MeV/c
			cos_theta = std::log(Random::shoot()*(std::exp(b)-std::exp(-b))+std::exp(-b))/b;
			sin_theta = std::sqrt(1-cos_theta*cos_theta);
			
		}
		else{
			const G4double Legendre_coef[431][11] = {
				{225,0.14007,-0.1329,0.00044,-0.00469,-0.00254,-0.00011,0,0,0,0},
				{230,0.14143,-0.1335,0.00058,-0.00417,-0.00248,-4E-05,0,0,0,0},
				{235,0.14283,-0.1179,0.00077,-0.00364,-0.00243,2E-05,0,0,0,0},
				{240,0.14428,-0.08345,0.00102,-0.00312,-0.00238,9E-05,0,0,0,0},
				{245,0.14582,-0.04111,0.00133,-0.00259,-0.00232,0.00015,0,0,0,0},
				{250,0.14746,-0.00771,0.00173,-0.00206,-0.00226,0.00022,0,0,0,0},
				{255,0.14924,0.00768,0.00222,-0.00153,-0.0022,0.00028,0,0,0,0},
				{260,0.15117,0.00618,0.00283,-0.00098,-0.00214,0.00035,0,0,0,0},
				{265,0.15327,-0.00651,0.00358,-0.00043,-0.00206,0.00041,0,0,0,0},
				{270,0.15552,-0.02353,0.00449,0.00014,-0.00198,0.00048,0,0,0,0},
				{275,0.15793,-0.03863,0.00558,0.00072,-0.0019,0.00054,0,0,0,0},
				{280,0.16051,-0.04941,0.00685,0.00131,-0.0018,0.0006,0,0,0,0},
				{285,0.16325,-0.05717,0.00834,0.00192,-0.00168,0.00067,0,0,0,0},
				{290,0.16617,-0.06091,0.01006,0.00255,-0.00155,0.00073,0,0,0,0},
				{295,0.16926,-0.05462,0.01201,0.0032,-0.00141,0.00079,0,0,0,0},
				{300,0.17254,-0.03426,0.0142,0.00387,-0.00124,0.00085,0,0,0,0},
				{305,0.17599,-0.00416,0.01663,0.00457,-0.00105,0.00091,0,0,0,0},
				{310,0.17961,0.02807,0.01931,0.00529,-0.00084,0.00097,0,0,0,0},
				{315,0.18341,0.05924,0.02221,0.00603,-0.0006,0.00102,0,0,0,0},
				{320,0.18737,0.09129,0.02534,0.00681,-0.00033,0.00107,0,0,0,0},
				{325,0.19147,0.12688,0.02868,0.0076,-2E-05,0.00112,0,0,0,0},
				{330,0.19568,0.16704,0.03221,0.00841,0.00032,0.00116,0,0,0,0},
				{335,0.20001,0.21052,0.03591,0.00927,0.0007,0.0012,0,0,0,0},
				{340,0.20444,0.25417,0.03977,0.01019,0.00113,0.00123,0,0,0,0},
				{345,0.20897,0.29637,0.04374,0.01121,0.0016,0.00125,0,0,0,0},
				{350,0.21361,0.34404,0.04781,0.01235,0.00211,0.00126,0,0,0,0},
				{355,0.21836,0.41178,0.05192,0.01363,0.00268,0.00126,0,0,0,0},
				{360,0.22323,0.51001,0.05605,0.01505,0.00329,0.00125,0,0,0,0},
				{365,0.22824,0.63548,0.06014,0.01663,0.00396,0.00123,0,0,0,0},
				{370,0.2334,0.7775,0.06415,0.01839,0.00469,0.0012,0,0,0,0},
				{375,0.23875,0.92674,0.06804,0.02031,0.00548,0.00114,0,0,0,0},
				{380,0.24435,1.07258,0.07178,0.02238,0.00632,0.00107,0,0,0,0},
				{385,0.25024,1.19622,0.07534,0.0246,0.00723,0.00099,1E-05,0,0,0},
				{390,0.25648,1.27148,0.07871,0.02695,0.00819,0.00088,1E-05,0,0,0},
				{395,0.26314,1.27773,0.08191,0.02941,0.00922,0.00074,1E-05,0,0,0},
				{400,0.2703,1.21317,0.08493,0.03197,0.0103,0.00058,1E-05,0,0,0},
				{405,0.27805,1.09842,0.08781,0.03461,0.01144,0.0004,1E-05,0,0,0},
				{410,0.28645,0.96106,0.09058,0.03732,0.01264,0.00018,1E-05,0,0,0},
				{415,0.29558,0.82322,0.09326,0.04009,0.01389,-7E-05,1E-05,0,0,0},
				{420,0.30549,0.70361,0.09589,0.04293,0.01518,-0.00035,2E-05,0,0,0},
				{425,0.31624,0.61949,0.09849,0.04579,0.01652,-0.00067,2E-05,0,0,0},
				{430,0.32784,0.57051,0.10106,0.04865,0.0179,-0.00102,2E-05,0,0,0},
				{435,0.34034,0.53542,0.10361,0.05148,0.0193,-0.00142,2E-05,0,0,0},
				{440,0.35375,0.49078,0.10615,0.05423,0.02072,-0.00186,2E-05,0,0,0},
				{445,0.36809,0.42743,0.10865,0.05688,0.02216,-0.00234,2E-05,0,0,0},
				{450,0.38335,0.35685,0.11111,0.05938,0.02361,-0.00287,3E-05,0,0,0},
				{455,0.39954,0.29473,0.11352,0.0617,0.02506,-0.00345,3E-05,0,0,0},
				{460,0.41664,0.24563,0.11587,0.0638,0.02653,-0.00407,3E-05,0,0,0},
				{465,0.43459,0.21189,0.11819,0.06569,0.02798,-0.00474,3E-05,0,0,0},
				{470,0.45332,0.19982,0.12049,0.06737,0.0294,-0.00546,3E-05,0,0,0},
				{475,0.47275,0.20898,0.12283,0.06886,0.03078,-0.00623,4E-05,0,0,0},
				{480,0.49282,0.22837,0.12528,0.07016,0.03211,-0.00705,4E-05,0,0,0},
				{485,0.51344,0.24427,0.12795,0.0713,0.03337,-0.00791,4E-05,0,0,0},
				{490,0.53457,0.25093,0.13095,0.07231,0.03456,-0.00882,4E-05,0,0,0},
				{495,0.55613,0.25424,0.13442,0.07322,0.03567,-0.00977,4E-05,0,0,0},
				{500,0.57806,0.26108,0.13848,0.07406,0.03671,-0.01076,4E-05,0,0,0},
				{505,0.60029,0.27305,0.14325,0.07486,0.03765,-0.01179,5E-05,0,0,0},
				{510,0.62275,0.2865,0.14878,0.07566,0.03851,-0.01285,5E-05,0,0,0},
				{515,0.64538,0.30095,0.15509,0.07648,0.03926,-0.01394,5E-05,0,0,0},
				{520,0.66812,0.32366,0.16211,0.07734,0.03992,-0.01506,5E-05,0,0,0},
				{525,0.69093,0.35704,0.16973,0.07824,0.04047,-0.0162,5E-05,0,0,0},
				{530,0.71377,0.36827,0.17776,0.07915,0.04093,-0.01737,4E-05,0,0,0},
				{535,0.7366,0.38051,0.18596,0.08008,0.0413,-0.01855,4E-05,0,0,0},
				{540,0.75939,0.39356,0.19409,0.08101,0.04158,-0.01975,4E-05,-1E-05,0,0},
				{545,0.78211,0.40692,0.20194,0.08198,0.04178,-0.02094,4E-05,-1E-05,0,0},
				{550,0.80474,0.42044,0.20931,0.083,0.04194,-0.02212,3E-05,-1E-05,0,0},
				{555,0.82725,0.43387,0.21611,0.08409,0.04205,-0.02328,3E-05,-1E-05,0,0},
				{560,0.84963,0.447,0.2222,0.08529,0.04216,-0.02441,2E-05,-1E-05,0,0},
				{565,0.87184,0.45974,0.22759,0.08661,0.04226,-0.0255,2E-05,-1E-05,0,0},
				{570,0.89386,0.47209,0.23233,0.08808,0.04239,-0.02654,1E-05,-1E-05,0,0},
				{575,0.91567,0.4841,0.23658,0.08972,0.04255,-0.02752,0,-1E-05,0,0},
				{580,0.93726,0.4957,0.24036,0.09155,0.04277,-0.02843,-1E-05,-1E-05,0,0},
				{585,0.95862,0.50708,0.24394,0.09359,0.04306,-0.02924,-2E-05,-1E-05,0,0},
				{590,0.97974,0.51822,0.24738,0.09584,0.04344,-0.02996,-3E-05,-1E-05,0,0},
				{595,1.00062,0.52933,0.25094,0.09833,0.04391,-0.03055,-4E-05,-1E-05,0,0},
				{600,1.02125,0.54045,0.2547,0.10105,0.0445,-0.03102,-6E-05,-1E-05,0,0},
				{605,1.04163,0.55179,0.25885,0.10403,0.04522,-0.03134,-7E-05,-1E-05,0,0},
				{610,1.06175,0.56353,0.2635,0.10727,0.04608,-0.03151,-9E-05,-1E-05,0,0},
				{615,1.08161,0.57592,0.26878,0.1108,0.0471,-0.0315,-0.00011,-1E-05,0,0},
				{620,1.10122,0.58905,0.27473,0.1146,0.0483,-0.03132,-0.00013,-1E-05,0,0},
				{625,1.12057,0.6033,0.28148,0.11868,0.0497,-0.03096,-0.00015,-1E-05,0,0},
				{630,1.13968,0.61885,0.28905,0.12305,0.05131,-0.03043,-0.00017,0,0,0},
				{635,1.15854,0.63595,0.2975,0.1277,0.05316,-0.02972,-0.0002,0,0,0},
				{640,1.17716,0.65493,0.30687,0.13264,0.05526,-0.02883,-0.00022,0,0,0},
				{645,1.19553,0.67602,0.31718,0.13783,0.05761,-0.02778,-0.00025,0,0,0},
				{650,1.21364,0.69939,0.32841,0.14324,0.06018,-0.02657,-0.00028,0,0,0},
				{655,1.23147,0.72528,0.34057,0.14882,0.06296,-0.02522,-0.00031,0,0,0},
				{660,1.249,0.7539,0.35367,0.15454,0.06592,-0.02375,-0.00034,0,0,0},
				{665,1.26621,0.78533,0.36764,0.16034,0.06903,-0.02217,-0.00038,0,0,0},
				{670,1.28308,0.8195,0.3824,0.1662,0.07229,-0.0205,-0.00041,0,0,0},
				{675,1.29958,0.85642,0.39788,0.17209,0.07568,-0.01875,-0.00045,0,1E-05,0},
				{680,1.31567,0.89591,0.41398,0.17799,0.07917,-0.01692,-0.00049,1E-05,1E-05,0},
				{685,1.3313,0.93777,0.43055,0.1839,0.08278,-0.01503,-0.00052,1E-05,1E-05,0},
				{690,1.3464,0.98156,0.44745,0.1898,0.08648,-0.01309,-0.00056,1E-05,1E-05,0},
				{695,1.36092,1.02685,0.46446,0.19569,0.09029,-0.01109,-0.0006,1E-05,1E-05,0},
				{700,1.3748,1.07309,0.48139,0.20155,0.09418,-0.00906,-0.00064,2E-05,1E-05,0},
				{705,1.38798,1.11974,0.49805,0.20738,0.09815,-0.00698,-0.00068,2E-05,1E-05,0},
				{710,1.4004,1.16608,0.51421,0.21317,0.10219,-0.00489,-0.00072,2E-05,1E-05,0},
				{715,1.41199,1.2115,0.52966,0.21892,0.10629,-0.00279,-0.00076,2E-05,1E-05,0},
				{720,1.4227,1.25536,0.54424,0.22461,0.11042,-0.00069,-0.0008,3E-05,2E-05,0},
				{725,1.43247,1.29714,0.55781,0.23025,0.11459,0.00139,-0.00083,3E-05,2E-05,0},
				{730,1.44128,1.33634,0.57029,0.23586,0.11881,0.00347,-0.00087,4E-05,2E-05,0},
				{735,1.4491,1.37258,0.58161,0.24145,0.12307,0.00557,-0.0009,4E-05,2E-05,0},
				{740,1.4559,1.40563,0.59178,0.24704,0.1274,0.0077,-0.00092,4E-05,2E-05,0},
				{745,1.46166,1.43537,0.60085,0.25266,0.1318,0.00989,-0.00094,5E-05,2E-05,0},
				{750,1.4664,1.46184,0.60893,0.25834,0.1363,0.01213,-0.00096,5E-05,3E-05,0},
				{755,1.47012,1.48507,0.61613,0.2641,0.14092,0.01446,-0.00097,6E-05,3E-05,0},
				{760,1.47283,1.50532,0.62261,0.27,0.14569,0.01687,-0.00098,6E-05,3E-05,0},
				{765,1.47454,1.52274,0.62849,0.27606,0.15063,0.01938,-0.00097,7E-05,3E-05,0},
				{770,1.47527,1.5377,0.634,0.2823,0.15572,0.02196,-0.00096,7E-05,3E-05,0},
				{775,1.47506,1.55045,0.63926,0.28875,0.16095,0.02458,-0.00094,8E-05,3E-05,0},
				{780,1.47395,1.56132,0.64445,0.29542,0.16628,0.02721,-0.0009,8E-05,3E-05,0},
				{785,1.47197,1.57062,0.64973,0.30235,0.17174,0.02983,-0.00086,9E-05,4E-05,0},
				{790,1.46919,1.5787,0.65524,0.30957,0.1773,0.03242,-0.0008,9E-05,4E-05,0},
				{795,1.46566,1.58579,0.66108,0.31709,0.18297,0.03496,-0.00072,0.0001,4E-05,0},
				{800,1.46143,1.59213,0.66731,0.32494,0.18871,0.03743,-0.00063,0.0001,4E-05,0},
				{805,1.45658,1.59801,0.67402,0.33312,0.19452,0.03981,-0.00052,0.00011,4E-05,0},
				{810,1.45114,1.60371,0.68124,0.34165,0.20038,0.04209,-0.00039,0.00011,4E-05,0},
				{815,1.44519,1.60936,0.68893,0.35054,0.20629,0.04424,-0.00023,0.00011,4E-05,0},
				{820,1.43878,1.61515,0.69704,0.35982,0.21224,0.04625,-6E-05,0.00012,4E-05,0},
				{825,1.43195,1.62131,0.70553,0.36949,0.21824,0.04813,0.00015,0.00012,4E-05,0},
				{830,1.42477,1.62791,0.71427,0.37956,0.22429,0.04984,0.00038,0.00012,3E-05,0},
				{835,1.41729,1.63504,0.72318,0.39006,0.23038,0.05138,0.00064,0.00012,3E-05,0},
				{840,1.40956,1.64282,0.73218,0.401,0.23651,0.05274,0.00093,0.00012,3E-05,0},
				{845,1.40161,1.65132,0.74124,0.41239,0.2427,0.05393,0.00126,0.00012,2E-05,0},
				{850,1.3935,1.66056,0.7503,0.42424,0.24895,0.05496,0.00162,0.00011,2E-05,0},
				{855,1.38526,1.67054,0.75937,0.43656,0.25527,0.05582,0.00201,0.00011,1E-05,0},
				{860,1.37693,1.68128,0.76851,0.44938,0.26166,0.05654,0.00245,0.0001,1E-05,0},
				{865,1.36855,1.69275,0.77776,0.46269,0.26815,0.05712,0.00292,9E-05,0,0},
				{870,1.36017,1.70495,0.78724,0.47652,0.27478,0.05759,0.00343,8E-05,-1E-05,0},
				{875,1.35182,1.71783,0.79705,0.49087,0.28156,0.05797,0.00399,7E-05,-2E-05,0},
				{880,1.34355,1.73135,0.80728,0.50576,0.28853,0.05828,0.00458,5E-05,-3E-05,0},
				{885,1.33538,1.74545,0.81803,0.52118,0.29569,0.05854,0.00522,4E-05,-4E-05,0},
				{890,1.32737,1.76009,0.82938,0.53712,0.30306,0.05876,0.00589,1E-05,-5E-05,0},
				{895,1.31955,1.77516,0.8414,0.55357,0.31064,0.05896,0.00661,-1E-05,-7E-05,0},
				{900,1.31195,1.79066,0.85422,0.57052,0.31846,0.05916,0.00737,-4E-05,-8E-05,0},
				{905,1.30462,1.80645,0.86774,0.58794,0.32651,0.05937,0.00817,-7E-05,-0.0001,0},
				{910,1.29758,1.82248,0.88204,0.60577,0.33479,0.05957,0.009,-0.0001,-0.00012,0},
				{915,1.29087,1.83865,0.89711,0.62396,0.34327,0.05975,0.00987,-0.00014,-0.00014,0},
				{920,1.28451,1.85489,0.91289,0.64244,0.35194,0.05992,0.01077,-0.00019,-0.00016,0},
				{925,1.27854,1.87113,0.92941,0.66113,0.36077,0.06006,0.01169,-0.00024,-0.00018,0},
				{930,1.27299,1.88729,0.94663,0.67995,0.36971,0.06017,0.01263,-0.00029,-0.00021,0},
				{935,1.26789,1.90333,0.9646,0.69881,0.37872,0.06024,0.01358,-0.00034,-0.00024,0},
				{940,1.26327,1.91917,0.98329,0.71765,0.38775,0.06027,0.01453,-0.00041,-0.00026,0},
				{945,1.25914,1.93478,1.00276,0.73636,0.39678,0.06025,0.01548,-0.00047,-0.00029,0},
				{950,1.25553,1.95014,1.02309,0.75486,0.40576,0.06017,0.01642,-0.00054,-0.00032,0},
				{955,1.25247,1.96526,1.04444,0.77305,0.41466,0.06003,0.01733,-0.00062,-0.00036,0},
				{960,1.24996,1.98014,1.06695,0.79084,0.42344,0.05981,0.0182,-0.0007,-0.00039,0},
				{965,1.24804,1.99479,1.09078,0.80815,0.43208,0.0595,0.01902,-0.00079,-0.00042,0},
				{970,1.24672,2.00926,1.11616,0.82491,0.44055,0.0591,0.01979,-0.00088,-0.00046,0},
				{975,1.24601,2.02357,1.14324,0.84102,0.44882,0.05858,0.02051,-0.00098,-0.00049,0},
				{980,1.24594,2.03775,1.17221,0.8564,0.45685,0.05794,0.02117,-0.00108,-0.00053,0},
				{985,1.2465,2.0518,1.20312,0.87098,0.46461,0.05716,0.02178,-0.00118,-0.00057,0},
				{990,1.24771,2.06568,1.23594,0.88468,0.47206,0.05623,0.02233,-0.00129,-0.0006,0},
				{995,1.24956,2.0793,1.27044,0.89743,0.47913,0.05514,0.02282,-0.0014,-0.00064,0},
				{1000,1.25203,2.09256,1.30631,0.90917,0.48577,0.05388,0.02322,-0.00151,-0.00067,0},
				{1005,1.25511,2.10532,1.34305,0.91985,0.49192,0.05246,0.02355,-0.00162,-0.0007,0},
				{1010,1.2588,2.11731,1.37985,0.92939,0.49755,0.05085,0.02379,-0.00174,-0.00073,0},
				{1015,1.26308,2.12833,1.41594,0.93779,0.5026,0.04907,0.02393,-0.00185,-0.00076,0},
				{1020,1.26795,2.13814,1.45041,0.94503,0.50705,0.0471,0.02399,-0.00197,-0.00078,0},
				{1025,1.27338,2.14656,1.48242,0.95109,0.51088,0.04496,0.02395,-0.00208,-0.0008,0},
				{1030,1.27938,2.15331,1.51104,0.95597,0.51404,0.04264,0.02382,-0.00219,-0.00082,0},
				{1035,1.28592,2.15832,1.53575,0.95964,0.51651,0.04015,0.0236,-0.00229,-0.00083,0},
				{1040,1.293,2.16155,1.55614,0.96211,0.51826,0.03747,0.02327,-0.00239,-0.00083,0},
				{1045,1.30061,2.163,1.57204,0.96334,0.51926,0.03463,0.02286,-0.00249,-0.00083,0},
				{1050,1.30872,2.16272,1.58343,0.96336,0.51949,0.03161,0.02235,-0.00257,-0.00082,0},
				{1055,1.31732,2.16081,1.5905,0.96221,0.51898,0.02843,0.02175,-0.00264,-0.00081,0},
				{1060,1.32638,2.15741,1.59355,0.95996,0.51774,0.0251,0.02108,-0.00271,-0.00078,0},
				{1065,1.33587,2.15274,1.59306,0.95665,0.51581,0.02165,0.02035,-0.00275,-0.00074,0},
				{1070,1.34578,2.14696,1.58939,0.95236,0.51321,0.01808,0.01956,-0.00278,-0.00069,0},
				{1075,1.35609,2.14027,1.58309,0.94716,0.50999,0.01443,0.01874,-0.0028,-0.00063,0},
				{1080,1.36676,2.13286,1.57454,0.94113,0.50619,0.01071,0.01787,-0.00279,-0.00056,0},
				{1085,1.37779,2.12492,1.56429,0.93433,0.50184,0.00695,0.01698,-0.00275,-0.00047,0},
				{1090,1.38914,2.11665,1.55271,0.92686,0.49697,0.00317,0.01608,-0.00269,-0.00037,0},
				{1095,1.40076,2.10821,1.54028,0.91878,0.49165,-0.00062,0.01518,-0.0026,-0.00025,0},
				{1100,1.41261,2.09974,1.52734,0.9102,0.4859,-0.00439,0.01429,-0.00247,-0.00011,0},
				{1105,1.42462,2.09138,1.51429,0.90121,0.47978,-0.00813,0.01344,-0.00231,5E-05,0},
				{1110,1.43677,2.08324,1.50142,0.89189,0.47332,-0.01183,0.01264,-0.0021,0.00023,0},
				{1115,1.44898,2.07537,1.48901,0.88234,0.46657,-0.01546,0.0119,-0.00185,0.00044,0},
				{1120,1.46122,2.06783,1.4773,0.87264,0.45958,-0.019,0.01126,-0.00154,0.00066,0},
				{1125,1.47343,2.06063,1.46643,0.86287,0.45238,-0.02245,0.01072,-0.00118,0.00091,0},
				{1130,1.48556,2.05378,1.45653,0.85308,0.44503,-0.02577,0.01029,-0.00076,0.00119,0},
				{1135,1.49758,2.04726,1.44767,0.84333,0.43758,-0.02897,0.00999,-0.00028,0.00149,0},
				{1140,1.50943,2.04106,1.43987,0.83366,0.43008,-0.03202,0.00984,0.00027,0.00183,0},
				{1145,1.52109,2.03518,1.43314,0.82413,0.4226,-0.03489,0.00984,0.00089,0.00219,0},
				{1150,1.53252,2.0296,1.42743,0.81479,0.4152,-0.03756,0.01,0.00159,0.00258,0},
				{1155,1.54367,2.02432,1.42269,0.8057,0.4079,-0.04002,0.01034,0.00238,0.003,0},
				{1160,1.5545,2.01935,1.41883,0.79694,0.40076,-0.04224,0.01087,0.00325,0.00345,0},
				{1165,1.56496,2.01471,1.41576,0.78856,0.3938,-0.04419,0.01158,0.00421,0.00393,0},
				{1170,1.57503,2.01042,1.41337,0.78064,0.38707,-0.04587,0.01249,0.00526,0.00444,0},
				{1175,1.58469,2.1965,1.41155,0.77321,0.38061,-0.04724,0.01361,0.00642,0.00498,0},
				{1180,1.5939,2.00299,1.41016,0.76633,0.37444,-0.04829,0.01496,0.00767,0.00554,0},
				{1185,1.60265,1.9999,1.40908,0.76005,0.3686,-0.049,0.01653,0.00902,0.00613,0},
				{1190,1.61092,1.99724,1.40818,0.7544,0.36312,-0.04934,0.01834,0.01048,0.00674,1E-05},
				{1195,1.6187,1.99499,1.40729,0.7494,0.35805,-0.0493,0.02039,0.01204,0.00737,1E-05},
				{1200,1.62597,1.99317,1.40635,0.74509,0.35343,-0.04887,0.02268,0.01371,0.00801,1E-05},
				{1205,1.63272,1.99173,1.40521,0.74151,0.34931,-0.04801,0.02523,0.01547,0.00866,1E-05},
				{1210,1.63895,1.99063,1.40379,0.73867,0.34575,-0.0467,0.02803,0.01733,0.00932,1E-05},
				{1215,1.64465,1.98982,1.40203,0.73662,0.34282,-0.0449,0.03109,0.01929,0.00997,1E-05},
				{1220,1.64984,1.98925,1.39991,0.73538,0.34058,-0.04259,0.03443,0.02134,0.01062,2E-05},
				{1225,1.65456,1.98885,1.39743,0.73499,0.33906,-0.03974,0.03803,0.02346,0.01124,2E-05},
				{1230,1.65885,1.98857,1.39463,0.73546,0.33833,-0.03633,0.0419,0.02566,0.01185,2E-05},
				{1235,1.66276,1.9884,1.39163,0.73683,0.33842,-0.03235,0.04604,0.02791,0.01241,3E-05},
				{1240,1.66633,1.98832,1.38853,0.7391,0.33937,-0.0278,0.05045,0.03022,0.01294,3E-05},
				{1245,1.66959,1.98831,1.38547,0.74229,0.34116,-0.02272,0.05512,0.03256,0.01341,3E-05},
				{1250,1.67261,1.98841,1.38262,0.74638,0.34381,-0.01713,0.06004,0.03492,0.01381,4E-05},
				{1255,1.67544,1.98864,1.38016,0.75139,0.34735,-0.01103,0.06523,0.03731,0.01416,4E-05},
				{1260,1.67811,1.98907,1.37827,0.75735,0.35181,-0.00444,0.07068,0.03972,0.01444,4E-05},
				{1265,1.6807,1.98975,1.37713,0.76427,0.35722,0.00267,0.07642,0.04215,0.01467,5E-05},
				{1270,1.68325,1.99077,1.37693,0.77218,0.36363,0.0103,0.08243,0.04461,0.01486,5E-05},
				{1275,1.68581,1.99224,1.37785,0.7811,0.37107,0.01845,0.08874,0.0471,0.01499,5E-05},
				{1280,1.68844,1.99422,1.38006,0.79104,0.37957,0.02713,0.09535,0.04962,0.01508,6E-05},
				{1285,1.69117,1.99685,1.38376,0.80203,0.38917,0.03633,0.10226,0.05216,0.01514,6E-05},
				{1290,1.69404,2.00021,1.38909,0.81409,0.39987,0.04604,0.10947,0.05472,0.01516,7E-05},
				{1295,1.69705,2.00443,1.39628,0.8272,0.41163,0.05625,0.11694,0.05729,0.01514,7E-05},
				{1300,1.70021,2.00965,1.4055,0.84135,0.42441,0.06693,0.12467,0.05985,0.01508,8E-05},
				{1305,1.70353,2.01597,1.41698,0.85652,0.43817,0.07806,0.13261,0.0624,0.015,8E-05},
				{1310,1.70703,2.02352,1.43096,0.87269,0.45284,0.08962,0.14074,0.06492,0.01488,8E-05},
				{1315,1.7107,2.0324,1.44765,0.88983,0.46839,0.10157,0.14903,0.0674,0.01473,9E-05},
				{1320,1.71455,2.04268,1.46721,0.90794,0.48476,0.11389,0.15746,0.06983,0.01456,9E-05},
				{1325,1.7186,2.0544,1.48977,0.92699,0.50191,0.12657,0.166,0.0722,0.01435,9E-05},
				{1330,1.72283,2.06752,1.51535,0.94694,0.5198,0.13956,0.17463,0.07448,0.01412,0.0001},
				{1335,1.72726,2.082,1.54394,0.96778,0.53838,0.15284,0.18333,0.07669,0.01386,0.0001},
				{1340,1.73188,2.09761,1.57517,0.98947,0.55762,0.16638,0.1921,0.07879,0.01357,0.0001},
				{1345,1.73669,2.11416,1.60872,1.01198,0.57747,0.18014,0.2009,0.08078,0.01327,0.0001},
				{1350,1.74168,2.13141,1.64411,1.03528,0.5979,0.19409,0.20973,0.08266,0.01294,0.0001},
				{1355,1.74687,2.14911,1.68074,1.05933,0.61886,0.2082,0.21857,0.0844,0.01259,0.0001},
				{1360,1.75224,2.16699,1.71798,1.08412,0.64031,0.22244,0.22739,0.086,0.01222,0.0001},
				{1365,1.7578,2.1849,1.75528,1.1096,0.66221,0.23678,0.23619,0.08744,0.01183,0.0001},
				{1370,1.76355,2.20267,1.79212,1.13575,0.68453,0.25118,0.24495,0.08872,0.01142,0.0001},
				{1375,1.76948,2.22025,1.82818,1.16253,0.70722,0.26561,0.25364,0.08982,0.011,9E-05},
				{1380,1.7756,2.23755,1.863,1.18992,0.73024,0.28004,0.26226,0.09073,0.01056,9E-05},
				{1385,1.78191,2.25468,1.89666,1.21789,0.75355,0.29444,0.27078,0.09145,0.0101,8E-05},
				{1390,1.78839,2.27153,1.92896,1.2464,0.77711,0.30879,0.2792,0.09196,0.00964,7E-05},
				{1395,1.79506,2.28807,1.95988,1.2754,0.80087,0.32308,0.28749,0.0923,0.00916,6E-05},
				{1400,1.80191,2.3043,1.9895,1.30486,0.8248,0.33732,0.29565,0.09246,0.00868,5E-05},
				{1405,1.80892,2.32031,2.01814,1.33474,0.84883,0.35149,0.30367,0.09248,0.00819,4E-05},
				{1410,1.81611,2.33611,2.04597,1.36499,0.87294,0.36561,0.31154,0.09237,0.00769,3E-05},
				{1415,1.82347,2.35183,2.0734,1.39557,0.89707,0.37966,0.31924,0.09214,0.0072,1E-05},
				{1420,1.83098,2.36761,2.10085,1.42644,0.92119,0.39364,0.32677,0.09182,0.0067,-1E-05},
				{1425,1.83866,2.3837,2.12881,1.45756,0.94524,0.40755,0.33411,0.09141,0.00621,-3E-05},
				{1430,1.8465,2.40026,2.15761,1.48889,0.96918,0.42139,0.34125,0.09094,0.00572,-5E-05},
				{1435,1.85448,2.41772,2.18799,1.52038,0.99296,0.43515,0.34818,0.09043,0.00524,-7E-05},
				{1440,1.86261,2.43632,2.22012,1.55198,1.01656,0.44883,0.35489,0.08987,0.00476,-0.0001},
				{1445,1.87088,2.4564,2.25442,1.58366,1.03994,0.46243,0.36137,0.08928,0.00429,-0.00013},
				{1450,1.87926,2.47821,2.29095,1.61534,1.06307,0.47594,0.36761,0.08864,0.00384,-0.00016},
				{1455,1.88775,2.50203,2.32979,1.64699,1.08592,0.48935,0.37359,0.08797,0.00338,-0.0002},
				{1460,1.89633,2.52791,2.37059,1.67854,1.10846,0.50267,0.37931,0.08724,0.00294,-0.00023},
				{1465,1.90498,2.55575,2.41275,1.70994,1.13066,0.51587,0.38474,0.08647,0.0025,-0.00027},
				{1470,1.9137,2.58528,2.45552,1.74115,1.1525,0.52897,0.38989,0.08565,0.00207,-0.00031},
				{1475,1.92247,2.61583,2.49765,1.77211,1.17394,0.54196,0.39473,0.08478,0.00164,-0.00036},
				{1480,1.93128,2.64682,2.53822,1.80277,1.19496,0.55482,0.39926,0.08386,0.00122,-0.00041},
				{1485,1.9401,2.67749,2.57631,1.83306,1.21553,0.56756,0.40347,0.08288,0.00081,-0.00046},
				{1490,1.94894,2.70719,2.61131,1.86296,1.23562,0.58015,0.40734,0.08184,0.0004,-0.00051},
				{1495,1.95779,2.73518,2.64271,1.89239,1.2552,0.59258,0.41088,0.08076,0,-0.00057},
				{1500,1.96663,2.76107,2.67053,1.92133,1.27426,0.60481,0.4141,0.07962,-0.0004,-0.00063},
				{1505,1.97546,2.78466,2.6951,1.94971,1.29278,0.61681,0.41699,0.07844,-0.0008,-0.00069},
				{1510,1.98427,2.80574,2.71662,1.9775,1.31074,0.62856,0.41955,0.07722,-0.00119,-0.00076},
				{1515,1.99305,2.82446,2.73579,2.00463,1.3281,0.64003,0.42179,0.07595,-0.00158,-0.00083},
				{1520,2.00181,2.84098,2.7532,2.03108,1.34486,0.65118,0.42371,0.07465,-0.00196,-0.0009},
				{1525,2.01052,2.85555,2.76943,2.05678,1.36098,0.66199,0.42531,0.07332,-0.00235,-0.00098},
				{1530,2.01919,2.86842,2.78501,2.08169,1.37646,0.67243,0.4266,0.07196,-0.00273,-0.00106},
				{1535,2.0278,2.87996,2.80048,2.10576,1.39126,0.68247,0.42757,0.07057,-0.0031,-0.00114},
				{1540,2.03635,2.8903,2.8161,2.12896,1.40537,0.6921,0.42823,0.06915,-0.00348,-0.00123},
				{1545,2.04483,2.89977,2.83218,2.15129,1.41881,0.7013,0.42858,0.06771,-0.00385,-0.00132},
				{1550,2.05324,2.90864,2.84896,2.17277,1.43157,0.71009,0.42864,0.06625,-0.00422,-0.00141},
				{1555,2.06156,2.91713,2.86654,2.1934,1.44367,0.71845,0.42841,0.06476,-0.00459,-0.00151},
				{1560,2.06979,2.92546,2.88488,2.2132,1.45512,0.7264,0.4279,0.06325,-0.00494,-0.00161},
				{1565,2.07792,2.93381,2.90387,2.23216,1.46592,0.73394,0.42712,0.06171,-0.00529,-0.00171},
				{1570,2.08595,2.94231,2.92325,2.25032,1.47609,0.74106,0.42607,0.06015,-0.00563,-0.00181},
				{1575,2.09387,2.95113,2.94275,2.26766,1.48563,0.74777,0.42477,0.05856,-0.00596,-0.00191},
				{1580,2.10166,2.96017,2.96173,2.28421,1.49455,0.75406,0.42322,0.05695,-0.00628,-0.00201},
				{1585,2.10933,2.96958,2.97993,2.29998,1.50287,0.75995,0.42143,0.05532,-0.00659,-0.00211},
				{1590,2.11686,2.97912,2.99666,2.31496,1.51058,0.76542,0.41941,0.05366,-0.00689,-0.00221},
				{1595,2.12426,2.98872,3.01163,2.32918,1.5177,0.77048,0.41717,0.05198,-0.00716,-0.0023},
				{1600,2.1315,2.99818,3.0245,2.34264,1.52424,0.77514,0.41471,0.05028,-0.00743,-0.0024},
				{1605,2.13859,3.00725,3.03509,2.35536,1.53021,0.77939,0.41204,0.04855,-0.00768,-0.00249},
				{1610,2.14551,3.01569,3.04332,2.36733,1.53561,0.78324,0.40917,0.0468,-0.00791,-0.00258},
				{1615,2.15227,3.02325,3.04931,2.37858,1.54046,0.78668,0.40611,0.04503,-0.00811,-0.00267},
				{1620,2.15884,3.02974,3.05324,2.38911,1.54476,0.78972,0.40287,0.04324,-0.0083,-0.00275},
				{1625,2.16523,3.03506,3.0554,2.39893,1.54852,0.79235,0.39945,0.04142,-0.00847,-0.00282},
				{1630,2.17143,3.03914,3.05609,2.40805,1.55175,0.79459,0.39587,0.03958,-0.00862,-0.0029},
				{1635,2.17743,3.04206,3.05566,2.41649,1.55446,0.79642,0.39212,0.03772,-0.00874,-0.00296},
				{1640,2.18323,3.0438,3.05428,2.42425,1.55667,0.79786,0.38823,0.03583,-0.00884,-0.00302},
				{1645,2.18883,3.04456,3.05224,2.43135,1.5584,0.79893,0.38421,0.03394,-0.0089,-0.00308},
				{1650,2.19425,3.04449,3.04972,2.43784,1.55969,0.79964,0.38008,0.03203,-0.00894,-0.00312},
				{1655,2.19949,3.04374,3.04683,2.44372,1.56058,0.80001,0.37586,0.03012,-0.00894,-0.00316},
				{1660,2.20457,3.04245,3.04363,2.44904,1.5611,0.80007,0.37157,0.02822,-0.0089,-0.00318},
				{1665,2.20949,3.04076,3.04018,2.4538,1.56127,0.79983,0.36722,0.02633,-0.00882,-0.00319},
				{1670,2.21426,3.03876,3.03649,2.45804,1.56113,0.79931,0.36285,0.02445,-0.00869,-0.00319},
				{1675,2.21889,3.03657,3.03256,2.46178,1.56073,0.79853,0.35847,0.0226,-0.00851,-0.00317},
				{1680,2.22339,3.03424,3.02844,2.46505,1.56008,0.79752,0.35409,0.02078,-0.00828,-0.00314},
				{1685,2.22778,3.03182,3.02406,2.46788,1.55922,0.79629,0.34975,0.019,-0.00799,-0.00309},
				{1690,2.23204,3.02936,3.01948,2.47029,1.5582,0.79487,0.34545,0.01726,-0.00764,-0.00302},
				{1695,2.2362,3.02689,3.01476,2.47235,1.55707,0.79332,0.34124,0.0156,-0.00723,-0.00294},
				{1700,2.24024,3.02446,3.00998,2.47413,1.55589,0.79171,0.33714,0.01403,-0.00674,-0.00283},
				{1705,2.24417,3.0221,3.00522,2.47568,1.55472,0.79009,0.33318,0.01259,-0.00618,-0.00269},
				{1710,2.24799,3.01984,3.00059,2.47707,1.55362,0.78852,0.3294,0.01129,-0.00554,-0.00253},
				{1715,2.2517,3.01771,2.99623,2.47837,1.55264,0.78708,0.32582,0.01016,-0.00482,-0.00234},
				{1720,2.2553,3.01574,2.99226,2.47962,1.55186,0.78581,0.32248,0.00922,-0.004,-0.00212},
				{1725,2.25879,3.01395,2.98884,2.48091,1.55133,0.78478,0.31941,0.00851,-0.00308,-0.00186},
				{1730,2.26217,3.01237,2.98611,2.48228,1.55111,0.78405,0.31663,0.00804,-0.00207,-0.00158},
				{1735,2.26543,3.01101,2.98419,2.48381,1.55125,0.78368,0.31418,0.00784,-0.00095,-0.00125},
				{1740,2.2686,3.00988,2.98319,2.48555,1.55181,0.78373,0.31209,0.00793,0.00028,-0.00089},
				{1745,2.27167,3.00897,2.98323,2.48753,1.55279,0.7842,0.31036,0.00833,0.00164,-0.00048},
				{1750,2.27466,3.00828,2.98436,2.48976,1.5542,0.78513,0.309,0.00904,0.00312,-4E-05},
				{1755,2.27759,3.00777,2.9866,2.49229,1.55605,0.78653,0.30801,0.01008,0.00473,0.00046},
				{1760,2.28046,3.00743,2.98993,2.49512,1.55834,0.78842,0.30742,0.01146,0.00649,0.00101},
				{1765,2.28329,3.00722,2.99427,2.49831,1.56109,0.7908,0.30721,0.0132,0.0084,0.00161},
				{1770,2.2861,3.00712,2.99948,2.50185,1.56429,0.79371,0.30741,0.0153,0.01047,0.00226},
				{1775,2.28889,3.00709,3.00539,2.5058,1.56796,0.79716,0.30802,0.01778,0.0127,0.00297},
				{1780,2.29169,3.1971,3.01177,2.51016,1.5721,0.80117,0.30905,0.02065,0.01511,0.00375},
				{1785,2.2945,3.00713,3.01836,2.51497,1.57672,0.80575,0.3105,0.02392,0.01769,0.00458},
				{1790,2.29733,3.00727,3.02501,2.52024,1.58182,0.81092,0.31238,0.0276,0.02047,0.00549},
				{1795,2.30019,3.00743,3.03129,2.52598,1.58741,0.81667,0.3147,0.03168,0.02343,0.00646},
				{1800,2.30308,3.00769,3.03701,2.53218,1.5935,0.82299,0.31748,0.03616,0.02657,0.0075},
				{1805,2.30599,3.1981,3.04192,2.53883,1.60008,0.82989,0.3207,0.04101,0.0299,0.00861},
				{1810,2.30894,3.00875,3.04579,2.54593,1.60716,0.83734,0.3244,0.04625,0.03341,0.00979},
				{1815,2.31191,3.00975,3.04841,2.55348,1.61474,0.84535,0.32856,0.05185,0.0371,0.01104},
				{1820,2.31492,3.01122,3.04963,2.56148,1.62284,0.85391,0.3332,0.05781,0.04098,0.01236},
				{1825,2.31795,3.01333,3.04935,2.56991,1.63144,0.86302,0.33833,0.06413,0.04503,0.01375},
				{1830,2.32102,3.01624,3.04762,2.57878,1.64056,0.87266,0.34395,0.07078,0.04926,0.01521},
				{1835,2.32413,3.02014,3.04457,2.58808,1.6502,0.88283,0.35007,0.07777,0.05367,0.01675},
				{1840,2.32727,3.02528,3.04081,2.59781,1.66037,0.89353,0.35671,0.08509,0.05826,0.01836},
				{1845,2.33044,3.03168,3.03673,2.60797,1.67106,0.90475,0.36386,0.09272,0.06301,0.02005},
				{1850,2.33366,3.03949,3.03328,2.61854,1.68229,0.91647,0.37154,0.10066,0.06795,0.02181},
				{1855,2.33691,3.04874,3.03143,2.62953,1.69405,0.9287,0.37975,0.1089,0.07305,0.02365},
				{1860,2.3402,3.0594,3.03217,2.64094,1.70635,0.94143,0.3885,0.11743,0.07833,0.02557},
				{1865,2.34353,3.07137,3.03635,2.65275,1.71919,0.95466,0.3978,0.12624,0.08378,0.02756},
				{1870,2.3469,3.08454,3.04459,2.66497,1.73259,0.96837,0.40765,0.13533,0.0894,0.02963},
				{1875,2.35032,3.09878,3.05724,2.67759,1.74653,0.98256,0.41807,0.14468,0.09518,0.03179},
				{1880,2.35378,3.11396,3.07431,2.69061,1.76103,0.99722,0.42906,0.15429,0.10114,0.03402},
				{1885,2.35728,3.12996,3.09565,2.70402,1.77609,1.01235,0.44063,0.16415,0.10726,0.03633},
				{1890,2.36083,3.1466,3.12083,2.71781,1.7917,1.02793,0.45277,0.17425,0.11354,0.03872},
				{1895,2.3644,3.16367,3.14918,2.73195,1.80781,1.04392,0.46544,0.18455,0.11996,0.04119},
				{1900,2.36801,3.181,3.18008,2.74642,1.82439,1.06029,0.4786,0.19504,0.1265,0.04371},
				{1905,2.37163,3.19834,3.21275,2.76118,1.84138,1.07698,0.4922,0.20568,0.13315,0.0463},
				{1910,2.37525,3.2154,3.24639,2.77619,1.85874,1.09395,0.50619,0.21644,0.13988,0.04894},
				{1915,2.37887,3.23188,3.28016,2.79144,1.87643,1.11116,0.52052,0.2273,0.14668,0.05162},
				{1920,2.38247,3.24747,3.31324,2.80688,1.89439,1.12858,0.53516,0.23822,0.15352,0.05433},
				{1925,2.38605,3.26188,3.34488,2.82248,1.91258,1.14615,0.55006,0.24919,0.16039,0.05707},
				{1930,2.3896,3.27487,3.37443,2.83822,1.93096,1.16384,0.56516,0.26017,0.16726,0.05983},
				{1935,2.3931,3.28629,3.40141,2.85406,1.94947,1.18161,0.58043,0.27113,0.17413,0.0626},
				{1940,2.39655,3.29596,3.42532,2.86997,1.96808,1.1994,0.59582,0.28205,0.18096,0.06538},
				{1945,2.39994,3.30388,3.44597,2.88593,1.98673,1.21718,0.61129,0.2929,0.18774,0.06815},
				{1950,2.40326,3.31012,3.46328,2.90189,2.00539,1.23491,0.62678,0.30364,0.19445,0.07092},
				{1955,2.4065,3.31482,3.47731,2.91783,2.02399,1.25255,0.64225,0.31426,0.20108,0.07366},
				{1960,2.40965,3.31816,3.48822,2.93371,2.04251,1.27005,0.65766,0.32472,0.20759,0.07638},
				{1965,2.41269,3.32044,3.49632,2.94951,2.06089,1.28736,0.67296,0.335,0.21398,0.07907},
				{1970,2.41563,3.32179,3.50179,2.96519,2.07908,1.30446,0.68811,0.34507,0.22022,0.08172},
				{1975,2.41845,3.32254,3.50506,2.98073,2.09704,1.32129,0.70305,0.35489,0.22629,0.08432},
				{1980,2.42114,3.32293,3.50649,2.99609,2.11473,1.33781,0.71775,0.36445,0.23218,0.08686},
				{1985,2.42369,3.32318,3.50645,3.01123,2.13209,1.35398,0.73216,0.3737,0.23786,0.08934},
				{1990,2.4261,3.32346,3.50529,3.02615,2.1491,1.36979,0.74625,0.38265,0.24332,0.09175},
				{1995,2.42837,3.3239,3.50335,3.04085,2.16577,1.38523,0.76001,0.3913,0.24858,0.0941},
				{2000,2.43052,3.3246,3.50095,3.05532,2.18212,1.40032,0.77347,0.39966,0.25362,0.09637},
				{2005,2.43254,3.32561,3.49838,3.0696,2.19817,1.4151,0.78662,0.40776,0.25847,0.09857},
				{2010,2.43445,3.32696,3.49592,3.08368,2.21393,1.42957,0.79948,0.41561,0.26313,0.10071},
				{2015,2.43626,3.32865,3.49379,3.09757,2.22941,1.44376,0.81205,0.42322,0.2676,0.10277},
				{2020,2.43797,3.33067,3.4922,3.11129,2.24465,1.45768,0.82434,0.43061,0.2719,0.10477},
				{2025,2.43959,3.33296,3.49131,3.12484,2.25964,1.47135,0.83637,0.4378,0.27602,0.10669},
				{2030,2.44113,3.3355,3.49125,3.13823,2.27442,1.48479,0.84813,0.4448,0.27998,0.10855},
				{2035,2.4426,3.33826,3.49211,3.15148,2.28899,1.49802,0.85963,0.45163,0.28378,0.11033},
				{2040,2.444,3.34111,3.49372,3.16458,2.30338,1.51106,0.87089,0.45831,0.28743,0.11205},
				{2045,2.44535,3.34405,3.49616,3.17756,2.31759,1.52392,0.88191,0.46485,0.29094,0.11369},
				{2050,2.44665,3.34702,3.49935,3.19042,2.33166,1.53664,0.8927,0.47126,0.29431,0.11527},
				{2055,2.44791,3.34996,3.50318,3.20317,2.34559,1.54921,0.90327,0.47758,0.29755,0.11678},
				{2060,2.44914,3.35283,3.50751,3.21581,2.3594,1.56167,0.91363,0.4838,0.30066,0.11821},
				{2065,2.45034,3.35558,3.51224,3.22837,2.37312,1.57403,0.92377,0.48995,0.30366,0.11957},
				{2070,2.45152,3.35816,3.51725,3.24084,2.38675,1.58631,0.93373,0.49604,0.30655,0.12087},
				{2075,2.4527,3.36053,3.52245,3.25325,2.40031,1.59853,0.94349,0.50209,0.30933,0.12209},
				{2080,2.45388,3.36265,3.52782,3.26559,2.41382,1.61071,0.95306,0.50811,0.31202,0.12324},
				{2085,2.45506,3.36448,3.53335,3.27788,2.4273,1.62286,0.96247,0.51413,0.31462,0.12433},
				{2090,2.45626,3.36599,3.53911,3.29012,2.44076,1.63501,0.97171,0.52015,0.31713,0.12534},
				{2095,2.45747,3.36713,3.54518,3.30232,2.4542,1.64714,0.98078,0.52617,0.31957,0.12627},
				{2100,2.4587,3.36785,3.5517,3.31448,2.46763,1.65927,0.98971,0.53219,0.32192,0.12714},
				{2105,2.45995,3.3681,3.55882,3.32661,2.48105,1.67138,0.99848,0.5382,0.32419,0.12793},
				{2110,2.46121,3.36785,3.56668,3.3387,2.49446,1.68349,1.1971,0.5442,0.32639,0.12864},
				{2115,2.46248,3.36709,3.57531,3.35077,2.50787,1.69558,1.01559,0.55019,0.32851,0.12928},
				{2120,2.46378,3.36582,3.58467,3.3628,2.52126,1.70766,1.02394,0.55616,0.33056,0.12983},
				{2125,2.4651,3.36413,3.59451,3.37482,2.53466,1.71973,1.03217,0.56211,0.33254,0.13031},
				{2130,2.46643,3.36214,3.60448,3.38682,2.54806,1.73178,1.04027,0.56805,0.33445,0.1307},
				{2135,2.46779,3.36001,3.61413,3.39879,2.56146,1.74382,1.04825,0.57395,0.33629,0.13101},
				{2140,2.46917,3.35797,3.62309,3.41076,2.57486,1.75585,1.05612,0.57983,0.33807,0.13124},
				{2145,2.47056,3.35605,3.63094,3.42271,2.58827,1.76786,1.06388,0.58568,0.33978,0.13138},
				{2150,2.47199,3.35441,3.63757,3.43466,2.60169,1.77986,1.07154,0.59149,0.34143,0.13143},
				{2155,2.47343,3.3531,3.64301,3.4466,2.61512,1.79184,1.0791,0.59726,0.34301,0.1314},
				{2160,2.4749,3.35214,3.64737,3.45854,2.62857,1.8038,1.08657,0.60299,0.34454,0.13127},
				{2165,2.47639,3.35154,3.65085,3.47048,2.64203,1.81575,1.09396,0.60868,0.34602,0.13106},
				{2170,2.47791,3.3513,3.65368,3.48243,2.65551,1.82768,1.10126,0.61432,0.34743,0.13075},
				{2175,2.47945,3.35147,3.65608,3.49439,2.66901,1.83959,1.10849,0.61991,0.3488,0.13035},
				{2180,2.48102,3.35223,3.65849,3.50635,2.68254,1.85148,1.11565,0.62544,0.35011,0.12985},
				{2185,2.48262,3.35353,3.66096,3.51833,2.69609,1.86335,1.12274,0.63092,0.35137,0.12925},
				{2190,2.48425,3.35566,3.66378,3.53033,2.70967,1.8752,1.12977,0.63634,0.35259,0.12856},
				{2195,2.4859,3.3589,3.66741,3.54235,2.72328,1.88704,1.13675,0.64169,0.35375,0.12777},
				{2200,2.48758,3.36365,3.6723,3.55439,2.73693,1.89885,1.14368,0.64697,0.35488,0.12688},
				{2205,2.4893,3.3704,3.67895,3.56646,2.7506,1.91064,1.15057,0.65219,0.35596,0.12589},
				{2210,2.49104,3.37963,3.68794,3.57856,2.76432,1.9224,1.15742,0.65733,0.357,0.12479},
				{2215,2.49282,3.39172,3.69969,3.59069,2.77807,1.93415,1.16424,0.6624,0.358,0.1236},
				{2220,2.49463,3.40674,3.7144,3.60285,2.79187,1.94587,1.17103,0.66738,0.35896,0.12229},
				{2225,2.49647,3.42421,3.73174,3.61506,2.80571,1.95756,1.17779,0.67228,0.35989,0.12088},
				{2230,2.49835,3.44326,3.75104,3.62731,2.8196,1.96923,1.18454,0.6771,0.36078,0.11936},
				{2235,2.50025,3.46231,3.77088,3.6396,2.83354,1.98088,1.19128,0.68183,0.36165,0.11773},
				{2240,2.5022,3.48011,3.78998,3.65194,2.84753,1.9925,1.19802,0.68646,0.36248,0.11598},
				{2245,2.50418,3.49581,3.80762,3.66433,2.86157,2.1941,1.20475,0.691,0.36328,0.11413},
				{2250,2.5062,3.50913,3.82344,3.67678,2.87567,2.01566,1.21149,0.69544,0.36406,0.11216},
				{2255,2.50825,3.52022,3.83754,3.68929,2.88983,2.0272,1.21823,0.69977,0.36481,0.11008},
				{2260,2.51034,3.52949,3.85027,3.70185,2.90405,2.03872,1.225,0.704,0.36554,0.10788},
				{2265,2.51247,3.53741,3.86203,3.71448,2.91833,2.0502,1.23178,0.70812,0.36624,0.10556},
				{2270,2.51464,3.54443,3.87322,3.72718,2.93268,2.06165,1.23859,0.71213,0.36693,0.10313},
				{2275,2.51685,3.5509,3.88415,3.73994,2.94709,2.07308,1.24543,0.71603,0.3676,0.10057},
				{2280,2.5191,3.55707,3.89503,3.75278,2.96158,2.08447,1.25231,0.71981,0.36826,0.09789},
				{2285,2.52139,3.56312,3.90596,3.7657,2.97614,2.09584,1.25923,0.72346,0.3689,0.09509},
				{2290,2.52372,3.56913,3.91699,3.77869,2.99077,2.10717,1.26619,0.72699,0.36952,0.09217},
				{2295,2.52609,3.57517,3.92809,3.79175,3.00546,2.11846,1.27319,0.73041,0.37014,0.08914},
				{2300,2.5285,3.58122,3.93914,3.80486,3.0202,2.12972,1.28021,0.73371,0.37074,0.08599},
				{2305,2.53093,3.58725,3.95002,3.81803,3.03497,2.14093,1.28726,0.73689,0.37132,0.08276},
				{2310,2.5334,3.5932,3.96055,3.83123,3.04978,2.15209,1.29431,0.73998,0.37189,0.07943},
				{2315,2.53589,3.599,3.97054,3.84446,3.06461,2.16319,1.30137,0.74296,0.37245,0.07603},
				{2320,2.5384,3.60457,3.97979,3.85771,3.07944,2.17424,1.30842,0.74584,0.37299,0.07255},
				{2325,2.54093,3.60985,3.98817,3.87097,3.09426,2.18523,1.31545,0.74863,0.37351,0.06901},
				{2330,2.54348,3.61467,3.99541,3.88423,3.10907,2.19615,1.32246,0.75133,0.37402,0.06542},
				{2335,2.54603,3.61904,4.00148,3.89749,3.12385,2.20701,1.32944,0.75395,0.3745,0.06178},
				{2340,2.5486,3.6229,4.00632,3.91072,3.1386,2.21778,1.33638,0.75649,0.37498,0.0581},
				{2345,2.55117,3.62622,4.00994,3.92392,3.15329,2.22848,1.34326,0.75895,0.37543,0.0544},
				{2350,2.55375,3.62901,4.01239,3.93709,3.16793,2.2391,1.35009,0.76134,0.37586,0.05068},
				{2355,2.55632,3.6313,4.01379,3.9502,3.1825,2.24963,1.35684,0.76366,0.37628,0.04694},
				{2360,2.5589,3.6331,4.01428,3.96326,3.19698,2.26006,1.36353,0.76592,0.37668,0.0432},
				{2365,2.56146,3.63449,4.01403,3.97626,3.21137,2.2704,1.37012,0.76813,0.37705,0.03947},
				{2370,2.56401,3.63552,4.0132,3.98917,3.22566,2.28065,1.37663,0.77027,0.37741,0.03576},
				{2375,2.56655,3.63624,4.01196,4.002,3.23983,2.29079,1.38303,0.77237,0.37774,0.03206}};
			
			const G4int coef_ener = G4int((pLab-Legendre_coef[0][0])/5);
			const G4double sup_ener = pLab/5. - coef_ener -Legendre_coef[0][0]/5;
			//std::cout << "sup_ener\t" << sup_ener << std::endl;
			
// assert(pLab >= Legendre_coef[coef_ener][0] && pLab < Legendre_coef[coef_ener+1][0]);
			
			// Legendre coefficient normalized
			const G4double A0 = 1.;
			const G4double A1 = (1-sup_ener)*Legendre_coef[coef_ener][1] + sup_ener*Legendre_coef[coef_ener+1][1];
			const G4double A2 = (1-sup_ener)*Legendre_coef[coef_ener][2] + sup_ener*Legendre_coef[coef_ener+1][2];
			const G4double A3 = (1-sup_ener)*Legendre_coef[coef_ener][3] + sup_ener*Legendre_coef[coef_ener+1][3];
			const G4double A4 = (1-sup_ener)*Legendre_coef[coef_ener][4] + sup_ener*Legendre_coef[coef_ener+1][4];
			const G4double A5 = (1-sup_ener)*Legendre_coef[coef_ener][5] + sup_ener*Legendre_coef[coef_ener+1][5];
			const G4double A6 = (1-sup_ener)*Legendre_coef[coef_ener][6] + sup_ener*Legendre_coef[coef_ener+1][6];
			const G4double A7 = (1-sup_ener)*Legendre_coef[coef_ener][7] + sup_ener*Legendre_coef[coef_ener+1][7];
			const G4double A8 = (1-sup_ener)*Legendre_coef[coef_ener][8] + sup_ener*Legendre_coef[coef_ener+1][8];
			const G4double A9 = (1-sup_ener)*Legendre_coef[coef_ener][9] + sup_ener*Legendre_coef[coef_ener+1][9];
			const G4double A10 = (1-sup_ener)*Legendre_coef[coef_ener][10] + sup_ener*Legendre_coef[coef_ener+1][10];
			
			// Theoritical max if all Ai > 0 (often the case)
			const G4double A = std::fabs(A0) + std::fabs(A1) + std::fabs(A2) + std::fabs(A3) + std::fabs(A4) + std::fabs(A5) + std::fabs(A6) + std::fabs(A7) + std::fabs(A8) + std::fabs(A9) + std::fabs(A10);
			
			G4bool success = false;
			G4int maxloop = 0;
			
			while(!success && maxloop < 1000){
				
				cos_theta = Random::shoot()*2-1.; // not optimized
				
				// Legendre Polynomial
				G4double P0 = A0;
				G4double P1 = A1*cos_theta;
				G4double P2 = A2/2.*(3*std::pow(cos_theta,2)-1);
				G4double P3 = A3/2.*(5*std::pow(cos_theta,3)-3*cos_theta);
				G4double P4 = A4/8.*(35*std::pow(cos_theta,4)-30*std::pow(cos_theta,2)+3);
				G4double P5 = A5/8.*(63*std::pow(cos_theta,5)-70*std::pow(cos_theta,3)+15*cos_theta);
				G4double P6 = A6/16.*(231*std::pow(cos_theta,6)-315*std::pow(cos_theta,4)+105*std::pow(cos_theta,2)-5);
				G4double P7 = A7/16.*(429*std::pow(cos_theta,7)-693*std::pow(cos_theta,5)+315*std::pow(cos_theta,3)-35*cos_theta);
				G4double P8 = A8/128.*(6435*std::pow(cos_theta,8)-12012*std::pow(cos_theta,6)+6930*std::pow(cos_theta,4)-1260*std::pow(cos_theta,2)+35);
				G4double P9 = A9/128.*(12155*std::pow(cos_theta,9)-25740*std::pow(cos_theta,7)+18018*std::pow(cos_theta,5)-4620*std::pow(cos_theta,3)+315*cos_theta);
				G4double P10 = A10/256.*(46189*std::pow(cos_theta,10)-109395*std::pow(cos_theta,8)+90090*std::pow(cos_theta,6)-30030*std::pow(cos_theta,4)+3465*std::pow(cos_theta,2)-63);
				
				G4double P = (P0 + P1 + P2 + P3 + P4 + P5 + P6 + P7 + P8 + P9 + P10);
				
				if(Random::shoot()*A < P) success = true;
				maxloop +=1 ;
				if(maxloop==1000) cos_theta = std::log(Random::shoot()*(std::exp(10.)-std::exp(-10.))+std::exp(-10.))/10.; // if no success in 1E4 shoot, probably angulard distribution piked very forward 
			}
			sin_theta = std::sqrt(1-cos_theta*cos_theta);
		}
		
		if(rho == 0) return ThreeVector(sin_theta*cos_phi,sin_theta*sin_phi,cos_theta)*norm;
		// Rotation in the direction of the incident kaon
		const G4double px = x/r*cos_theta - y/rho*sin_theta*cos_phi + z/r*x/rho*sin_theta*sin_phi;
		const G4double py = y/r*cos_theta + x/rho*sin_theta*cos_phi + z/r*y/rho*sin_theta*sin_phi;
		const G4double pz = z/r*cos_theta - rho/r*sin_theta*sin_phi;
		
		
		return ThreeVector(px,py,pz)*norm;
	}
}
