package libkb

import (
	jsonw "github.com/keybase/go-jsonw"
)

func merkleRootFromString(raw string) (*MerkleRoot, error) {
	jw, err := jsonw.Unmarshal([]byte(raw))
	if err != nil {
		return nil, err
	}
	ret, err := NewMerkleRootFromJSON(jw, MerkleOpts{})
	if err != nil {
		return nil, err
	}
	return ret, nil
}

func merkleRootFromStringOrPanic(raw string) *MerkleRoot {
	root, err := merkleRootFromString(raw)
	if err != nil {
		panic(err)
	}
	return root
}

// Merkle Sequence #5323658, our current test merkle checkpoint
var prodCheckpoint5323658 = merkleRootFromStringOrPanic(
	`{"status":{"code":0,"name":"OK"},"hash":"84598ec296e526e7f299f425d72d487273892c14f385e828577d78cd740c013defe07cfb33cfd0c73271c9f77dae0ec78054314c2baa2e48900985bc575712d5","seqno":5323658,"ctime_string":"2019-05-07T18:39:10.000Z","ctime":1557254350,"payload_json":"{\"body\":{\"kbfs\":{\"private\":{\"root\":\"iKNlcGvEIH2J6deUCtelAZ682yv+zt/7UH5BmczbfonfrUpP1b19oWjEQFQ9kf0SCAKwVRW0MMYECfQq9mrmwlt3bPqonX0gKq87Yax4nkYlAwluIe+oL/eTTmsPYlmhVFzdIjWR0UxV/xajbm9uxBgLnx1jEzZknH7s3m3JbJFPcYJI741IE5WicHLEQMsO3W350HphuYbhCYThAyfe9XfKwwNCo1HgprVsrAG1kkyq+AE54mJdecjnDJKLFxJvfJCZsFp5h/puaIUXFT+ic27NZ1GhdAKidHPOXNHH4KF2AQ==\",\"version\":1},\"privateteam\":{\"root\":\"iKNlcGvEINojxeGAH+JJZo2IEBjsotElIa+scsmGKIEvPp5OPCk7oWjEQCWTxxZ7FlZWPxW39XOT1WnWGss4ybAKBmCR25XofMs08uwd2M9DQHaJh9Gno4Tp+8s8NVlUu84ohO4o08YNn4Gjbm9uxBj8l4GniZSP5Mh9/m1ZzrM3Gr1FvxFiXN+icHLEQJ+JU0cZj7R8hCp7K/aCydFMdvqCSbzmLocdDslR0blC14/47uf4MBDoK+o63ZapXLnXGmfmuxhY1HSZnrP8HqWic27NOxmhdAOidHPOXNHH4aF2AQ==\",\"version\":1},\"public\":{\"root\":\"hqFoxED55swUePDdjFmSa8c/lei2qwSq/WKjxV9wj+TbUmUmYDyxlYnPS2HMbctryl0OlJrlsmgo79DzkcIhdA4QJMy8onByxEDHa6+9XuvtNYjaVFMB0vyrnWytQE/+FlpasGVrUEQj50/r6Ab1HGzFz6NrB9g/a15DaBgvqA/1YcX9zZh2O9igonNuzWdjoXQBonRzzlzRx7mhdgE=\",\"version\":1}},\"key\":{\"fingerprint\":\"03e146cdaf8136680ad566912a32340cec8c9492\"},\"legacy_uid_root\":\"0c58a74abcb1db656146ebf667b5ea5cc55e208fbb534cd1b1e9454890358e5e\",\"prev\":\"132baf64b8a6ce5a165855200c36fafc36936085a8383852555168f42a2486ed75503d7c7d0634fc30a32f1b41d265ed727b1436b92bf33e89d67f5e0333db7c\",\"proof_services_hash\":\"c2ba2ac10ae88fd44620a5a911e3c9c060317423fb934c0563da96eb49b88caa1f9f03ac0952fbb1b45b80b375f399f34db0c5d82262ac2007e7c3c2090217d8\",\"pvl_hash\":\"bb64e9d94bea9e0e34e50bb020078bca8d4f8d22833ef877c052e98fe93ab0cd0946f16b7cbf551155a546aceda2b7131fd99e003c7258c0157b30879aa61b8b\",\"root\":\"84598ec296e526e7f299f425d72d487273892c14f385e828577d78cd740c013defe07cfb33cfd0c73271c9f77dae0ec78054314c2baa2e48900985bc575712d5\",\"seqno\":5323658,\"skips\":{\"1129354\":\"a4050d2c4adad6a3c1d8b5a9865db17099d288dcbc618022ea1a6780a23f227c\",\"3226506\":\"446f5752c4cf7f9378c7dbd596a9327499cfe559d168158e8720fdef176ebcf2\",\"4275082\":\"92e62fd26df550746e673f6a4cc5646e7771cd197c14a28e56a517ba5d85b613\",\"4799370\":\"341f3286918f5183b09b9307480720659709f659c771c6f436503f40997dcf75\",\"5061514\":\"20f01b509c345f7aeda5518f6a08a043eadf2050f17e336058db8ccf8f8ec928\",\"5192586\":\"4af89a99851461f0d761215e575c1b42815de31f978cf37576da0a2bf3557dbe\",\"5258122\":\"239bca2afe7e27034a11f9d7f4b5c2cded6857845f9b65f9dbea92f712240ba7\",\"5290890\":\"c0d7e22c9297c5ce3e0d8da2e6da2669c12d22d70f302f92c23586857a6fe4d9\",\"5307274\":\"6382e614e83579bf8cf851771faa82eeef6489e82a1a18997e40b804249f922c\",\"5315466\":\"204621960e887966fcc371ff0643f4de00c8098bf7b5b7b6fa3f4415870e9cc8\",\"5319562\":\"83ba107c78995c5851f565034b9791258335b84c53d8afd53b646c879ecf5063\",\"5321610\":\"14da36084f1edb912261e2d3463927fc6e41d048b09473cf8b276337f8e64541\",\"5322634\":\"07c7bc19a98d45f4b0c9d04176df4e9d481db01983df039aae405ce96bdd44b0\",\"5323146\":\"fdfc86be190c47d16d299d4317e95abd1ea6e09db0da6ebd33a624d15f6c0d3e\",\"5323402\":\"92b272094c55bbc4f5c07bf1981d2c96502fb4f1d7f8f0ff6640f035c6502586\",\"5323530\":\"ec237f22af0f6da778b495f89f2a40b66ee491509f71b1d562f3dc5cffbeee8c\",\"5323594\":\"52adc77439479e33220260c7f90011e16efef74c5ebf2f64ecbc619abf2549ad\",\"5323626\":\"203341629b8a220ca5960a1cc13053c20cb1edc4f9b91eec6f8be9ef431db05c\",\"5323642\":\"9a191f55c878878e1f3023603f1c6be960f2a61218bf992e863a8a0008e22bdd\",\"5323650\":\"44209b4a053af41c0825e14f1a31a05fc3680a98ba4297856fb17002ce9f1a69\",\"5323654\":\"db3d3fc66783edb25653dabb46a13504c346d058d5b58ac14ffd28017c9111b5\",\"5323656\":\"ce071810ff5b51fb3e5afc775f2c26b6ba131d525073361a6d5c1a8e9936daa7\",\"5323657\":\"cb9eba820ad2b38614a1c6b0626ac3c2e4d287b433ec5c88fb03a80b06661fb7\"},\"txid\":\"0894629d55e838f53dcb191a16d58315\",\"type\":\"merkle_root\"},\"ctime\":1557254350,\"tag\":\"signature\"}","txid":"0894629d55e838f53dcb191a16d58315","hash160":"a01e1f81868c2941096072fb26e4ab23b68346e0","payload_hash":"fcaedac1f83bd44afa9b536d119c7936a66c938029c29b7e93d49980029873b1","hash_meta":"fcaedac1f83bd44afa9b536d119c7936a66c938029c29b7e93d49980029873b1","sigs":{"010159baae6c7d43c66adf8fb7bb2b8b4cbe408c062cfc369e693ccb18f85631dbcd0a":{"sig":"-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n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\nveHD\n=KDa4\n-----END PGP MESSAGE-----\n","fingerprint":"03E146CDAF8136680AD566912A32340CEC8C9492"},"01209ec31411b9b287f62630c2486005af27548ba62a59bbc802e656b888991a20230a":{"sig":"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","fingerprint":null}},"key_fingerprint":"03E146CDAF8136680AD566912A32340CEC8C9492"}`,
)

func GetMerkleCheckpoint(m MetaContext) *MerkleRoot {
	switch m.G().Env.GetRunMode() {
	case ProductionRunMode:
		return prodCheckpoint5323658
	default:
		return nil
	}
}
