/*  rspoint.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef RSPOINT_H_
#define RSPOINT_H_

#include <set>
#include <ginac/ginac.h>
#include "functions.h"
#include "yamlconfigurable.h"

// forward declarations
namespace YAML {
class Node;
class Emitter;
}
namespace Reduze {
class Sector;
class INT;
}

namespace Reduze {

/// point in the (r,s)-plane
class RSPoint {
public:
	RSPoint();
	RSPoint(int r, int s);
	int r() const;
	int s() const;
	bool operator<(const RSPoint& other) const {
		if (r_ != other.r_)
			return (r_ < other.r_);
		return s_ < other.s_;
	}
private:
	int r_, s_;
};

//
//
//


/// generic selection of ranges in the (r,s)-plane, default is a full range
class RSGenericSelection: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("rs_generic_selection");
		s.set_short_description("Range in (r,s) for general t.");
		s.set_long_description(""//
					"Generic range for r: [r_min, r_max] and s: [s_min, s_max]."
					" r is the sum of the propagator exponents in the denominator of an integral (r >= t)"
					" s is the sum of the propagator exponents in the numerator of an integral (s >=0)"
					" r_min and r_max may depend on the symbol \"t\" (number of propagators)."
					" The value -1  for the end points r_max and s_max means no upper limit"
					" A full range is r: [t, -1] and s: [0, -1]."
					" The empty range is r: [0, 0] and s: [0, 0]");
		s.add_option("r", false, "sequence", ""//
					"Minimal r, maximal r.");
		s.add_option("s", false, "sequence", ""//
					"Minimal s, maximal s.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	/// constructs a full RSGenericSelection
	RSGenericSelection() :
		t_symbol_("t"), r_min_(0), r_max_(-1), s_min_(0), s_max_(-1) {
	}
	virtual ~RSGenericSelection() {
	}

	virtual void print(YAML::Emitter& os) const;
	virtual void read(const YAML::Node&);

	bool is_finite() const;
	/// empty ranges has r_min_, r_max_, s_min_, s_max_ set to zero
	bool is_empty() const;
	int r_min(int t) const;
	int r_max(int t) const;
	int s_min(int t) const;
	int s_max(int t) const;
	std::pair<GiNaC::ex, GiNaC::ex> r_bounds() const;
	std::pair<GiNaC::ex, GiNaC::ex> s_bounds() const;
	virtual void set_r_bounds(const GiNaC::ex& rmin, const GiNaC::ex& rmax);
	virtual void set_s_bounds(const GiNaC::ex& smin, const GiNaC::ex& smax);
	bool contains(int t, const INT& i) const;
	static const RSGenericSelection full_range;

private:
	int get_explicite_value(const GiNaC::ex& e, int t) const;
	GiNaC::symbol t_symbol_;
	GiNaC::ex r_min_, r_max_;
	GiNaC::ex s_min_, s_max_;
};

inline void operator>>(const YAML::Node& n, RSGenericSelection& r) {
	r.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const RSGenericSelection& r) {
	r.print(ye);
	return ye;
}

//
//
//


/// generic selection of ranges in the (r,s)-plane, default is an empty range
class RSFiniteGenericSelection: public RSGenericSelection {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s = RSGenericSelection::yaml_spec();
		s.set_keyword("rs_finite_generic_selection");
		s.set_short_description("Finite range in (r,s) for general t.");
		s.set_long_description(""//
					"Finite generic range for r: [r_min, r_max] and s: [s_min, s_max]."
					" r_min and r_max may depend on the symbol \"t\" (number of"
					" propagators). r is the sum of the propagator exponents in"
					" the denominator of an integral (r >= t) s is the sum of"
					" the propagator exponents in the numerator of an integral"
					" (s >=0)."
					" The empty range is r: [0, 0] and s: [0, 0]");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	// contructs an empty selection
	RSFiniteGenericSelection() {
		set_r_bounds(0, 0);
		set_s_bounds(0, 0);
		ASSERT(is_empty());
	}
	RSFiniteGenericSelection(const RSGenericSelection& r);
	virtual ~RSFiniteGenericSelection() {
	}
	void find_points(int t, std::set<RSPoint>& points) const;
	virtual void read(const YAML::Node&);
	virtual void set_r_bounds(const GiNaC::ex& rmin, const GiNaC::ex& rmax);
	virtual void set_s_bounds(const GiNaC::ex& smin, const GiNaC::ex& smax);
	static void find_upper_border(
			const std::list<RSFiniteGenericSelection>& rs, int t, std::set<
					RSPoint>& points);

	long int num_integrals(int n, int t) const;

protected:
	void init();
};

inline void operator>>(const YAML::Node& n, RSFiniteGenericSelection& r) {
	r.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye,
		const RSFiniteGenericSelection& r) {
	r.print(ye);
	return ye;
}

/// whether integral is contained in any of the ranges in r for given t
template<class R>
bool rsranges_contain(const std::list<R>& r, int t, const INT& i);

//
//
//

/// generic selection of identities by type and finite (r,s)-ranges
class IdentityGenericSelection: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		static const std::string rs_sequence =
				"sequence of rs_finite_generic_selection";
		YAMLSpec s;
		s.set_keyword("identity_generic_selection");
		s.set_short_description(""//
					"Finite generic ranges of identities by keywords");
		s.set_long_description(""//
					"A class to select identities of different types by a"
					" selection of integrals belonging to a finite range of the"
					" propagator exponent sums r and s from which the identities"
					" are then generated. Typical identities are"
					" integration-by-parts (IBP) identities,"
					" lorentz-invariant (LI) identities and identities generated"
					" from shift relations as they are defined in the sector"
					" mapping files.");

		s.add_option("ibp", false, rs_sequence, ""//
					"Selects a range of IBP identities.");
		s.add_option("ibp_dim", false, rs_sequence, ""//
					"Selects a range of the subset of the IBPs which all have"
					" the dimension symbol included in the coefficients.");
		s.add_option("ibp_dim_free", false, rs_sequence, ""//
					"Selects a range of the subset of the IBPs which don't have"
					" the dimension symbol included in the coefficients.");
		s.add_option("ibp_lee", false, rs_sequence, ""//
					"Selects a range of IBPs according to arXiv:0804.3008v2"
					" [hep-ph].");
		s.add_option("lorentz", false, rs_sequence, ""//
					"Selects a range of LI identities.");
		s.add_option("sector_relations", false, rs_sequence, ""//
					"Selects a range of identities built from sector relations.");
		s.add_option("sector_symmetries", false, rs_sequence, ""//
					"Select a range of identites built from sector symmetries.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	IdentityGenericSelection() {
		init();
	}
	virtual ~IdentityGenericSelection() {
	}

	virtual void read(const YAML::Node& n);
	virtual void print(YAML::Emitter& os) const;

	const std::map<std::string, std::list<RSFiniteGenericSelection> > & identities() const {
		return identities_;
	}

	bool empty() const;

protected:
	virtual void init();

private:
	std::map<std::string, std::list<RSFiniteGenericSelection> > identities_;

};

inline void operator>>(const YAML::Node& n, IdentityGenericSelection& r) {
	r.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye,
		const IdentityGenericSelection& r) {
	r.print(ye);
	return ye;
}

//
//
//

class ReductionGenericOptions: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("reduction_generic_options");
		s.set_short_description("Options and generic selections for reductions");
		s.set_long_description(""//
					"Options and generic selections of integrals and identities"
					" relevant for reductions");
		s.add_option("requested_solutions", false, ""//
					"sequence of rs_generic_selection", ""//
					"The (r,s) ranges of integrals for which solutions are"
					" required. If set to finite (r,s) ranges, some equations"
					" with leading integrals beyond these ranges will not be"
					" reduced based on heuristic rules. This might save some"
					" computation time. If you want a full reduction of all"
					" given identities, leave this option on its default value.");
		s.add_option("set_subsectors_to_zero", false, "boolean", "" //
					"Whether sub-sectors should be set to zero, see"
					" identity_generator_options.");
		s.add_option("map_on_coeff", false, "sequence of 2-element sequences",
					"Substitutions to perform in coefficients, see"
					" identity_generator_options.");
		s.add_option("discard_new_subsector_identities", false, "boolean", "" //
					"Whether new sub-sector results in the reduction should be"
					" discarded.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	ReductionGenericOptions() :
		set_subsectors_to_zero_(false), discard_new_subsector_identities_(false) {
		requested_solutions_.push_back(RSGenericSelection());
	}
	virtual ~ReductionGenericOptions() {
	}

	virtual void read(const YAML::Node& n);
	virtual void print(YAML::Emitter& os) const;

private:
	friend class ReduceSectors;
	friend class RunReduction;

private:
	/// range of integrals for which we would like to have reductions
	//  default is one full range
	std::list<RSGenericSelection> requested_solutions_;
	/// identities with leading integral not in this range are left untouched
	//  default is one full range
	std::list<RSGenericSelection> allowed_reductions_;

	bool set_subsectors_to_zero_;
	GiNaC::exmap map_on_coeff_;

	/// whether new identities for subsectors should be deleted right away
	bool discard_new_subsector_identities_;
};

inline void operator>>(const YAML::Node& n, ReductionGenericOptions& r) {
	r.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye,
		const ReductionGenericOptions& r) {
	r.print(ye);
	return ye;
}

} // namespace Reduze

#endif /* RSPOINT_H_ */
