/*  job_distributeexternal.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_distributeexternal.h"
#include "files.h"
#include "filedata.h"
#include "job_runreduction.h"
#include "yamlutils.h"

using namespace std;

namespace Reduze {

namespace {
JobProxy<DistributeExternal> dummy;
}

bool DistributeExternal::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	// this job leads to severe file I/O
	// for simplicity we just wait for everything else before
	in.insert(in.end(), outothers.begin(), outothers.end());
	// we don't exactly know yet what we will produce or update, assume worst
	set<Sector> secs = sector_selection_.find_sectors();
	for (set<Sector>::const_iterator s = secs.begin(); s != secs.end(); ++s) {
		string name = Files::instance()->get_filename_sectorreduction(*s);
		if (is_readable_file(name))
			out.push_back(name);
	}
	return true;
}

void DistributeExternal::init() {
}

std::string DistributeExternal::get_description() const {
	return "distribute external";
}

void DistributeExternal::run_serial() {

	list<string> explicitinput = equation_files_;

	while (true) {
	// iterate as long as external* or explicit input files present
    // (in all cases we saw so far, the first iteration was enough)

	Files* files = Files::instance();
	string tmpdir = files->get_tmp_directory() + "/external_dir";

	LOG("\n\nStep 1: determine external* and explicit input files\n");
    // input: reductions/external_*, explicit input files
	// output: reductions/external_merged
	list<string> external, eqns(explicitinput), no;
    set<Sector> secs = sector_selection_.find_sectors();
    for (set<Sector>::const_iterator s = secs.begin(); s != secs.end(); ++s) {
    	string n = files->get_filename_sectorexternal(*s);
    	if (is_readable_file(n)) {
    		LOG("  found file \"" << n << "\"");
    		external.push_back(n);
    	}
    }
    eqns.insert(eqns.end(), external.begin(), external.end());
    if (eqns.empty()) {
    	LOG("\n\nFinished the cleanup !");
    	return;
    }
    LOG("\nDetected external files to distribute !");

    LOG("\n\nStep 2: interreduce external* and explicit input files\n");
    string merged = files->get_reductions_directory() + "external_merged";
    RunReduction jobredfor(tmpdir, eqns, no, merged, reduzer_options_, "", no, no);
	jobredfor.set_conditional(false);
	jobredfor.run_serial();
    // delete previous external* files
	// this is necessary since reductions might generate new versions for them
	remove(external);
    explicitinput.clear();
    LOG("Generated file \"" << merged << "\"");

	LOG("\n\nStep 3: split output file according to sector of leading integral\n");
    // input: reductions/external_merged
	// output: tmp/external_dir/reduction_*
	make_directory(tmpdir);
	InFileEquations in(merged);
	EquationHLight eq;
	map<Sector, string> extrafiles;
	SectorSelection updatesel =
		SectorSelection::find_compact_recursive_selection(secs);
	in.get(eq);
	while (in) {
		if (eq.empty())
			continue;
		Sector currsec(eq.max_INT().get_sector());
		LOG("  found identities for " << currsec);
		string fname = files->get_filename_sectorreduction(currsec, tmpdir);
		OutFileEquations of(fname);
		do {
			of << eq;
		} while (in.get(eq) && eq.max_INT().get_sector() == currsec);
		of.finalize();
		//fsecs.push_back(currsec);
		updatesel.deselect_independents_.push_back(currsec);
		extrafiles[currsec] = fname;
	}

	LOG("\n\nStep 4: reduce allowed sectors where new eqns could possibly occur\n");
	// input: tmp/external_dir/reduction_*, prev reductions
    // output: reductions/reduction*
	set<Sector> update = updatesel.find_sectors();
	set<Sector> distributed;
	set<Sector>::const_iterator s;
	for (s = update.begin(); s != update.end(); ++s) {
		string out = Files::instance()->get_filename_sectorreduction(*s);
		// skip sector if no new info and no existing reduction
		if (extrafiles.find(*s) == extrafiles.end() && !is_readable_file(out))
			continue;
		LOG("\nClean up for sector " << *s << "\n");
		// crossed sectors
		if (s->get_uncrossed() != *s) {
			// solution1: just delete crossed reductions file
			// (leads to some dependency and workflow problems)
			// solution2: keep dirty files
			if (is_readable_file(out))
				remove(out);
			continue;
		}
		distributed.insert(*s); // remember we took care of these identities
		list<string> in, subst;
		SectorSelection recsel;
		recsel.select_recursively_.push_back(*s);
		set<Sector> rec = recsel.find_sectors();
		for (set<Sector>::const_iterator r = rec.begin(); r != rec.end(); ++r) {
			if (extrafiles.find(*r) == extrafiles.end())
				continue;
			if (*r == *s)
				in.push_back(extrafiles[*r]);
			else
				subst.push_back(extrafiles[*r]);
		}
		string outfor = Files::instance()->get_filename_sectorexternal(*s);
		if (is_readable_file(out))
			in.push_back(out);
		RunReduction job(*s, tmpdir, in, subst, out, outfor,
				ReductionGenericOptions(), reduzer_options_, "", no, no);
		job.subst_subsectors_ = false;
		job.run_serial();
	}
	// merge leftovers and old global subst to new global subst
	LOG("\n\nStep 5: append new crossed sector reductions to shared file\n");
	list<string> leftovers; // new global shared substitutions (crossed secs)
	map<Sector, string>::const_iterator e;
	for (e = extrafiles.begin(); e != extrafiles.end(); ++e)
		if (distributed.find(e->first) == distributed.end())
			leftovers.push_back(e->second);
	string shared = files->get_filename_sectorexternal_shared();
	if (is_readable_file(shared))
		leftovers.push_back(shared);
	if (!leftovers.empty()) {
		RunReduction job(tmpdir, leftovers, no, merged, reduzer_options_, "", no, no);
		job.run_serial();
		rename(merged, shared);
		LOG("Updated " << shared);
	} else {
		LOG("No shared (crossed) reductions found");
	}
	LOGX("Cleanup temporary files");
	remove_directory_with_files(tmpdir);
	remove(merged);
	}
}

}
