/*  job_analyzediagrams.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_analyzediagrams.h"
// \todo check the following headers
#include "functions.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "files.h"
#include "diagram.h"
#include "graph.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "int.h"

using namespace std;

namespace Reduze {

typedef std::pair<CanonicalLabel, CanonicalRelabeling> LabelPair;

// register job type at JobFactory
namespace {
JobProxy<AnalyzeDiagrams> dummy;
}

size_t AnalyzeDiagrams::get_graphs_of_diagrams(std::map<int, std::list<
		std::pair<Graph, Diagram> > >& diasbyloop) const {
	LOG("Loading diagrams and from file: " << qgraf_filename_);

	list<Diagram> alldias;
	read_diagrams(alldias, qgraf_filename_, names_);
	size_t num_dias = alldias.size();
	while (!alldias.empty()) {
		Diagram& dia = *alldias.begin();
		Graph graph = dia.get_Graph();
		//		if (minimize_graphs_by_twists_)
		//			graph.minimize_by_twists();
		int loops = dia.num_loops();
		list<pair<Graph, Diagram> >& graphlist = diasbyloop[loops];
		graphlist.push_back(make_pair(graph, dia));
		alldias.erase(alldias.begin());
	}
	LOG("Loaded " << num_dias << " diagrams.");
	return num_dias;
}

///* Function objects for STL sort() etc. */
//struct ex_is_less : public std::binary_function<ex, ex, bool> {
//	bool operator() (const ex &lh, const ex &rh) const { return lh.compare(rh) < 0; }
//};

struct propagator_is_less: public std::binary_function<GiNaC::ex, GiNaC::ex,
		bool> {
	bool operator()(const GiNaC::ex& lh, const GiNaC::ex& rh) {
		if (GiNaC::is_a<Propagator>(lh) && GiNaC::is_a<Propagator>(rh)) {
			const Propagator& lp = GiNaC::ex_to<Propagator>(lh);
			const Propagator& rp = GiNaC::ex_to<Propagator>(rh);
			if (lp.momentum().nops() != rp.momentum().nops()) {
				return lp.momentum().nops() < rp.momentum().nops();
			} else {
				ostringstream sl, sr;
				sl << lh;
				sr << rh;
				return string(sl.str()) < string(sr.str());
			}
		}
		return lh.compare(rh) < 0;
	}
};

void AnalyzeDiagrams::run_serial() {
	//const Kinematics* kin = Files::instance()->kinematics();

	LOG("Analyzing diagrams:");
	LOGX("creating/cleaning output_directory: " << output_directory_);
	if (is_readable_directory(output_directory_))
		remove_directory_with_files(output_directory_);
	make_directory(output_directory_);
	string od = output_directory_;
	if (od[-1] != '/')
		od += '/';
	string dia_dir = od + "diagrams/";
	string topo_dir = od + "topologies/";
	string top_topo_dir = od + "top_level_topologies/";
	LOGX("creating directories: " << dia_dir << ", " << topo_dir << ", " << top_topo_dir);
	make_directory(dia_dir);
	make_directory(topo_dir);
	make_directory(top_topo_dir);

	// read in the diagrams

	map<int, list<pair<Graph, Diagram> > > diasbyloop;
	size_t num_dias = get_graphs_of_diagrams(diasbyloop);
	if (num_dias == 0)
		return;

	int max_t = -1;
	// cache one graph per label (by t)
	LOG("Printing diagrams to " << dia_dir);
	LOG("Determining topologies (" << num_dias << " graphs)");
	LOG("  - contract multiple propagators");
	LOG("  - disconnect vacuum components");
	LOG("  - contract_bridges");
	map<int, map<CanonicalLabel, Graph> > clean_graph_by_t;
	int lauf = 0, num_to_twist = 0;
	map<int, list<pair<Graph, Diagram> > >::iterator it;
	map<CanonicalLabel, Diagram> dia_representant;
	for (it = diasbyloop.begin(); it != diasbyloop.end(); ++it) {
		list<pair<Graph, Diagram> >& dias = it->second;
		list<pair<Graph, Diagram> >::iterator g;
		for (g = dias.begin(); g != dias.end(); ++g) {
			++lauf;
			const Diagram& dia = g->second;
			Graph graph = g->first;
			ostringstream ss;
			ss << setfill('0') << setw(number_of_digits(num_dias)) << lauf;
			graph.print_dot(dia_dir + "diagram_" + ss.str() + "_name_"
					+ graph.name() + ".dot");
			graph.contract_multiple_internal_propagators();
			graph.contract_bridges();
			graph.disconnect_vacuum_components();
			int num_props = graph.num_internal_edges();
			max_t = std::max(max_t, num_props);
			LabelPair label = graph.find_canonical_label();
			if (clean_graph_by_t[num_props].find(label.first)
					== clean_graph_by_t[num_props].end()) {
				clean_graph_by_t[num_props][label.first] = graph;
				dia_representant.insert(make_pair(label.first, dia));
				++num_to_twist;
			}
		}
	}
	LOG("  - minimize topologies by twists (" << num_to_twist << " graphs)");
	int num_topos = 0;
	map<CanonicalLabel, Diagram> tmp_dia;
	dia_representant.swap(tmp_dia);
	map<int, map<CanonicalLabel, Graph> > tmp;
	clean_graph_by_t.swap(tmp);
	for (map<int, map<CanonicalLabel, Graph> >::iterator t = tmp.begin(); t
			!= tmp.end(); ++t) {
		map<CanonicalLabel, Graph>::iterator g;
		for (g = t->second.begin(); g != t->second.end(); ++g) {
			Graph& graph = g->second;
			map<CanonicalLabel, Diagram>::const_iterator d = tmp_dia.find(g->first);
			VERIFY(d != tmp_dia.end());
			const Diagram& dia = d->second;
			if (true)
				graph.minimize_by_twists();
			CanonicalLabel l = graph.find_canonical_label().first;
			ostringstream ss;
			++num_topos;
			ss << "topo_t" << t->first << "_" << setfill('0') << setw(
					number_of_digits(num_dias)) << num_topos;
			g->second.set_name(ss.str());
			if (clean_graph_by_t[t->first].insert(make_pair(l, graph)).second) {
				graph.print_dot(topo_dir + graph.name() + ".dot");
				dia_representant.insert(make_pair(l, dia));
			} else {
				--num_topos;
			}
		}
	}
	LOG("Found " << num_topos << " topologies.");

	LOG("Determine top level topologies");

	VERIFY(!clean_graph_by_t.empty() && max_t > 0);
	int num_top_topos = 0;
	map<int, map<CanonicalLabel, Graph> > tops_by_t;

	tops_by_t[max_t] = clean_graph_by_t[max_t];
	map<int, map<CanonicalLabel, Graph> >::reverse_iterator t;
	map<CanonicalLabel, Graph>::const_iterator s;
	for (t = clean_graph_by_t.rbegin(); t != clean_graph_by_t.rend();) {
		map<CanonicalLabel, Graph> contracted;
		contract_internal_edges(t->second, contracted);
		if (++t == clean_graph_by_t.rend())
			break;
		for (s = t->second.begin(); s != t->second.end(); ++s)
			if (contracted.find(s->first) == contracted.end())
				tops_by_t[t->first].insert(*s);
		t->second.insert(contracted.begin(), contracted.end());
	}
	list<Graph> tops;
	list<Diagram> top_level_dias;
	for (t = tops_by_t.rbegin(); t != tops_by_t.rend(); ++t) {
		for (s = t->second.begin(); s != t->second.end(); ++s) {
			tops.push_back(s->second);
			s->second.print_dot(top_topo_dir + s->second.name() + ".dot");
			map<CanonicalLabel, Diagram>::const_iterator d = dia_representant.find(
					s->first);
			VERIFY(d != dia_representant.end());
			top_level_dias.push_back(d->second);
			++num_top_topos;
		}
	}
	LOG("Found " << num_top_topos << " top level topologies.");

	LOG("Writing top evel diagrams to " << top_level_diagrams_file_);
	write_diagrams(top_level_dias, top_level_diagrams_file_);

	LOG("Propagators of top level diagrams:");
	for (list<Graph>::const_iterator t = tops.begin(); t != tops.end(); ++t) {
		LOG("\n" + t->name());
		//		int NoLM = t->loop_momenta().nops();
		//		int NoEM = kin->independent_external_momenta().nops();
		//		unsigned num_props_fam = (NoLM + 1) * NoLM;
		//		num_props_fam /= 2;
		//		num_props_fam += NoLM * NoEM;
		YAML::Emitter ye;
		ye << YAML::BeginSeq;
		vector<Propagator> p = t->get_propagators_with_loop_momenta();
		set<GiNaC::ex, propagator_is_less> props(p.begin(), p.end());
		for (set<GiNaC::ex, propagator_is_less>::const_iterator i =
				props.begin(); i != props.end(); ++i) {
			Propagator pi = GiNaC::ex_to<Propagator>(*i);
			ye << YAML::Flow << YAML::BeginSeq << pi.momentum()
					<< pi.squaredmass() << YAML::EndSeq;
		}
		ye << YAML::EndSeq;
		LOG(ye.c_str());
		//		VERIFY(p.size() <= num_props_fam);
		//		int num_aux_props = num_props_fam - p.size();
		//		GiNaC::exset props;
		//		props.insert(p.begin(), p.end());
		//		//		LOG("Propagators for topology " << s->second.name() << ":");
		//		//		LOG("  " << props);
		//		list<GiNaC::exmap> shifts;
		//		t->find_symmetry_shifts(shifts, kin);
		//		list<GiNaC::exmap>::const_iterator s;
		//		for (s = shifts.begin(); s != shifts.end(); ++s) {
		//
		//		}
	}

	LOG("");
	LOG("To generate postcript files type: (replace 'PROG' by e.g. 'dot' or 'neato')");
	LOG("  PROG -Tps -o diagrams.ps " << dia_dir << "diagram_*.dot");
	LOG("  PROG -Tps -o topologies.ps " << topo_dir << "topo_t*.dot");
	LOG("  PROG -Tps -o top_level_topologies.ps " << top_topo_dir << "topo_t*.dot");
}

bool AnalyzeDiagrams::find_dependencies(const set<string>& outothers, list<
		string>& in, list<string>& out, list<Job*>& auxjobs) {
	in.push_back(qgraf_filename_);
	return true;
}

std::string AnalyzeDiagrams::get_description() const {
	return "analyze diagrams from file " + short_filename(qgraf_filename_);
}

}
