/*
 * edge.cpp
 *
 *  Created on: Jul 23, 2016
 *      Author: cedric
 */

#include "edge.h"
#include "functions.h"
#include "yamlutils.h"

namespace Reduze {

// Edge

int Edge::opposite(int node) const {
	if (node == from)
		return to;
	else if (node == to)
		return from;
	else
		ABORT("node " << node << " not connected by this edge: " << *this);
	return 0; // suppress compiler warning
}

void Edge::reverse() {
	std::swap(from, to);
}

bool Edge::is_self_loop() const {
	return from == to;
}

bool Edge::is_between(int a, int b) const {
	return (a == from && b == to) || (a == to && b == from);
}

int Edge::compare_undirected(const Edge& other) const {
	int amin = std::min(from, to);
	int amax = std::max(from, to);
	int bmin = std::min(other.from, other.to);
	int bmax = std::max(other.from, other.to);
	if (amin != bmin)
		return (amin < bmin ? -1 : 1);
	else if (amax != bmax)
		return (amax < bmax ? -1 : 1);
	else
		return 0;
}

bool Edge::operator<(const Edge& other) const {
	ASSERT(id != other.id || (from == other.from && to == other.to));
	return id < other.id;
}

bool Edge::operator==(const Edge& other) const {
	ASSERT(id != other.id || (from == other.from && to == other.to));
	return id == other.id;
}

std::ostream& operator<<(std::ostream& os, const Edge& e) {
	return os << "(" << e.from << "," << e.to << ";" << e.id << ")";
}

// colored multi edge

colored_multi_edge::colored_multi_edge(int num_colors) :
		num_edges_of_color_(std::vector<int>(num_colors, 0)) {
}

colored_multi_edge::colored_multi_edge(const YAML::Node& yn) {
	using namespace YAML;
	if (yn.Type() != NodeType::Sequence)
		throw std::runtime_error("expected a sequence " + position_info(yn));
	std::vector<int> col;
	yn >> col;
	num_edges_of_color_.swap(col);
}

bool colored_multi_edge::is_zero() const {
	for (unsigned i = 0; i < num_edges_of_color_.size(); ++i)
		if (num_edges_of_color_[i] != 0)
			return false;
	return true;
}

void colored_multi_edge::increment_at(int pos) {
	ASSERT(pos >= 0 && pos < (int ) num_edges_of_color_.size());
	++(num_edges_of_color_[pos]);
}

colored_multi_edge& colored_multi_edge::operator+=(
		const colored_multi_edge& other) {
	ASSERT(num_edges_of_color_.size() == other.num_edges_of_color_.size());
	for (unsigned i = 0; i < num_edges_of_color_.size(); ++i)
		num_edges_of_color_[i] += other.num_edges_of_color_[i];
	return *this;
}

/*
 const colored_multi_edge colored_multi_edge::operator-() const {
 colored_multi_edge res = *this;
 for (unsigned i = 0; i < num_edges_of_color_.size(); ++i)
 res.num_edges_of_color_[i] = -res.num_edges_of_color_[i];
 return res;
 }

 colored_multi_edge& colored_multi_edge::operator-=(
 const colored_multi_edge& other) {
 *this += -other;
 return *this;
 }

 const colored_multi_edge colored_multi_edge::operator+(
 const colored_multi_edge& other) const {
 colored_multi_edge result = *this;
 result += other;
 return result;
 }

 const colored_multi_edge colored_multi_edge::operator-(
 const colored_multi_edge& other) const {
 colored_multi_edge result = *this;
 result -= other;
 return result;
 }
 */

bool colored_multi_edge::operator<(const colored_multi_edge& other) const {
	ASSERT(num_edges_of_color_.size() == other.num_edges_of_color_.size());
	return num_edges_of_color_ < other.num_edges_of_color_;
}

bool colored_multi_edge::operator==(const colored_multi_edge& other) const {
	ASSERT(num_edges_of_color_.size() == other.num_edges_of_color_.size());
	return (num_edges_of_color_ == other.num_edges_of_color_);
}

bool colored_multi_edge::operator!=(const colored_multi_edge& other) const {
	return !(*this == other);
}

YAML::Emitter& operator<<(YAML::Emitter& ye, const colored_multi_edge& cme) {
	ye << YAML::Flow << cme.num_edges_of_color_;
	return ye;
}

std::ostream& operator<<(std::ostream& os, const colored_multi_edge& cme) {
	YAML::Emitter ye;
	ye << cme;
	os << ye.c_str();
	return os;
}

// edge_attributes

int edge_attributes::compare(const edge_attributes& other) const {

	// compare masses lexicographically
	int m1 = masses_.nops();
	int m2 = other.masses_.nops();
	GiNaC::lst::const_iterator l1 = masses_.begin(), l2 = other.masses_.begin();
	for (; l1 != masses_.end() && l2 != other.masses_.end(); ++l1, ++l2) {
		if (!l1->is_equal(*l2)) {
			mass_is_less mil;
			return ((mil(*l1, *l2)) ? -1 : 1);
		}
	}
	if (m1 != m2)
		return m1 - m2;

	// compare propagator types lexicographically
	if (propagator_types_ != other.propagator_types_)
		return ((propagator_types_ < other.propagator_types_) ? -1 : 1);

	return 0;
}

bool edge_attributes::operator<(const edge_attributes& other) const {
	return (this->compare(other) < 0);
}
bool edge_attributes::operator==(const edge_attributes& other) const {
	return (this->compare(other) == 0);
}
bool edge_attributes::operator!=(const edge_attributes& other) const {
	return !(*this == other);
}

std::ostream& operator<<(std::ostream& os, const edge_attributes& ea) {
	YAML::Emitter ye;
	ye << ea;
	os << ye.c_str();
	return os;
}

YAML::Emitter& operator<<(YAML::Emitter& ye, const edge_attributes& ea) {
	using namespace YAML;
	ye << BeginMap;
	ye << Key << "masses" << Value << Flow << ea.masses_;
	ye << Key << "propagator_types" << Value << Flow << ea.propagator_types_;
	ye << EndMap;
	return ye;
}

void edge_attributes::read(const YAML::Node& yn, const GiNaC::lst& symbs) {
	using namespace YAML;
	using namespace std;
	if (yn.Type() != NodeType::Map || yn.size() != 4)
		throw runtime_error("expected a 4-element map " + position_info(yn));

	const Node& mass_node = yn["masses"];
	GiNaC::lst masses;
	if (mass_node.Type() != NodeType::Sequence)
		throw runtime_error("expected a sequence " + position_info(mass_node));
	Reduze::read<GiNaC::ex>(mass_node, masses, symbs);
	masses_ = masses;

	const Node& prop_node = yn["propagator_types"];
	if (prop_node.Type() != NodeType::Sequence)
		throw runtime_error("expected a sequence " + position_info(prop_node));
	set<int> props;
	prop_node >> props;
	propagator_types_.swap(props);
}

void edge_attributes::swap(edge_attributes& other) {
	propagator_types_.swap(other.propagator_types_);
	GiNaC::lst tmp = masses_;
	masses_ = other.masses_;
	other.masses_ = tmp;
}

} // namespace Reduze
