/*  diagram.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef DIAGRAM_H_
#define DIAGRAM_H_

#include "functions.h"
#include "kinematics.h"
#include "sector.h"
#include "yamlconfigurable.h"

namespace Reduze {

/**
 * A 'Field' is defined through a name, commutator sign, a momentum,
 * a mass and an index nummer.
 *
 * A 'FieldTuple' is a collection of n fields and represents a
 * basic building block to set up the analytical expression for
 * a Feynman diagram. For propagators: n = 2, for external particles:
 * n = 1, for Vertices: n = 3, 4, ...
 *
 * A Feynman Diagram is considered as a collection of "FieldTuples"
 * which are connected by equal field indices.
 *
 */

class FeynmanRules;
class Graph;
class Amplitude;
class FieldTuple;

/// template function to check the derived type (external, propagator, vertex) of the FieldTuple
template<class T> inline bool ft_is_a(const FieldTuple* f) {
	return dynamic_cast<const T*> (f) != 0;
}
/// template function to convert a FieldTuple to a derived type
template<class T> inline const T* ft_to(const FieldTuple* f) {
	ASSERT(ft_is_a<T>(f));
	return static_cast<const T*> (f);
}

/// Type of a field as used in QGRAF
class FieldType {
public:
	enum {
		undef = 0, incoming = 1, outgoing = 2, internal = 3
	};
};

/// class to store a field: name, commutator sign, momentum, mass, field index
class Field {
public:
	Field();
	virtual ~Field();

	// setters
	void set_field_index(int new_field_index);

	// getters
	const std::string& name() const;
	int field_type() const;
	bool anti_commuting() const;
	GiNaC::ex momentum() const;
	GiNaC::ex mass() const;
	int field_index() const;

	/// applies the map on the momentum
	void substitute_momenta(const GiNaC::exmap& m, unsigned options = 0);
	/// applies the map on the mass
	void substitute_mass(const GiNaC::exmap& m, unsigned options = 0);

	// input/output
	void read(const YAML::Node&, const GiNaC::lst& symbols);
	void print(YAML::Emitter& os) const;
	void read_stream(std::istream& s, const GiNaC::lst& symbols);
	void write_stream(std::ostream& s) const;

private:
	/// name of the field
	std::string name_;
	/// field type takes one of the values of FieldType (incoming, outgoing, internal)
	int field_type_;
	/// whether the field anti-commutes or not
	bool anti_commuting_;
	/// momentum, mass
	GiNaC::ex momentum_, mass_;
	/// field index as in QGRAF
	int field_index_;
};

inline const std::string& Field::name() const {
	return name_;
}
inline int Field::field_type() const {
	return field_type_;
}
inline bool Field::anti_commuting() const {
	return anti_commuting_;
}
inline GiNaC::ex Field::momentum() const {
	return momentum_;
}
inline GiNaC::ex Field::mass() const {
	return mass_;
}
inline int Field::field_index() const {
	return field_index_;
}

//
//
//

/// tupel of fields: base class for external fields, propagators, vertices
class FieldTuple {
public:
	FieldTuple();
	virtual ~FieldTuple();
	virtual FieldTuple* clone() const = 0;

	/// set the symbols
	void set_symbols(const GiNaC::lst& symbols);

	/// applies the map on all momenta of all fields
	virtual void
	substitute_momenta(const GiNaC::exmap& m, unsigned options = 0);
	/// applies the map on the mass of all fields
	void substitute_mass(const GiNaC::exmap& m, unsigned options = 0);
	/// replaces the field indices old->new
	void replace_field_indices(const std::map<int, int>& old_to_new);

	/// get the symbols
	const GiNaC::lst& symbols() const;
	/// get the Feynman rule identifier
	const std::string& fr_id() const;
	/// number of fields
	int size() const;
	/// get the momentum of field number i (i >=1 )
	GiNaC::ex momentum(int i) const;
	/// get the mass of field number i (i >=1 )
	GiNaC::ex mass(int i) const;
	/// get the field index of field number i (i >=1 )
	int field_index(int i) const;

	/// returns true if the FieldTuple contains a anti-commuting field
	bool has_anti_commuting_field() const;

	/// returns the feynman rule value for this object (external field, propagator, vertex)
	/* the map 'indices' contains the symbols to be used for the indices if available, otherwise
	 * they will be created and inserted into the map
	 */
	virtual GiNaC::ex feynman_rule_value(const FeynmanRules* fr, std::map<
			std::string, GiNaC::symbol>& indices) const;

	// input/output
	virtual void read_stream(std::istream& s);
	virtual void write_stream(std::ostream& s) const;

protected:
	/// Feynman rule identifier
	std::string fr_id_;
	/// the fields
	std::vector<Field> fields_;
	/// masses and momenta
	GiNaC::lst symbols_;

private:
	/// returns the map how to replace the momenta, masses and indices in the Feynman rules by their actual values
	/** the necessary indices are taken from the map 'indices' or if not present already,
	 * new created and inserted into the map
	 */
	GiNaC::exmap mom_mass_indices_map(const FeynmanRules* fr, std::map<
			std::string, GiNaC::symbol>& indices) const;
	friend void operator>>(const YAML::Node& n, FieldTuple& fn);
	friend YAML::Emitter& operator<<(YAML::Emitter& os, const FieldTuple& fn);
};

inline const std::string& FieldTuple::fr_id() const {
	return fr_id_;
}
inline int FieldTuple::size() const {
	return fields_.size();
}

//
//
//

/// class to represent an external field
class ExternalField: public FieldTuple, public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("external_field");
		s.set_short_description("A class representing an external field.");
		s.set_long_description(""//
					"A class representing an external field. \\todo more doc");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	ExternalField();
	virtual ~ExternalField();
	virtual ExternalField* clone() const;

	void set_polarization_orthogonal_vector(const GiNaC::ex& ortho) {
		polarization_orthogonal_vector_ = ortho;
	}
	const GiNaC::ex& polarization_orthogonal_vector() const {
		return polarization_orthogonal_vector_;
	}

	virtual void
	substitute_momenta(const GiNaC::exmap& m, unsigned options = 0);

	inline int from_vertex() const {
		return from_vertex_;
	}
	inline int to_vertex() const {
		return to_vertex_;
	}

	/// whether the external leg is incoming or outgoing
	bool is_incoming() const;
	/// whether the external leg is the beginning of a fermion line
	bool begin_fermion_line(const FeynmanRules* fr) const;
	/// whether the external leg is the ending of a fermion line
	bool end_fermion_line(const FeynmanRules* fr) const;

	// input/output
	virtual void read(const YAML::Node&);
	virtual void print(YAML::Emitter& os) const;
	virtual void read_stream(std::istream& stream);
	virtual void write_stream(std::ostream& s) const;

private:
	int from_vertex_, to_vertex_;
	/// polarization orthogonal vector
	GiNaC::ex polarization_orthogonal_vector_;
};

inline void operator>>(const YAML::Node& n, ExternalField& f) {
	f.read(n);
}
inline YAML::Emitter& operator<<(YAML::Emitter& ye, const ExternalField& f) {
	f.print(ye);
	return ye;
}

/// class to represent a line (2 fields)
class PropagatorFields: public FieldTuple, public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("propagator_fields");
		s.set_short_description(
				"A class representing the two fields of a propagator");
		s.set_long_description(""//
					"A class representing the two fields of a propagator \\todo more doc");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	PropagatorFields();
	virtual ~PropagatorFields();
	virtual PropagatorFields* clone() const;
	inline GiNaC::ex mass() const {
		return fields_[1].mass();
	}
	inline int from_vertex() const {
		return from_vertex_;
	}
	inline int to_vertex() const {
		return to_vertex_;
	}

	// input/output
	virtual void read(const YAML::Node&);
	virtual void print(YAML::Emitter& os) const;
	virtual void read_stream(std::istream& stream);
	virtual void write_stream(std::ostream& s) const;

private:
	/// from- and to-node, direction of the momentum flow (and particle flow)
	int from_vertex_, to_vertex_;
};

inline void operator>>(const YAML::Node& n, PropagatorFields& f) {
	f.read(n);
}
inline YAML::Emitter& operator<<(YAML::Emitter& ye, const PropagatorFields& f) {
	f.print(ye);
	return ye;
}

//
//
//

/// class to represent a Node (n fields)
class VertexFields: public FieldTuple, public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("vertex_fields");
		s.set_short_description(
				"A class representing the fields around a vertex");
		s.set_long_description(""//
					"A class representing the fields around a vertex \\todo more doc");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	VertexFields();
	virtual ~VertexFields();
	virtual VertexFields* clone() const;

	// input/output
	virtual void read(const YAML::Node&);
	virtual void print(YAML::Emitter& os) const;
	virtual void read_stream(std::istream& stream);
	virtual void write_stream(std::ostream& s) const;
};

inline void operator>>(const YAML::Node& n, VertexFields& f) {
	f.read(n);
}
inline YAML::Emitter& operator<<(YAML::Emitter& ye, const VertexFields& f) {
	f.print(ye);
	return ye;
}

//
//
//

/// class for diagrams
class Diagram: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("diagram");
		s.set_short_description("The diagram class.");
		s.set_long_description(""//
					"The diagram class.");
		s.add_option("process_name", false, "string", ""//
					"The name of the process the diagram belongs to.");
		s.add_option("name", true, "string", ""//
					"The name of the diagram.");
		s.add_option("num_loops", true, "integer", ""//
					"The number of loops.");
		s.add_option("loop_momenta", false, "sequence of strings", ""//
					"The loop momenta.");
		s.add_option("num_legs_in", true, "integer", ""//
					"The number of incoming legs.");
		s.add_option("num_legs_out", true, "integer", ""//
					"The number of outgoing legs.");
		s.add_option("num_propagators", true, "integer", ""//
					"The number of propagators.");
		s.add_option("num_vertices", true, "integer", ""//
					"The number of vertices.");
		s.add_option("external_legs", false, "sequence of external_field", ""//
					"The externel legs.");
		s.add_option("propagators", false, "sequence of propagator_fields", ""//
					"The propagators fields.");
		s.add_option("vertices", false, "sequence of vertex_fields", ""//
					"The vertex fields.");
		s.add_option("sector", false, "sequence", ""//
					"The sector of the diagram.");
		s.add_option("symmetry_factor", true, "numeric", ""//
					"The symmetry factor of the diagram.");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	// constructors

	/// construct Diagram with given kinematics
	Diagram(const Kinematics* kin);
	/// deep copy
	Diagram(const Diagram& other);
	virtual ~Diagram();
	Diagram& operator=(const Diagram& other);
	void swap(Diagram& other);

	// setters
	/// set the name of the diagram
	void set_name(const std::string& s);
	// /// sets a new kinematics (sets also the symbols)
	// void set_kinematics(const std::string& kin);
	/// set the process name
	void set_process_name(const std::string& s);
	/// set the loop momenta
	void set_loop_momenta(const GiNaC::lst& loop_momenta);
	/// set sector the diagram matches
	void set_sector(const Sector& s);
	/// sets the polarization_orthogonal_vector of the field tuple, aborts if it is not an external field
	void set_polarization_orthogonal_vector(int tuple, const GiNaC::ex& ortho);

	// getters
	const Kinematics* kinematics() const {
		return kinematics_;
	}
	const std::string& name() const;
	const std::string& process_name() const;
	int num_loops() const;
	int num_external_legs() const;
	GiNaC::ex symmetry_factor() const;
	const GiNaC::lst& loop_momenta() const;
	/// returns sector the diagram has been matched to, throws if none defined
	const Sector& sector() const;

	// substitutions

	/// applies the map on all momenta
	void
	substitute_momenta(const GiNaC::exmap& substitutions, unsigned opt = 0);
	/// applies the map on all masses
	void substitute_mass(const GiNaC::exmap& substitutions, unsigned opt = 0);

	std::map<int, int> find_following_fermion(const FeynmanRules* fr) const;
	/// returns the sorted ids of the open and closed fermion lines and the rest
	/** fermion line ids are sorted in direction of the charge flux */
	void find_sorted_ids(std::list<std::list<int> >& open_fl, std::list<
			std::list<int> >& closed_fl, std::list<int>& rest,
			const FeynmanRules* fr) const;

	/// get the Feynman rules for the FieldTuple with ID 'id'
	GiNaC::ex feynman_rule_value(int id, const FeynmanRules* fr, std::map<
			std::string, GiNaC::symbol>& indices) const;

	/// get the Graph
	Graph get_Graph() const;
	/// get the Amplitude, sets also the sector if available
	Amplitude get_amplitude(const FeynmanRules* fr) const;

	/// returns the external field tuples which start a fermionic line
	std::set<int> begin_fermion_lines(const FeynmanRules* fr) const;

	/// returns the field tuple of a vertex
	int tuple_of_vertex(int vertex) const;

	// input/output
	virtual void read(const YAML::Node&);
	virtual void print(YAML::Emitter& os) const;
	void read_stream(std::istream& s);
	void write_stream(std::ostream& s) const;

private:
	/// identifier for the process
	std::string process_name_;
	/// identifier for a diagram
	std::string name_;
	/// number of loops, legs, propagators and vertices
	int num_loops_, num_legs_in_, num_legs_out_, num_propagators_,
			num_vertices_;
	/// the field tuples
	std::vector<FieldTuple*> field_tuples_;
	/// symmetry factor of the diagram
	GiNaC::ex symmetry_factor_;
	/// zero or one-element list of possible mappings to sectors
	/** list is a workaround to avoid default Sector or pointer **/
	std::list<Sector> sector_;

private:
	Diagram();
	/// sets the member: node_by_vertex_
	void init();
	const FieldTuple* field_tuple(int node) const;
	/// inserts a FieldTuple, returns the id
	int insert(FieldTuple* field_tuple);

	/// map of vertex number -> field tuple number
	std::map<int, int> tuple_by_vertex_;

	const Kinematics* kinematics_;
	/// the local loop momenta of this diagram
	// they have to be set manually before a diagram (with loops) can be read
	GiNaC::lst loop_momenta_;
	/// external momenta and kinematics
	// set by constructor (for caching)
	GiNaC::lst symbols_without_loop_momenta_;
	/// external momenta, kinematics and (if already set) also the loop momenta
	GiNaC::lst symbols_;

	void clear_field_tuples();
	/// replaces the negative field indices by the negative leg number
	void rename_negative_field_indices();

	friend std::ostream& operator<<(std::ostream& os, const Diagram& dia);
	friend class YAMLProxy<Diagram> ;
};

inline void operator>>(const YAML::Node& n, Diagram& f) {
	f.read(n);
}
inline YAML::Emitter& operator<<(YAML::Emitter& ye, const Diagram& f) {
	f.print(ye);
	return ye;
}

inline const std::string& Diagram::name() const {
	return name_;
}
inline const std::string& Diagram::process_name() const {
	return process_name_;
}
inline int Diagram::num_loops() const {
	return num_loops_;
}
inline int Diagram::num_external_legs() const {
	return num_legs_in_ + num_legs_out_;
}

inline GiNaC::ex Diagram::symmetry_factor() const {
	return symmetry_factor_;
}
inline const GiNaC::lst& Diagram::loop_momenta() const {
	return loop_momenta_;
}

//
//
//


class qgraf_globals {
public:
	qgraf_globals() :
		num_loops_(-1) {
	}

	const GiNaC::lst& loop_momenta() const {
		return loop_momenta_;
	}

	//	GiNaC::lst get_ext_momenta() const {
	//		return ext_momenta_;
	//	}

	const std::string& process_name() const {
		return process_name_;
	}
private:
	std::list<std::string> globals_;

	int num_loops_;
	GiNaC::lst loop_momenta_, ext_momenta_;
	std::string process_name_;

	void set_num_loops();
	void set_loop_momenta();
	//void set_ext_momenta();
	void set_process_name();
	friend void operator >>(const YAML::Node& node, qgraf_globals& qgraf);
};

// functions

/// reads diagrams from a file
/** uses default values from a qgraf_globals section if present, loads default kinematics **/
void read_diagrams(std::list<Diagram>& diagrams, const std::string& fn,
		const std::set<std::string>& names = std::set<std::string>(),
		const Kinematics* kin = 0);
/// reads the diagram at position 'pos' from a file
/** avoids parsing unwanted diagrams, loads default kinematics **/
Diagram read_diagram(const std::string& fn, int pos);
/// writes diagrams to a file
void write_diagrams(const std::list<Diagram>& diagrams, const std::string& fn);

/// reads diagrams from a file
/** uses default values from a qgraf_globals section if present, loads default kinematics **/
void read_diagrams_yaml(std::list<Diagram>& diagrams, const std::string& fn,
		const std::set<std::string>& names = std::set<std::string>(),
		const Kinematics* kin = 0);
/// reads the diagram at position 'pos' from a file
/** avoids parsing unwanted diagrams, loads default kinematics **/
Diagram read_diagram_yaml(const std::string& fn, int pos);
/// writes diagrams to a file
void write_diagrams_yaml(const std::list<Diagram>& diagrams,
		const std::string& fn);

/// reads diagrams from a file
/** uses default values from a qgraf_globals section if present, loads default kinematics **/
void read_diagrams_stream(std::list<Diagram>& diagrams, const std::string& fn,
		const std::set<std::string>& names = std::set<std::string>(),
		const Kinematics* kin = 0);
/// reads the diagram at position 'pos' from a file
/** avoids parsing unwanted diagrams, loads default kinematics **/
Diagram read_diagram_stream(const std::string& fn, int pos);
/// writes diagrams to a file
void write_diagrams_stream(const std::list<Diagram>& diagrams,
		const std::string& fn);

/** read the diagrams from the qgraf file. If names are not empty then only selected
 ** diagrams will be read in. the diagrams are read in with the given kinematics,
 ** empty kinematics means default kinematics. The graphs can be manipulated with
 ** the options amputate_external_legs, join_external_nodes, minimize_graphs_by_twists
 **/
size_t find_graphs_of_diagrams( //
		const std::string& qgraf_filename, //
		const std::set<std::string>& names, //
		const std::string& kinematics, //
		bool amputate_external_legs, //
		bool join_external_nodes, //
		bool minimize_graphs_by_twists, //
		std::list<std::pair<Graph, Diagram> >& graph_dia_pairs);

/** graphs/diagrams by loop **/
size_t find_graphs_of_diagrams(const std::string& qgraf_filename,
		const std::set<std::string>& names, //
		const std::string& kinematics, //
		bool amputate_external_legs, //
		bool join_external_nodes, //
		bool minimize_graphs_by_twists, std::map<int, std::list<std::pair<
				Graph, Diagram> > >& graph_dia_pairs_by_loop);

/** graphs/diagrams by number of propagators (t) and by loop
 ** If rename_diagrams is set to true then the N diagrams are renamed
 ** by the string "1" to "N"; their original name is the name of the
 ** corresponding graph **/
size_t find_graphs_of_diagrams(
		const std::string& qgraf_filename,
		const std::set<std::string>& names, //
		const std::string& kinematics, //
		bool amputate_external_legs, //
		bool join_external_nodes, //
		bool minimize_graphs_by_twists, std::map<int, std::map<int, std::list<
				std::pair<Graph, Diagram> > > >& graph_dia_pairs_by_t_by_loop,
		bool rename_diagrams);

size_t find_top_level_graphs(const std::map<int, std::map<int, std::list<
		std::pair<Graph, Diagram> > > >& graph_dia_pairs_by_t_by_loop);

} // namespace Reduze

#endif /* DIAGRAM_H_ */
