/*  dfspalmtree.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "dfspalmtree.h"
#include "functions.h"

using namespace std;

namespace Reduze {

DFSPalmTree::DFSPalmTree(const Topology& t, int start_node, int max_node_id,
		int max_edge_id, int max_orig_edge_id) ://
	num_numbers_(0),//
			root_(start_node),//
			pos_(max_node_id + 1, -1),//
			index_of_pos_(t.num_nodes(), -1),//
			lowpt1_(max_node_id + 1),//
			lowpt2_(max_node_id + 1),//
			nd_(max_node_id + 1), //
			max_edge_id_(max_edge_id),//
			max_orig_edge_id_(max_orig_edge_id), //
			outgoing_(max_node_id + 1),//
			edge_type_(max_edge_id + 1, NotVisited),//
			is_first_edge_of_path_(max_edge_id + 1, false)//
{
	//VERIFY(t.num_edges() >= 2); // can't be biconnected otherwise
	if (t.num_edges() < 2)
		if (t.nodes().empty() || *t.nodes().begin() < 0//
				|| t.edges().empty() || t.edges().begin()->first < 0)
			ABORT("graph node or edge numbers don't match required convention");
	build_DFS_palm_tree(t, start_node);
	if (num_numbers_ != (int) t.num_nodes())
		ABORT("can't handle disconnected graph");
#ifdef DEBUG
	LOGXX("\nbuilt palm tree with " << t.num_nodes() << " nodes and "
			<< t.num_edges() << " edges" << endl);
	LOGXX("root=" << root_<< endl);
	LOGXX("original adjacencies:");
	for (int i = 0; i <= max_node_id; ++i) {
		if (! outgoing_[i].empty()) {
			LOGNXX(i << ":  ");
			list<Edge>::const_iterator e;
			for ( e = outgoing_[i].begin(); e != outgoing_[i].end(); ++e)
			LOGNXX(" " << *e);
			LOGXX("");
		}
	}
	LOGXX("");
	LOGXX("node\tpos\tlowpt1\tlowpt2\tnd");
	const set<int>& ns = t.nodes();
	for (set<int>::const_iterator n = ns.begin(); n != ns.end(); ++n)
	LOGXX(*n << "\t" << pos_[*n] << "\t" << lowpt1_[*n] << "\t"<< lowpt2_[*n] << "\t" << nd_[*n]);
	LOGXX("");
	print_dot("dfstree.dot");
#endif // DEBUG
}

void DFSPalmTree::update_min(int& lo1, int& lo2, int cand) {
	if (pos_[cand] < pos_[lo1]) {
		lo2 = lo1;
		lo1 = cand;
	} else if (pos_[cand] > pos_[lo1] && pos_[cand] < pos_[lo2]) {
		lo2 = cand;
	}
}

void DFSPalmTree::build_DFS_palm_tree(const Topology& topo, int v /*node*/) {
	pos_[v] = num_numbers_++;
	index_of_pos_[pos_[v]] = v;
	lowpt1_[v] = lowpt2_[v] = v;
	nd_[v] = 1;

	int num_chops = 0; // number of components chopped off current node
	const list<Edge>& el = topo.edges_of_node(v);
	for (list<Edge>::const_iterator e = el.begin(); e != el.end(); ++e) {
		if (edge_type_[e->id] != NotVisited) // need one visit
			continue;
		int w = e->opposite(v); // son of v
		// insert new edge into tree (directed, only inserted for origin)
		outgoing_[v].push_back(*e);
		if (pos_[w] < 0) { // w is new node (edge: tree arc)
			edge_type_[e->id] = TreeArc;
			build_DFS_palm_tree(topo, w);
			update_min(lowpt1_[v], lowpt2_[v], lowpt1_[w]);
			update_min(lowpt1_[v], lowpt2_[v], lowpt2_[w]);
			nd_[v] += nd_[w];
		} else { // w is old node (edge: palm frond)
			edge_type_[e->id] = PalmFrond;
			update_min(lowpt1_[v], lowpt2_[v], w);
		}
		if (((w == v) // self-loop
				|| (edge_type_[e->id] == TreeArc && lowpt1_[w] == w) // single edge
				|| (edge_type_[e->id] == TreeArc && lowpt1_[w] == v) // loop bcc
		) && !(v == root_ && num_chops == 0)) // found articulation point
			ABORT("can't handle graph since it is not biconnected");
	}
}

size_t DFSPalmTree::num_nodes() const {
	return index_of_pos_.size();
}

int DFSPalmTree::max_node_id() const {
	return pos_.size() - 1;
}

Edge DFSPalmTree::directed_edge(const Edge& e) const {
	int lo = (pos_[e.from] < pos_[e.to] ? e.from : e.to);
	int hi = (pos_[e.from] < pos_[e.to] ? e.to : e.from);
	if (edge_type_[e.id] == PalmFrond)
		return Edge(hi, lo, e.id);
	else
		// TreeArc
		return Edge(lo, hi, e.id);
}

void DFSPalmTree::construct_acceptable_adjacencies() {
	vector<list<Edge> > bucket(3 * index_of_pos_.size());
	vector<list<Edge> >::const_iterator el;
	list<Edge>::const_iterator e;
	for (el = outgoing_.begin(); el != outgoing_.end(); ++el) {
		for (e = el->begin(); e != el->end(); ++e) {
			Edge de = directed_edge(*e);
			// modified sorting function according to Gutwenger, Mutzel
			int phi;
			if (edge_type_[e->id] == PalmFrond)
				phi = 3 * pos_[de.to] + 1;
			else if (pos_[lowpt2_[de.to]] < pos_[de.from])
				phi = 3 * pos_[lowpt1_[de.to]];
			else
				phi = 3 * pos_[lowpt1_[de.to]] + 2;
			LOGXX(de << " has phi=" << phi);
			bucket[phi].push_back(*e);
		}
	}
	outgoing_.assign(max_node_id() + 1, list<Edge> ());
	for (el = bucket.begin(); el != bucket.end(); ++el) {
		for (e = el->begin(); e != el->end(); ++e) {
			Edge de = directed_edge(*e);
			outgoing_[de.from].push_back(*e);
		}
	}
#ifdef DEBUG
	LOGXX("\nacceptable adjacencies:");
	for (int i = 0; i <= max_node_id(); ++i) {
		if (! outgoing_[i].empty()) {
			LOGNXX(i << ":  ");
			list<Edge>::const_iterator e;
			for ( e = outgoing_[i].begin(); e != outgoing_[i].end(); ++e)
			LOGNXX(" " << *e);
			LOGXX("");
		}
	}
#endif
}

void DFSPalmTree::find_paths(int v /*node*/, //
		int& m /*last num assigned*/, //
		bool& is_path_started) {
	pos_[v] = m - nd_[v];
	index_of_pos_[pos_[v]] = v;
	const list<Edge>& el = outgoing_[v];
	for (list<Edge>::const_iterator e = el.begin(); e != el.end(); ++e) {
		if (!is_path_started) {
			is_path_started = true;
			paths_.push_back(list<Edge> ());
			is_first_edge_of_path_[e->id] = true;
		}
		ASSERT(!paths_.empty());
		paths_.back().push_back(*e);
		int w = e->opposite(v); // son of v
		incoming_[w].push_back(*e);
		if (edge_type_[e->id] == TreeArc) {
			find_paths(w, m, is_path_started);
			--m;
		} else { // PalmFrond
			ASSERT(edge_type_[e->id] == PalmFrond);
			is_path_started = false;
		}
	}
}

const list<list<Edge> >& DFSPalmTree::find_paths() {
	construct_acceptable_adjacencies();
	pos_.assign(max_node_id() + 1, -1);
	incoming_.assign(max_node_id() + 1, list<Edge> ());
	paths_.clear();
	int m = num_nodes();
	bool is_path_started = false;
	find_paths(root_, m, is_path_started);
	return paths_;
}

DFSPalmTree::Triple::Triple() :
	h(-1), a(-1), b(-1) {
}

DFSPalmTree::Triple::Triple(int h, int a, int b) :
	h(h), a(a), b(b) {
}

bool DFSPalmTree::Triple::is_valid() const {
	return h >= 0 && a >= 0 && b >= 0;
}

int DFSPalmTree::first_son(int v) const {
	const list<Edge>& es = outgoing_[v];
	return es.empty() ? -1 : es.front().opposite(v);
}

const Edge& DFSPalmTree::father_arc(int v) const {
	const list<Edge>& el = incoming_[v];
	ASSERT(v != root_);
	ASSERT(!el.empty() && edge_type_[el.front().id] == TreeArc);
	ASSERT(el.front().opposite(v) != v);
	return el.front();
}

int DFSPalmTree::father(int v) const {
	return father_arc(v).opposite(v);
}

int DFSPalmTree::highpos(int w) const {
	// note: we could just take first frond if we would ensure
	// new_vedges() inserts new edges at the right position
	int hp = -1;
	list<Edge>::const_iterator f;
	for (f = incoming_[w].begin(); f != incoming_[w].end(); ++f)
		if (edge_type_[f->id] == PalmFrond)
			hp = max(hp, pos_[f->opposite(w)]);
	return hp;
}

void DFSPalmTree::print_estack(const stack<Edge>& estack) {
	LOGXX("estack:");
	stack<Edge> es = estack;
	while (!es.empty()) {
		LOGXX(es.top());
		es.pop();
	}
}

void DFSPalmTree::print_tstack(const stack<Triple>& tstack) {
	LOGXX("tstack:");
	stack<Triple> ts = tstack;
	while (!ts.empty()) {
		Triple t = ts.top();
		if (t.is_valid())
			LOGXX("(" << t.h << "," << t.a << "," << t.b << ")");
		else
			LOGXX("EOS");
		ts.pop();
	}
}

Edge DFSPalmTree::new_vedges(int n1, int n2, list<Edge>& comp) {
	// new virtual edge, direction as for palm frond
	int id = ++max_edge_id_;
	Edge e = (pos_[n1] > pos_[n2] ? Edge(n1, n2, id) : Edge(n2, n1, id));
	LOGXX("    inserting " << e);
	// add virtual edge to *this
	// (note: tree arcs ordered before palm fronds in incoming_)
	incoming_[e.to].push_back(e);
	outgoing_[e.from].push_front(e);
	incoming_it_[e.id] = --incoming_[e.to].end();
	outgoing_it_[e.id] = outgoing_[e.from].begin();
	ASSERT(e.from != e.to); // should be biconnected
	++degree_[e.from];
	++degree_[e.to];
	edge_type_[e.id] = PalmFrond;
	// add virtual edge to comp
	comp.push_back(e);
	return e;
}

void DFSPalmTree::move_edge(Edge e, list<Edge>& comp) {
	LOGXX("    removing edge " << e);
	// remove e from *this
	Edge de = directed_edge(e);
	if (incoming_it_[e.id] != incoming_[de.to].end()) {
		ASSERT(!incoming_[de.to].empty());
		incoming_[de.to].erase(incoming_it_[e.id]);
	}
	if (outgoing_it_[e.id] != outgoing_[de.from].end()) {
		ASSERT(!outgoing_[de.from].empty());
		outgoing_[de.from].erase(outgoing_it_[e.id]);
	}
	incoming_it_[e.id] = incoming_[de.to].end();
	outgoing_it_[e.id] = outgoing_[de.from].end();
	ASSERT(de.from != de.to); // should be biconnected
	--degree_[de.from];
	--degree_[de.to];
	edge_type_[e.id] = NotVisited;
	// add e to comp
	comp.push_back(e);
}

// makes virtual edge a tree arc if appropriate
void DFSPalmTree::make_tree_arc(const Edge& e) {
	LOGXX("    setting " << e << " to tree arc");
	int lo = (pos_[e.from] < pos_[e.to] ? e.from : e.to);
	int hi = (pos_[e.from] < pos_[e.to] ? e.to : e.from);
	const list<Edge>& inhi = incoming_[hi];
	if (!inhi.empty() && edge_type_[inhi.front().id] == TreeArc) {
		LOGXX("      ignoring wish since " << hi << " already has father arc");
		return; // higher node already has a father arc, don't make two of them
	}
	LOGXX("      tree arc: " << lo << "->" << hi);
	// delete edge
	incoming_[e.to].erase(incoming_it_[e.id]);
	outgoing_[e.from].erase(outgoing_it_[e.id]);
	// add again as tree arc
	incoming_[hi].push_front(e);
	outgoing_[lo].push_front(e);
	incoming_it_[e.id] = incoming_[hi].begin();
	outgoing_it_[e.id] = outgoing_[lo].begin();
	edge_type_[e.id] = TreeArc;

	/*
	 LOGXX("    setting " << e << " to tree arc");
	 list<Edge>& in = incoming_[e.to];
	 // father arc must come at front of incoming edges, move edge to first or second pos
	 if (incoming_it_[e.id] != in.end())
	 in.erase(incoming_it_[e.id]);
	 //if (!in.empty() && edge_type_[in.front().id] == TreeArc)
	 //	LOGXX("have already a tree arc for " << e.to << ": " << in.front());
	 in.insert(in.begin(), e);
	 incoming_it_[e.id] = in.begin();
	 edge_type_[e.id] = TreeArc;
	 */
}

template<class T>
T top_pop(std::stack<T>& s) {
	T t = s.top();
	s.pop();
	return t;
}

template<class T>
T top(std::stack<T>& s) {
	return s.empty() ? T() : s.top();
}

void DFSPalmTree::check_for_type_2_pairs(int v, stack<Triple>& ts,//
		stack<Edge>& es, list<list<Edge> >& comps, int& w)//
{
	LOGXX("check for type 2 pairs");
	//print_estack(es);
	ASSERT(v == root_ || degree_[w] < 2 || first_son(w) >= 0);
	while ((top(ts).a == v || (degree_[w] == 2 && pos_[first_son(w)] > pos_[w]))
			&& v != root_) {
		Triple t = top(ts); // default triple if stack empty
		if (t.a == v && father(t.b) == v) {
			ts.pop();
		} else {
			Edge eab, ev; // eab.id < 0 means HT73::flag == false
			int x; // second node of separation pair (v is first node)
			if (degree_[w] == 2 && pos_[first_son(w)] > pos_[w]) {
				LOGXX("found type 2 separation pair (A): (" << v << "," << first_son(w) << ")");
				x = first_son(w);
				list<Edge> c; // new comp (triangle)
				move_edge(top_pop(es), c);
				move_edge(top_pop(es), c);
				ASSERT(x == c.back().from || x == c.back().to); // consistency check
				ev = new_vedges(v, x, c);
				if (top(es).is_between(v, x))
					eab = top_pop(es);
				comps.push_back(c); // done comp
			} else { // t.a == v && father(t.b) != t.a
				LOGXX("found type 2 separation pair (B): (" << t.a << "," << t.b << ")");
				ASSERT(t.a == v);
				x = t.b;
				ts.pop();
				list<Edge> c; // new comp (triangle or tricomp)
				while (!es.empty()) {
					Edge e = es.top();
					int elo = min(pos_[e.from], pos_[e.to]);
					int ehi = max(pos_[e.from], pos_[e.to]);
					if (!(pos_[t.a] <= elo && ehi <= pos_[t.h]))
						break;
					else if (e.is_between(t.a, t.b))
						eab = top_pop(es);
					else
						move_edge(top_pop(es), c);
				}
				ev = new_vedges(t.a, t.b, c);
				comps.push_back(c); // done TriComp or Triangle
			}
			if (eab.id >= 0) {
				list<Edge> c; // new comp (bond)
				move_edge(eab, c);
				move_edge(ev, c);
				ev = new_vedges(v, x, c);
				comps.push_back(c); // done comp
			}
			es.push(ev);
			make_tree_arc(ev);
			w = x;
		}
	}
}

void DFSPalmTree::check_for_type_1_pair(int v, stack<Triple>& ts,//
		stack<Edge>& es, list<list<Edge> >& comps, int& w)//
{
	LOGXX("check for type 1 pair");
	//print_estack(es);
	if (pos_[lowpt2_[w]] >= pos_[v] && pos_[lowpt1_[w]] < pos_[v] && //
			(father(v) != root_ || num_unvisited_outarcs_[v] > 0)) {
		LOGXX("found type 1 separation pair: (" << v << "," << lowpt1_[w] << ")");
		list<Edge> c; // new comp (triangle or tricomp)
		int plo = pos_[w];
		int phi = pos_[w] + nd_[w];
		while (!es.empty()) {
			int px = pos_[top(es).from];
			int py = pos_[top(es).to];
			if (!((px >= plo && px < phi) || (py >= plo && py < phi)))
				break;
			move_edge(top_pop(es), c);
		}
		Edge ev = new_vedges(lowpt1_[w], v, c);
		comps.push_back(c); // done comp (tricomp or triangle)
		// check if we produced multiple edge
		if (top(es).is_between(lowpt1_[w], v)) {
			list<Edge> c; // new comp (bond)
			move_edge(ev, c);
			move_edge(top_pop(es), c);
			ev = new_vedges(v, lowpt1_[w], c);
			comps.push_back(c);
		}
		if (lowpt1_[w] != father(v)) {
			es.push(ev);
			make_tree_arc(ev); //
		} else {
			list<Edge> c; // new comp (bond)
			move_edge(ev, c);
			move_edge(father_arc(v), c); // lowpt1_[w] -> v
			ev = new_vedges(lowpt1_[w], v, c);
			make_tree_arc(ev);
			comps.push_back(c); // done comp
		}
	}
}

void DFSPalmTree::find_split_components(int v,//
		stack<Triple>& ts,//
		stack<Edge>& es,//
		list<list<Edge> >& comps)//
{
	list<Edge>& el = outgoing_[v];
	list<Edge>::iterator e, enext;
	for (e = enext = el.begin(); e != el.end(); e = enext) {
		++enext; // e might be invalidated by check_for_*() calls
		int w = e->opposite(v); // son of v
		if (edge_type_[e->id] == TreeArc) {
			LOGXX("\narc v -> w: " << v << "->" << w);

			--num_unvisited_outarcs_[v];
			bool path_started = false;
			if (is_first_edge_of_path_[e->id]) {
				LOGXX("edge starts a path");
				path_started = true; // save beyond e validity
				Triple t(index_of_pos_[pos_[w] + nd_[w] - 1], lowpt1_[w], v);
				while (top(ts).is_valid() && pos_[top(ts).a] > pos_[lowpt1_[w]]) {
					t.h = pos_[top(ts).h] > pos_[t.h] ? top(ts).h : t.h;
					t.b = ts.top().b;
					ts.pop();
				}
				ts.push(t);
				ts.push(Triple()); // end-of-stack marker
			}
			print_estack(es);
			print_tstack(ts);
			find_split_components(w, ts, es, comps); // might invalidate e
			es.push(father_arc(w)); // (v,w) (might differ from e!)
			LOGXX("\ncontinue: " << v << "->" << w);
			print_estack(es);
			print_tstack(ts);
			check_for_type_2_pairs(v, ts, es, comps, w); // might invalidate e
			check_for_type_1_pair(v, ts, es, comps, w); // might invalidate e
			LOGXX("checked for separation pairs");
			if (path_started)
				LOGXX("path was started");
			while (path_started && !ts.empty() && top_pop(ts).is_valid())
				;
			while (top(ts).is_valid() /* && top(ts).a != v */&& top(ts).b != v
					&& highpos(v) > pos_[top(ts).h])
				ts.pop();
		} else if (edge_type_[e->id] == PalmFrond) {

			if (is_first_edge_of_path_[e->id]) {
				Triple t(v, w, v);
				while (top(ts).is_valid() && pos_[top(ts).a] > pos_[w]) {
					if (t.a == w || pos_[top(ts).h] > pos_[t.h])
						t.h = top(ts).h;
					t.b = ts.top().b;
					ts.pop();
				}
				ts.push(t);
			}
			//if (w == father(v)) {
			//	list<Edge> c; // new comp
			//	move_edge(*e, c);
			//	Edge ev = new_vedges(w, v, c);
			//	make_tree_arc(ev);
			//} else {
			es.push(*e);
			//}
		}
	}
}

list<list<Edge> > DFSPalmTree::find_split_components() {
	VERIFY(degree_.empty()); // find_split_components() never called before (!)
	find_paths();
	stack<Triple> tstack;
	stack<Edge> estack;
	// a bit generous (for real edges only we would need max (3*num_edges-6)):
	int emax = 3 * edge_type_.size(); // size of edge containers (room for virtual edges)
	int nmax = pos_.size();
	edge_type_.resize(emax);
	incoming_it_.resize(emax);
	outgoing_it_.resize(emax);
	degree_.assign(nmax, 0);
	num_unvisited_outarcs_.assign(nmax, 0);
	for (int p = 0; p < num_numbers_; ++p) {
		int v = index_of_pos_[p];
		list<Edge>& out = outgoing_[v];
		for (list<Edge>::iterator e = out.begin(); e != out.end(); ++e) {
			int w = e->opposite(v); // child of v
			++degree_[v];
			if (w != v)
				++degree_[w];
			outgoing_it_[e->id] = e;
			if (edge_type_[e->id] == TreeArc)
				++num_unvisited_outarcs_[v];
		}
		list<Edge>& in = incoming_[v];
		for (list<Edge>::iterator e = in.begin(); e != in.end(); ++e)
			incoming_it_[e->id] = e;
	}
#ifdef DEBUG
	LOGXX("\nsplit to components:\n");
	LOGXX("node\tpos\tlowpt1\tlowpt2\tnd\tdegree\thighpts");
	for (int p = 0; p < num_numbers_; ++p) {
		int n = index_of_pos_[p];
		LOGNXX(n << "\t" << pos_[n] << "\t" << lowpt1_[n] << "\t"<< lowpt2_[n]
				<< "\t" << nd_[n] << "\t" << degree_[n] << "\t");
		list<Edge>::const_iterator e;
		for (e = incoming_[n].begin(); e != incoming_[n].end(); ++e)
		if (edge_type_[e->id] == PalmFrond)
		LOGNXX(e->opposite(n) << " ");
		LOGXX("");
	}
	LOGXX("");
#endif //DEBUG
	// note: current recursive method will mess up outgoing_ etc.
	list<list<Edge> > comps;
	find_split_components(root_, tstack, estack, comps);
	// all remaining estack edges to new component
	list<Edge> comp; // new comp
	while (!estack.empty())
		comp.push_back(estack.top()), estack.pop();
	comps.push_back(comp);
	return comps;
}

bool DFSPalmTree::is_virtual_edge(int id) {
	return id > max_orig_edge_id_;
}

int DFSPalmTree::max_edge_id() const {
	return max_edge_id_;
}

int DFSPalmTree::max_original_edge_id() const {
	return max_orig_edge_id_;
}

void DFSPalmTree::print_dot(const string& filename) const {
	ofstream of(filename.c_str());
	of << "digraph DFSPalmTree {\n";
	of << "  rankdir=BT;\n";
	// directed edges: tree arcs and palm fronds
	for (size_t p = 0; p < num_nodes(); ++p) {
		int i = index_of_pos_[p];
		const list<Edge>& l = outgoing_[i];
		if (l.empty())
			continue;
		for (list<Edge>::const_iterator e = l.begin(); e != l.end(); ++e) {
			of << i << " -> " << e->opposite(i) << " [label=\"" << e->id
					<< "\",fontsize=10";
			if (edge_type_[e->id] == PalmFrond)
				of << ",style=dotted";
			else if (edge_type_[e->id] == NotVisited)
				of << ",style=dashed";
			of << "];\n";
		}
	}
	// nodes
	for (size_t p = 0; p < index_of_pos_.size(); ++p) {
		int i = index_of_pos_[p];
		of << i << " [label=\"" << i << " (" << pos_[i] << ";" << lowpt1_[i]
				<< "," << lowpt2_[i] << ")\",fontcolor=\"blue\",color=blue];\n";
		//			of << i << " [fontcolor=\"blue\",shape=circle,style=filled,color=blue,fixedsize=true,width=0,height=0];\n";
	}
	of << "}\n";
	of.close();
}

void DFSPalmTree::print_postscript(const string& filename) const {
	string dotfilename = filename + ".dot";
	print_dot(dotfilename);
	stringstream ss;
	ss << "dot -Tps " << dotfilename << " -o " << filename;
	int ret = system(ss.str().c_str());
	VERIFY(ret == 0);
}

}
// namespace Reduze
