/*
 * canonicallabel.h
 *
 *  Created on: Jul 23, 2016
 *      Author: cedric
 */

#ifndef REDUZE_CANONICALLABEL_H_
#define REDUZE_CANONICALLABEL_H_

#include "edge.h"

namespace YAML {
class Node;
class Emitter;
}

namespace Reduze {

/// canonical label for a Topology or Graph
class CanonicalLabel {
public:
	bool operator<(const CanonicalLabel&) const;
	bool operator==(const CanonicalLabel&) const;
	bool operator!=(const CanonicalLabel&) const;
	friend std::ostream& operator<<(std::ostream&, const CanonicalLabel&);
	friend YAML::Emitter& operator<<(YAML::Emitter&, const CanonicalLabel&);
	void read(const YAML::Node&, const GiNaC::lst&);
	void swap(CanonicalLabel& other);

private:
	int compare(const CanonicalLabel& other) const;
	friend class IFindCanonicalLabel;
	/// Edge attributes
	edge_attributes edge_attributes_;
	/// adjacency list, for each node (position in the vector) the connected nodes (keys of the map)
	/// with a colored multi edge which contains the number and color of the edges between the nodes
	/// the color number is a unique number which determines the mass and the propagator type of the edge
	std::vector<std::map<int, colored_multi_edge> > adjacencies_;
	/// The node colors, can be empty. A color number set manually for each node.
	/// E.g. for external nodes in order not to permute them
	std::vector<int> node_colorings_;
};

/// relation between original Topology and canonical label
class CanonicalRelabeling {
public:
	/// returns node map for Relabelings of Topologies with same Label
	/** From original Topology nodes of '*this' to those of 'to'. **/
	std::map<int, int> find_node_permutation(
			const CanonicalRelabeling& to) const;
	const std::vector<int>& node_permutation() const;
	void swap(CanonicalRelabeling& other);

	friend YAML::Emitter& operator<<(YAML::Emitter&,
			const CanonicalRelabeling&);
	friend void operator>>(const YAML::Node&, CanonicalRelabeling&);
	friend std::ostream& operator<<(std::ostream&, const CanonicalRelabeling&);

	static std::map<int, int> find_node_permutation(
			const std::vector<int>& from, const std::vector<int>& to);
private:
	friend class IFindCanonicalLabel;
	std::vector<int> node_permutation_; // from canonical to original ids
};

/// interface for classes which need a canonical label eg. Topology, Graph, SectorGraph
class IFindCanonicalLabel {
public:
	IFindCanonicalLabel();
	virtual ~IFindCanonicalLabel();

	// members to be set in derived class
	virtual const std::set<int>& nodes() const = 0;
	virtual std::set<int> external_nodes() const = 0;
	virtual const std::map<int, Edge>& edges() const = 0;
	virtual std::list<Edge> external_edges() const = 0;

private:
	virtual std::pair<std::list<std::map<int, int> >, edge_attributes> get_edge_coloring() const = 0;
	virtual std::list<std::map<int, int> > get_node_coloring(
			bool permute_external_nodes = true) const = 0;
public:
	int max_num_edge_colors() const;

	// graph isomorphism

	/// returns a canonical label and the mapping to it
	/** Two Topology instances are isomorphic if and only if their canonical
	 ** labels are equal. The Relabeling gives the node relabeling to arrive
	 ** at the canonical form described by the Label. **/
	std::pair<CanonicalLabel, CanonicalRelabeling> find_canonical_label(
			bool perm_ext_nodes = true) const;

	/// returns the permutations of the nodes that leave the topology and the canonical label invariant
	/** If 'suppress_free_ext_node_perms' is set to true then external nodes are suppressed
	 ** of being permuted freely but only if internal symmetries force the permutation of
	 ** external nodes. **/
	std::list<std::map<int, int> > find_node_symmetry_group(
			bool suppress_free_ext_node_perms = false) const;

	// members for alternative finding of edge symmetries (via line graph)

	/// finds the permutations of the edges with the orientation. e_from --> (sign, e_to)
	/** Transformations that only permute external legs are suppressed.
	 ** The edge permutations are obtained by the node permutations of the line graph.
	 ** In reconstructing a node permutation invalid edge permutations are filtered
	 ** out and the sign of the edge permutation is determined. **/
	void find_edge_symmetry_group(
			std::list<std::map<int, std::pair<int, int> > >& edge_permutations,
			bool suppress_free_ext_node_perms = false) const;
};

template<class T>
std::list<T*> select_minimals(const std::list<T*>& graphs) {
	typedef IFindCanonicalLabel IFC;
	// all isomorphic minimal graphs (canonical label calculated with permuting external nodes)
	std::list<T*> minimals, result;
	typename std::list<T*>::const_iterator g;
	CanonicalLabel label, best;
	for (g = graphs.begin(); g != graphs.end(); ++g) {
		if ((dynamic_cast<IFC*>(*g)) == 0)
			throw std::runtime_error("error");
		label = (dynamic_cast<IFC*>(*g))->find_canonical_label().first;
		if ((g == graphs.begin()) || label < best) {
			best = label;
			minimals.clear();
			minimals.push_back(*g);
		} else if (label == best) {
			minimals.push_back(*g);
		}
	}
	if (minimals.size() < 2
			|| (dynamic_cast<IFC*>(*(minimals.begin())))->external_edges().size()
					== 0)
		return minimals;

	// find minimal canonical label with fixed external nodes
	for (g = minimals.begin(); g != minimals.end(); ++g) {
		label = (dynamic_cast<IFC*>(*g))->find_canonical_label(false).first;
		if (g == minimals.begin() || label < best) {
			best = label;
			result.clear();
			result.push_back(*g);
		} else if (label == best) {
			result.push_back(*g);
		}
	}
	return result;
}

template<class T>
T* select_minimal(const std::list<T*>& graphs) {
	std::list<T*> minimals = select_minimals(graphs);
	if (minimals.empty())
		throw std::runtime_error("Error: empty list");
	return *minimals.begin();
}

} // namespace Reduze

#endif /* REDUZE_CANONICALLABEL_H_ */
