# Copyright (c) HashiCorp, Inc.
# SPDX-License-Identifier: BUSL-1.1

require "optparse"
require 'vagrant/util/install_cli_autocomplete'

module VagrantPlugins
  module CommandAutocomplete
    module Command
      class Root < Vagrant.plugin("2", :command)
        def self.synopsis
          "manages autocomplete installation on host"
        end

        def initialize(argv, env)
          super

          @main_args, @sub_command, @sub_args = split_main_and_subcommand(argv)

          @subcommands = Vagrant::Registry.new
          @subcommands.register(:install) do
            require File.expand_path("../install", __FILE__)
            Install
          end
        end

        def execute
          if @main_args.include?("-h") || @main_args.include?("--help")
            # Print the help for all the box commands.
            return help
          end

          # If we reached this far then we must have a subcommand. If not,
          # then we also just print the help and exit.
          command_class = @subcommands.get(@sub_command.to_sym) if @sub_command
          return help if !command_class || !@sub_command
          @logger.debug("Invoking command class: #{command_class} #{@sub_args.inspect}")

          # Initialize and execute the command class
          command_class.new(@sub_args, @env).execute
        end

        def help
          opts = OptionParser.new do |opts|
            opts.banner = "Usage: vagrant autocomplete <subcommand>"
            if !Vagrant.server_mode?
              opts.separator ""
              opts.separator "Available subcommands:"

              # Add the available subcommands as separators in order to print them
              # out as well.
              keys = []
              keys = @subcommands.keys.map(&:to_s)

              keys.sort.each do |key|
                opts.separator "     #{key}"
              end
            end

            opts.separator ""
            opts.separator "For help on any individual subcommand run `vagrant autocomplete <subcommand> -h`"
          end

          @env.ui.info(opts.help, prefix: false)
        end
      end
    end
  end
end
