//==========================================
//  xhkeys_mixer.c
//------------------------------------------
// Audio Mixer Control.
// A plug-in module for xhkeys
// Michael Glickman  Jun-2004
//==========================================


#include <X11/Xlib.h>	// True False
#include "xhkeys_conf.h"

#ifdef MIXER_SUPPORT
#include <stdio.h>	// NULL
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <linux/soundcard.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/stat.h>
#define XHKEYS_PLUGIN_CODE
#include "xhkeys_plugin.h"
#include "plugin_common.h"

enum CmdCodes {
    MIXER_SNDVOLUME,
    MIXER_SNDBALANCE,
    MIXER_SNDMUTE,
    MIXER_SNDSTATUS,
    MIXER_CODE_COUNT
};    

static const struct 
{
    const char *opname;
    int flags;

}
OpData[] =
{
    { "Volume", XHKEYS_PLUGOPTYPE_PARMREQUIRED},
    { "Balance", XHKEYS_PLUGOPTYPE_PARMREQUIRED},
    { "Mute", XHKEYS_PLUGOPTYPE_PARMOPTIONAL},
    { "Status", XHKEYS_PLUGOPTYPE_PARMOPTIONAL | XHKEYS_PLUGOPTYPE_CANCONTINUE}
};

static const char *parmNames[] = {"devname", "devgrab", "channel", "osd"};
static const int parmCount = sizeof(parmNames) / sizeof(const char *);

#define MAX_VOLUME 100
#define MIN_VOLUME 0

const char *SoundDeviceLabels[] = SOUND_DEVICE_LABELS;


const char *Description = "Audio Mixer Control";

const char *IoErrorMessage = "%s: mixer device I/O error";
const char *InvArgMessage = "Invalid %s argument";
const char *InvChannelMessage = "Invalid %s mixer channel, (-1) or non-negative less than %d expected";
const char *NoArgMessage = "Missing %s argument";
const char *validDelim = ";.,-";
const char *validTerm = " \t#";

typedef struct
{
    char MixerDevName[256];
    int mixer_fd;
    int mixerChannel;
    Bool grabMixer;
    Bool osd;
    int oldVolumeLeft;
    int oldVolumeRight;
} BufferData;    

int get_mixer_fd(BufferData *buffer)
{
    int mixer_fd = buffer->mixer_fd;
    
    if (mixer_fd < 0)
	mixer_fd = open(buffer->MixerDevName, O_RDWR);

    return mixer_fd;     
}


void release_mixer_fd(BufferData *buffer)
{
    int mixer_fd = buffer->mixer_fd;
     
    if (mixer_fd >= 0 && buffer->grabMixer == False) {
	close(mixer_fd);
	buffer->mixer_fd = -1;
    }	
}

static Bool mixer_query_volumes(int mixer_fd, int *devPtr, int *volLeft, int *volRight)
{
    static const int devices[] = {SOUND_MIXER_VOLUME, SOUND_MIXER_PCM, SOUND_MIXER_ALTPCM};
    int device = *devPtr;
    int volumes;
    int i;
    Bool success;

    if (device >= 0)
	success =(ioctl(mixer_fd, MIXER_READ(device), &volumes) != -1);
    else {	  
	 for (i=0;
    	     i<3 && ioctl(mixer_fd, MIXER_READ(device=devices[i]), &volumes) == -1;
		 i++);
	 success = (i < 3);
    }
  
    if (success) {
        *devPtr = device;
	*volLeft = volumes & 0xff;;
        *volRight = (volumes>>8) & 0xff;
    }        
    
    
  return success;	
}


static Bool mixer_set_volumes(int mixer_fd, int device, int volLeft, int volRight)
{
  if (volLeft < MIN_VOLUME) volLeft=MIN_VOLUME;
  else
  if (volLeft > MAX_VOLUME) volLeft=MAX_VOLUME;
  
  if (volRight < MIN_VOLUME) volRight=MIN_VOLUME;
  else
  if (volRight > MAX_VOLUME) volRight=MAX_VOLUME;
    
  volLeft |= (volRight<<8);  
  return (ioctl(mixer_fd, MIXER_WRITE(device), &volLeft) != -1);
}

void left_right_to_vol_balance(int volLeft, int volRight, int *volPtr, int *balPtr)
{
    int volume, balance;    

    if (volRight >= volLeft) {
	// Non-negative
	volume = volRight;
	balance = volRight - volLeft;
	// If balance > 0, then volRight is guaranteed to be positive
	if (balance > 0) balance = (balance * 50) / volRight;
    } else {
	// Negative balance (volLeft > volRight, therefore
	// volLeft is guaranteed to be positive
	volume = volLeft;
	balance = (volRight - volLeft) * 50 / volLeft; 
    } 

//    *volPtr = volume * 100 / MAX_VOLUME;
    *volPtr = volume;
    *balPtr = balance;

}


void vol_balance_to_left_right(int volume, int balance, int *volLeftPtr, int *volRightPtr)
{
    int left, right;

    if (balance == 0)
	left = right = volume;
    else 
    if (balance > 0) {
	right = volume;
        left = volume - (volume * balance) / 50;    
    } else {
	left = volume;
	right = volume + (volume * balance) / 50;     
    }	

//    *volLeftPtr = left * MAX_VOLUME / 100;
//    *volRightPtr = right * MAX_VOLUME / 100;
    *volLeftPtr = left;
    *volRightPtr = right;
}

static Bool mixer_show_status(int mixer_fd, int device, char *message, int msglen)
{
    int volLeft, volRight;
    int volume, balance;

    if (mixer_query_volumes(mixer_fd, &device, &volLeft, &volRight) == False)
	return False;
	
    left_right_to_vol_balance(volLeft, volRight, &volume, &balance);

    snprintf (message, msglen, 
              "%s: volume %d  balance %d", 
	      SoundDeviceLabels[device], volume, balance);
    return True;	      
}


static Bool process_channel(BufferData *buffer, int opcode, const char *parm, int *devPtr, char *message, int msglen)
{
    char *nextText;
    int device;

    device = buffer->mixerChannel;

    if (parm != NULL && *parm != '\0'
                     && strchr(validTerm, *parm) == NULL) {
    
	device = strtol(parm, &nextText, 0);
        if ((*nextText != '\0' && strchr(validTerm, *nextText) != NULL) ||
	    device < -1 || device >= SOUND_MIXER_NRDEVICES) {
	    const char *opName = OpData[opcode].opname;
	    snprintf(message, msglen, InvChannelMessage, opName, SOUND_MIXER_NRDEVICES);
	    return False;
        }	
    }
    *devPtr = device;
    return True;
}


static Bool process_parm(BufferData *buffer, int opcode, const char *parm, int *valPtr, short *dirPtr, int *devPtr,
                         char *message, int msglen)
{
    const char *opName = OpData[opcode].opname;
    char *cmdCopy = NULL, *chanTxt;
    char sep;
    Bool success = False;

    // Parameter must be present
    if (parm == NULL || (cmdCopy = strdup(parm)) == NULL) {
	snprintf(message, msglen, NoArgMessage, opName);
	goto HastaLaVista;
    }

    if (parse_parm_plusminus(cmdCopy, valPtr, dirPtr, &chanTxt) == False) {
	snprintf(message, msglen, InvArgMessage, opName);
	goto HastaLaVista;
    }

    sep = *chanTxt;

    if (sep != '\0' && strchr(validDelim, sep) != NULL)
	chanTxt++;
    else 	
    if (sep != '\0' && strchr(validTerm, sep) == NULL) {
	snprintf(message, msglen, InvArgMessage, opName);
	goto HastaLaVista;
    }	
    else
	chanTxt = NULL;
	
    success = process_channel(buffer, opcode, chanTxt, 
	           devPtr, message, msglen);

HastaLaVista:
    if (cmdCopy) free(cmdCopy);    
    return success;    
}    


//static Bool mixer_status(BufferData *buffer, Bool force_osd, const char *parm, char *message, int msglen)
static Bool mixer_status(BufferData *buffer, const char *parm, char *message, int msglen)
{
    Bool success = False;
    int device;
    int mixer_fd;

    success = process_channel(buffer, MIXER_SNDMUTE, parm, &device, message, msglen);
    if (success == False) return success;

    mixer_fd = get_mixer_fd(buffer);

    if (mixer_fd < 0) goto OutOfHere;

    // We show status even if OSD is disabled	   
    success = mixer_show_status(mixer_fd, device, message, msglen);
    

OutOfHere:
    release_mixer_fd(buffer);
    return success;
}



static Bool mixer_volume(BufferData *buffer, const char *parm, char *message, int msglen )
{
    Bool success = False;
    int volLeftOld, volRightOld;
    int volLeft, volRight, value;
    int volume, balance;
    short dir;
    int mixer_fd;
    int device;

    if (process_parm(buffer, MIXER_SNDVOLUME, parm,
           &value, &dir, &device, message, msglen) == False)
	return False;    

    mixer_fd = get_mixer_fd(buffer);

    if (mixer_fd < 0 ||
	mixer_query_volumes(mixer_fd, &device, &volLeftOld, &volRightOld) == False) {
        snprintf(message, msglen, IoErrorMessage, OpData[MIXER_SNDVOLUME].opname);
	goto OutOfHere;
    }

    left_right_to_vol_balance(volLeftOld, volRightOld, &volume, &balance);
  
    if (dir == 2 || dir == 0) {
	// Absolute volume
	if (value < 0 || value > 100) {
	    snprintf(message, msglen, InvArgMessage, OpData[MIXER_SNDVOLUME].opname);
	    goto OutOfHere;
	}
	volume = value;    
        vol_balance_to_left_right(volume, balance, &volLeft, &volRight);
    } else 
	// Relative zero - no change
    if (value == 0) goto JustShowIt;
    else {
	// Relative nonzero - avoid dead lock
        int iter = 3;
	Bool needBreak = False;

        if (dir < 0) value = -value;

        do {
	    volume += value;
	    
	    if (volume < 0) {
		volume = 0;
		if (value < 0) needBreak = True;
	    } 
	    else
	    if (volume > 100) {
		volume = 100;
		if (value > 0) needBreak = True;
	    }	 
		
	    vol_balance_to_left_right(volume, balance, &volLeft, &volRight);
		
	} while (volLeft == volLeftOld && volRight == volRightOld &&
	         needBreak == False && --iter > 0);
	
    }
    
    success = mixer_set_volumes(mixer_fd, device, volLeft, volRight);
    
JustShowIt:
    if (success && buffer->osd)  
	mixer_show_status(mixer_fd, device, message, msglen);

OutOfHere:
    release_mixer_fd(buffer);
    return success;
}

static Bool mixer_mute(BufferData *buffer, const char *parm, char *message, int msglen)
{
    int device; 
    int volLeft, volRight;
    Bool success = False;
    int mixer_fd;

    success = process_channel(buffer, MIXER_SNDMUTE, parm, &device, message, msglen);
    if (success == False) return success;
	
    mixer_fd = get_mixer_fd(buffer);
    
    if (mixer_fd < 0 ||
	mixer_query_volumes(mixer_fd, &device, &volLeft, &volRight) == False) {
        snprintf(message, msglen, IoErrorMessage, OpData[MIXER_SNDMUTE].opname);
	goto OutOfHere;
    }	
    
    if (buffer->oldVolumeLeft == -1) {
	// Make it mute
	success = mixer_set_volumes(mixer_fd, device, 0, 0);
	
	if (success) {
	    buffer->oldVolumeLeft = volLeft;
	    buffer->oldVolumeRight = volRight;
	}    
    } else {
	// Restore sound
	success = mixer_set_volumes(mixer_fd, device, buffer->oldVolumeLeft, buffer->oldVolumeRight);
	
	if (success)
	    buffer->oldVolumeLeft = buffer->oldVolumeRight = -1;
    }

    if (success && buffer->osd)  
        mixer_show_status(mixer_fd, device, message, msglen);

OutOfHere:
    release_mixer_fd(buffer);
    return success;
}

static Bool mixer_balance(BufferData *buffer, const char *parm, char *message, int msglen )
{
    Bool success = False;
    int volLeftOld, volRightOld;
    int volLeft, volRight, value;
    int volume, balance;
    short dir;
    int mixer_fd;
    int device;

    if (process_parm(buffer, MIXER_SNDVOLUME, parm,
           &value, &dir, &device, message, msglen) == False)
	return False;    

    mixer_fd = get_mixer_fd(buffer);

    if (mixer_fd < 0 ||
	mixer_query_volumes(mixer_fd, &device, &volLeftOld, &volRightOld) == False) {
        snprintf(message, msglen, IoErrorMessage, OpData[MIXER_SNDBALANCE].opname);
	goto OutOfHere;
    }

    left_right_to_vol_balance(volLeftOld, volRightOld, &volume, &balance);
  
    if (dir == 2 || dir == 0) {
	// Absolute volume
	if (value < -50 || value > 50) {
	    snprintf(message, msglen, InvArgMessage, OpData[MIXER_SNDBALANCE].opname);
	    goto OutOfHere;
	}
	balance = value;    
        vol_balance_to_left_right(volume, balance, &volLeft, &volRight);
    } else 
	// Relative zero - no change
    if (value == 0) goto JustShowIt;
    else {
	// Relative nonzero - avoid dead lock
        int iter = 3;
	Bool needBreak = False;

        if (dir < 0) value = -value;

        do {
	    balance += value;
	    
	    if (balance < -50) {
		balance = -50;
		if (value < 0) needBreak = True;
	    } 
	    else
	    if (balance > 50) {
		balance = 50;
		if (value > 0) needBreak = True;
	    }	 
		
	    vol_balance_to_left_right(volume, balance, &volLeft, &volRight);
		
	} while (volLeft == volLeftOld && volRight == volRightOld &&
	         needBreak == False && --iter > 0);
	
    }
    
    success = mixer_set_volumes(mixer_fd, device, volLeft, volRight);
    
JustShowIt:
    if (success && buffer->osd)  
	mixer_show_status(mixer_fd, device, message, msglen);

OutOfHere:
    release_mixer_fd(buffer);
    return success;
}


//==============================================================
// Exported functions
//==============================================================
Bool init(void **bufReturn, const char *command, char *message, int msglen)
{
    char MixerDevName[256] = "/dev/mixer";
    int mixer_fd = -1;
    int mixerChannel = -1;
    Bool grabMixer = False;
    Bool osd = True;

    const int  parmAttr[] = {ATTRIB_CHAR_CUSTOM_SEP | sizeof(MixerDevName),
                                ATTRIB_BOOL, ATTRIB_INT | 4, ATTRIB_BOOL};
    void *parmValPtr[] = {MixerDevName, &grabMixer, &mixerChannel, &osd};
    BufferData *buffer;
    Bool result;
    

    *bufReturn = NULL;
    
    result = True;
    if (command != NULL && *command != '\0')
	result = lookup_parms(command, parmCount,
	     parmNames, parmAttr, parmValPtr, message, msglen);
    if (result == False) return False;

/*	 	
    printf ("MixerDevName = '%s' holdMixer = %d, channed = %d, osd = %d\n",
	    MixerDevName, grabMixer, mixerChannel, osd);
*/

    if (mixerChannel < -1 || mixerChannel >= SOUND_MIXER_NRDEVICES) {
        snprintf(message, msglen, InvChannelMessage, "init", SOUND_MIXER_NRDEVICES);
	return False;
    }	
    
    buffer = (BufferData *)malloc(sizeof(BufferData));
    if (buffer == NULL) return False;

    *bufReturn = (void *) buffer;
    
    strcpy(buffer->MixerDevName, MixerDevName);
    buffer->mixer_fd = mixer_fd;
    buffer->mixerChannel = mixerChannel;
    buffer->grabMixer = grabMixer;
    buffer->osd = osd;

    buffer->oldVolumeLeft = 
    buffer->oldVolumeRight = -1;
    
    return result;
}



int getopcount(void)
{
    return MIXER_CODE_COUNT;
}

Bool getopdata(int code, char opname[XHKEYS_PLUGIN_OPNAME_LENGTH], int *flags)
{
    if (code < 0 || code >= MIXER_CODE_COUNT) return False;
    
    if (opname != NULL) {
	strncpy(opname, OpData[code].opname, XHKEYS_PLUGIN_OPNAME_LENGTH);
        opname[XHKEYS_PLUGIN_OPNAME_LENGTH-1] = '\0';
    }	

    if (flags != NULL) *flags = OpData[code].flags;
    return True;
}   

Bool process(void *bufferVoidPtr, int code, const char *parm, int behaviour,
            char *message, int msglen)
{
  Bool success = False;
  BufferData *buffer = (BufferData *)bufferVoidPtr;
//  int retCode = XHKEYS_PLUGPROCRC_ERROR;

  *message = '\0';
  
  if (behaviour == XHK_BEHAVIOUR_DISCONTINUE) return 0;

  if (behaviour != XHK_BEHAVIOUR_ONCE && code != MIXER_SNDSTATUS)
    return success;
//      return retCode;

  switch(code) {
	case MIXER_SNDVOLUME:
	    success = mixer_volume(buffer, parm, message, msglen);
	    break;

	case MIXER_SNDBALANCE:
	    success = mixer_balance(buffer, parm, message, msglen);
	    break;

	case MIXER_SNDMUTE:
	    success = mixer_mute(buffer, parm, message, msglen);
	    break;

	case MIXER_SNDSTATUS:
	    success = mixer_status(buffer, parm, message, msglen);
	    break;
  }

//    if (success) 
//	retCode = (*message == '\0') ? 0 : XHKEYS_PLUGPROCRC_OSDTEXT;

//    return retCode;
    return success;
  
}

void term(void *buffer)
{
    int mixer_fd = ((BufferData *)buffer)->mixer_fd;
    
    if (mixer_fd >= 0) close(mixer_fd);
}

Bool property(void *buffer, XHKEYS_PLUGIN_PROPERTY code, ...)
{
    Bool success = True;
    int  intVal;
    int  *intPtr;
    char *charPtr;
    
    va_list arg_list;
    
    va_start(arg_list, code);
    
    switch(code)
    {
	case XHK_PLUGIN_VERSION:
	    if ((intPtr = va_arg(arg_list, int *)) != NULL) {
		*intPtr = XHKEYS_PLUGIN_VERSION;
		success = True;
	    }
	    break;
    
	case XHK_PLUGIN_DESCRIPTION:
	    if ((charPtr = va_arg(arg_list, char *)) != NULL &&
		(intVal = va_arg(arg_list, int)) > 0) {
	        strncpy(charPtr, Description, intVal);
		success = True;
	    }	
	    // Break is not necessary
	    
	default:
	    break;    
    }

    va_end(arg_list);
    
    return success;
}

#endif

