/*===================================================*/
/*         Hot key Processing                        */
/*---------------------------------------------------*/
/*       20.07.02                          M.G.      */
/*===================================================*/


#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "xhkeys_conf.h"

#include <sys/types.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <pwd.h>

#if defined(PLUGIN_SUPPORT) || defined(OSD_SUPPORT)
#define USE_TIMER_HANDLER
#include <sys/time.h>
#endif

#define _GNU_SOURCE
#include <getopt.h>


#ifdef USE_DMALLOC
#include <dmalloc.h>
#endif

#include "xhkeys.h"

/* system */
extern char **environ;

/* resources.c */
extern CmdDescr *Commands;
extern int CommandCount;
extern int maxCodes;
extern const char *LogModeCodes;
extern int logMode;		// 0 - none, 1 - error, 2 - all
extern int logConsole;		// 1 force logging to console
#ifdef OSD_SUPPORT
static char *dispName = NULL;
extern int osdEnabled;
#endif

Display *dpy = NULL;
Window  wnd = None;
Window  root_wnd = None;
Atom WMStateAtom, XHKeysPidAtom;
Bool debug = False;
Bool help = False;
int screenNo;

static const char *resFileName = NULL;
static void (*oldChildExitHandler)(int) = NULL;
static void (*oldTerminateHandler)(int) = NULL;

#ifdef USE_TIMER_HANDLER
#define TICK_PROCESSOR_COUNT 2

typedef void (TICK_PROCESSOR)(const struct timeval *curTime);
extern TICK_PROCESSOR ProcessPluginTick, ProcessOSDTick;

static void (*oldTimeHandler)(int) = NULL;
static struct itimerval oldTimValue;
static struct timeval TickFireTime[TICK_PROCESSOR_COUNT];
static TICK_PROCESSOR *TickProcessors[TICK_PROCESSOR_COUNT] =
{
# ifdef PLUGIN_SUPPORT
    ProcessPluginTick,
# else
    NULL,
# endif

# ifdef OSD_SUPPORT
    ProcessOSDTick
# else    
    NULL
# endif        
};
#endif

static void terminateHandler(int sigCode);
static void childExitHandler(int sigCode);



static void showHelp(void) {
  static const char *helpText[] = {
	"",
	"xhkeys ver. " VERSION_TEXT " " RELEASE_DATE ", compiled " __DATE__ " " __TIME__,
	"Allows using keyboard extra keys with X server.",
	"(C) 2002,2004  Michael Glickman <wmalms@yahoo.com>",
	"Web: http://wmalms.tripod.com",
	"     http://www.geocities.com/wmalms",
	"",
	"Command line arguments:",
	"\t-c --console      log to console", 
	"\t-d --debug        no fork, log to stdout", 
	"\t-f --file <name>  configuration file name", 
	"\t                  (default ~/.XHkeys)",
	"\t-h --help         this page", 
	"\t-l --log <.>      log: N-none, E-error, A-all", 
//	"\-c --console <.>  log to: Y-console, N-syslog", 
#ifdef OSD_SUPPORT
	"\t-o --noosd        disable on-screen display", 
	"\t-O --osd [disp.name] enable on-screen display", 
	"\t                  optionally with display name", 
	"\t                  as Server:ScreenNumber", 
#endif	
	"\t-s --syslog       log to syslog", 
	"", 
	"Use xhkconf for configuration",
	"See manual.html for details", 
	"", NULL };
	
  const char *helpLine;
  int i;
  
  for (i=0; (helpLine=helpText[i])!= NULL; i++)
	puts(helpLine);
}


static Bool ProcessCommandLine(int argc, char * argv[])
{

  int result;
  
  static struct option long_opts[] =
  { { "file", required_argument, NULL, 'f'},
    { "debug", no_argument, NULL, 'd'},
    { "help", no_argument, NULL, 'h'},
    { "log", required_argument, NULL, 'l'},
 //   { "console", optional_argument, NULL, 'c'},
    { "console", no_argument, NULL, 'c'},
#ifdef OSD_SUPPORT    
    { "noosd", no_argument, NULL, 'o'},
    { "osd", optional_argument, NULL, 'O'},
#endif    
    { "syslog", no_argument, NULL, 's'},
    {NULL, 0, NULL,  0} 
  };

  resFileName = NULL;
  debug = False;
  help = False;

  logMode = -1;
  logConsole = -1;

#ifdef OSD_SUPPORT    
  dispName = NULL;
  osdEnabled = -1;    
#endif    

  
  while ((result = getopt_long(argc, argv, "f:dhl:csoO::", long_opts, NULL)) != -1) {
	switch(result) {
	  case 'f':
		resFileName = optarg;
		break;	

	  case 'h':
		help = True;
		return True;

	  case 'l':
	  {
	    	char *tmp = strchr(LogModeCodes, toupper(*optarg));
		if (tmp == NULL) {
		  fprintf (stderr, "-- invalid log action\n");
		  return False;
		}  
		else
		    logMode = (tmp-LogModeCodes);  
	  }
		break;	

	  case 'd':
		debug = True;
		// No break !
		
	  case 'c':    	
		logConsole = 1;
		break;	

	  case 's':
		logConsole = 0;	
		break;

// If OSD_SUPPORT is not pressent, all fall to '?'
	  case 'o':
#ifdef OSD_SUPPORT    
		osdEnabled = 0;
		break;
#endif

	  case 'O':
#ifdef OSD_SUPPORT    
		osdEnabled = 1;
		if (optarg != NULL) {
		    if (dispName != NULL) free(dispName);
		    dispName = strdup(optarg);
		}
		break;
#else
		log_message(True, "Arguments 'o' and 'O' are not accepted - no OSD support");
#endif
	  case '?':
		return False;	
	}
  
  }

  
  return True;

}

void terminate(void)
{
    int i;
    
#ifdef USE_TIMER_HANDLER
    if (oldTimeHandler != NULL) {
	signal(SIGALRM, oldTimeHandler);
	setitimer(ITIMER_REAL, &oldTimValue, NULL);
    }	
#endif    

#ifdef PLUGIN_SUPPORT
    DeinitPluginSupport();
#endif

#ifdef OSD_SUPPORT
    if (dispName != NULL) free(dispName);
    DeinitOSD();
#endif

    
    if (oldTerminateHandler != NULL)
	signal(SIGTERM, oldTerminateHandler);

    if (oldChildExitHandler != NULL)
        signal(SIGCHLD, oldChildExitHandler);

	
    for (i=0; i<CommandCount; i++) 
        XUngrabKey(dpy, Commands[i].keycode, Commands[i].modifier, wnd);

    //  DeinitSpecFunctions();
    FreeResources();

    if (XHKeysPidAtom != None) 
        XDeleteProperty(dpy, root_wnd, XHKeysPidAtom);

    //  if (wnd) XDestroyWindow(dpy, wnd); 
    if (dpy != NULL) {
        XCloseDisplay(dpy);
        dpy = NULL;
    }    
        

}

#ifdef USE_TIMER_HANDLER
static void  timeHandler(int sigAction);

static void JustClearTickTime(int index)
{
    TickFireTime[index].tv_sec = TickFireTime[index].tv_usec = 0;
}

static void FireTimeTicker(int index, const struct timeval *curTimePtr)
{
    TICK_PROCESSOR   *tickProcessor = TickProcessors[index];
    JustClearTickTime(index);
    if (tickProcessor != NULL) (*tickProcessor)(curTimePtr);
}


void  AdjustITimer(const struct timeval *curTimePtr)
{
    long usecs_remained;
    long usecs_best = 0x7fffffffl;
    long secs1, curSecs;
    long usecs1, curUsecs;
    int i;
    
    curSecs = curTimePtr->tv_sec;
    curUsecs = curTimePtr->tv_usec;

    for (i=0; i<TICK_PROCESSOR_COUNT; i++) {
	secs1 = TickFireTime[i].tv_sec;	    
	usecs1 = TickFireTime[i].tv_usec;	    
	if (secs1 == 0 && usecs1 == 0) continue;    
	usecs_remained = (secs1 - curSecs) * 1000000l + (usecs1 - curUsecs);
	if (usecs_remained <= 0)  FireTimeTicker(i, curTimePtr);
	else
	if (usecs_remained < usecs_best) usecs_best = usecs_remained;	    
    }

    if (usecs_best == 0x7fffffffl) {
        signal(SIGALRM, SIG_IGN);
        setitimer(ITIMER_REAL, &oldTimValue, NULL);
    } else {
        struct itimerval tval;
        tval.it_interval.tv_sec = oldTimValue.it_interval.tv_sec;
        tval.it_interval.tv_usec = oldTimValue.it_interval.tv_usec;
	tval.it_value.tv_sec = usecs_best / 1000000;
	tval.it_value.tv_usec = usecs_best % 1000000; ;
	setitimer(ITIMER_REAL, &tval, NULL);
	signal(SIGALRM, timeHandler);
    }

}

void ClearTickTime(int index)
{
    struct timeval curTime;


    JustClearTickTime(index);
    gettimeofday(&curTime, NULL);
    AdjustITimer(&curTime);
}

void SetTickTime(int index, unsigned long usecs_plus)
{
    struct timeval curTime;
    long oldSecs, newSecs;
    long newUsecs;
    
    gettimeofday(&curTime, NULL);

    oldSecs = TickFireTime[index].tv_sec;

    usecs_plus += curTime.tv_usec;
    newSecs = curTime.tv_sec + usecs_plus / 1000000; 
    newUsecs = usecs_plus % 1000000; 

    // If old time is dummy (sec and usec are zero), it will still work !
    if (newSecs > oldSecs ||
        (newSecs == oldSecs && newUsecs > TickFireTime[index].tv_usec)) {
        TickFireTime[index].tv_sec = newSecs;
	TickFireTime[index].tv_usec = newUsecs;
        AdjustITimer(&curTime);
    }	
}

static void  timeHandler(int sigAction)
{
    struct timeval curTime;
    long secs1, usecs1;
    int i;

    gettimeofday(&curTime, NULL);

    for (i=0; i<TICK_PROCESSOR_COUNT; i++) {
	secs1 = TickFireTime[i].tv_sec;	    
	usecs1 = TickFireTime[i].tv_usec;	    
	if (secs1 == 0 && usecs1 == 0) continue;    

	if (curTime.tv_sec > secs1 ||
	    (curTime.tv_sec == secs1 && curTime.tv_usec >= usecs1)) 
	    FireTimeTicker(i, &curTime);
    }

    AdjustITimer(&curTime);    
}

#endif

static void terminateHandler(int sigCode)
{
/*
  terminate(False, 
#ifdef USE_SYSLOG
	"xkeys killed"
#else
	NULL
#endif  
  );
*/
  char msg[61];
  
  sprintf (msg, "xhkeys [pid %d] killed",  getpid());
  log_message(False, msg);

  terminate();
  exit(4);
}

// This will help to get rid of zombies
static void childExitHandler(int sigCode)
{
	wait(NULL);
}

int main(int argc, char *argv[])
{
  XEvent xev;
  char *errMsg = NULL;
  const char * command;
  int i, keycode, type;
  unsigned int modif;
  Bool success = False; 
  char msg[61];
  pid_t pid;


  resFileName = NULL;
  
  if (!ProcessCommandLine(argc, argv) || help) {
  	showHelp();
	return help ? 0 : 16;
  }	  
 
  if (!debug) // && getpid() == tcgetpgrp(0))
  {
    if ((i=fork()) != 0) {
	  if (i == -1) errMsg = "Forking error!";
	         else success = True;
	  goto OutOfHere; 
  	} 
  }

#ifdef OSD_SUPPORT
  dpy = XOpenDisplay(dispName);
  if (dpy == NULL) {
    if (dispName == NULL)
	errMsg = "Can't open display";
    else {
	snprintf(msg, sizeof(msg), "Invalid display name '%s'", dispName);
	errMsg = msg;
    }	
#else
  dpy = XOpenDisplay(NULL);
  if (dpy == NULL) {
    errMsg = "Can't open display";
#endif    

    goto OutOfHere;
  }  

 screenNo = XDefaultScreen(dpy);    

  root_wnd = DefaultRootWindow(dpy);
  if (root_wnd == None) {
	errMsg = "No root window";
	goto OutOfHere;
  }  

  wnd = root_wnd;
/*
  wnd = XCreateSimpleWindow(dpy, root_wnd, 0, 0, 10, 10, 1, 0, 0);
  if (wnd == None) {
	errMsg = "Couldn't create a window";
	goto OutOfHere;
  }  

  XMapWindow(dpy, wnd);
  XMoveWindow(dpy, wnd, -1000, -1000);
*/  
    // We are getting both commands, plugins and default plugins
  if (!ProcessResources(resFileName, True, 7, &errMsg)) goto OutOfHere;

  if (CommandCount == 0) {
  	errMsg = "No scan codes specified";
	goto OutOfHere;
  }

  if (CommandCount < maxCodes) 
    Commands = realloc(Commands, CommandCount*sizeof(CmdDescr));

  success = (Commands != NULL);  	
//  if (!InitSpecFunctions(&errMsg)) goto OutOfHere;

#ifdef OSD_SUPPORT
    InitOSD(argc, argv);
#endif

#ifdef PLUGIN_SUPPORT
    InitPluginSupport();
#endif

    pid = getpid();

//#ifdef USE_SYSLOG
    {
	char msg[81];
	struct passwd * pwd = getpwuid(getuid());
	
	if (pwd == NULL) 
	  sprintf (msg, "xhkeys [pid %d] started", pid);
	else  
	  sprintf (msg, "xhkeys [pid %d] started by '%s'", pid, pwd->pw_name); 
	  
	log_message(False, msg);

    }	
//#endif

  XHKeysPidAtom = XInternAtom(dpy, "XHKeysPid", False);

  if (XHKeysPidAtom != None) {
	Atom actual_type;
	int actual_format;
	unsigned long nitems;
	unsigned long bytes_after;
	unsigned char *prop;

	// Kill old process if any
	XGetWindowProperty(dpy, root_wnd, XHKeysPidAtom, 0, sizeof(pid), False,
       AnyPropertyType, &actual_type, &actual_format, &nitems, &bytes_after, &prop);
	if (actual_type == XHKeysPidAtom) {
	  pid_t oldPid = *(pid_t *)prop; 
	  kill(oldPid, SIGTERM);
	  while (kill(oldPid, 0) == 0) sleep (0);
	} 
	  
	// Store current pid
    XChangeProperty(dpy, root_wnd, XHKeysPidAtom, XHKeysPidAtom, 8, PropModeReplace,
	                     (unsigned char *) &pid, sizeof(pid));

  }

#ifdef USE_TIMER_HANDLER
  oldTimeHandler      = signal(SIGALRM, SIG_IGN);
  getitimer(ITIMER_REAL, &oldTimValue);
  for (i =0; i < 2; i++) JustClearTickTime(i);
#endif
  
  oldTerminateHandler = signal(SIGTERM, terminateHandler);
  oldChildExitHandler = signal(SIGCHLD, childExitHandler);
  
  WMStateAtom = XInternAtom(dpy, "WM_STATE", True);
  
  for (i =0; i < CommandCount; i++) {
    XGrabKey(dpy, Commands[i].keycode, Commands[i].modifier, wnd, False, GrabModeAsync, GrabModeAsync);
  }	

  // From this point we cant fail!
  // XSelectInput(dpy, wnd, KeyPressMask);

  // Now we can start using 'success' for a slightly different purpose
  for(;;) {
	XNextEvent(dpy, &xev);

#ifdef OSD_SUPPORT
	if (xev.type == Expose) {
	    ProcessOSDExposeEvent(&(xev.xexpose));
	    continue;
	}    
#endif
	
	if (xev.type != KeyPress) continue;

	success = False;
	keycode = xev.xkey.keycode;
	modif = xev.xkey.state;

	for (i =0; i < CommandCount &&
	      (Commands[i].keycode != keycode || Commands[i].modifier != modif); i++) ;
	if (i>=CommandCount) continue;
	  
	command = Commands[i].command;

	switch(type = Commands[i].type)
	{ 
	  case CT_INTERNAL:
		i = ProcessSpecFunction(i);
		if (i<0) goto OutOfLoop;
		success = (i>0);
		break;
		
	  case CT_EXTAPP:
		success = ProcessExtApp(command, Commands[i].type_modifier);
		break;	
		
	  case CT_KEYEVENT:
		success = ProcessWndKey(command, Commands[i].target_selection,
		                                 Commands[i].type_modifier);
		break;

	  case CT_BTNEVENT:
		success = ProcessWndButton(command, Commands[i].target_selection);
		break;
#ifdef PLUGIN_SUPPORT
	  case CT_PLUGIN:
		success = ProcessPlugin(i);
	 	break;
#endif		
	}		

    if (success == False) XBell(dpy, 50);

  } 

  
OutOfLoop:
  terminate();

//#ifdef USE_SYSLOG
    sprintf (msg, "xhkeys [pid %d] finished", pid);
    log_message(False, msg);
 	
//#endif  
  success = True;

OutOfHere:
  if (errMsg) log_message(!success, errMsg);
  return success;
}
