#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <time.h>

#include "xhkeys_conf.h"

#ifdef HAVE_SYSLOG_H
#include <syslog.h>
#endif

#ifdef USE_DMALLOC
#include <dmalloc.h>
#endif

#ifdef PLUGIN_SUPPORT
#include <dlfcn.h>	// dlsym
#endif

#include "xhkeys.h"
#include "plugin_common.h"

extern Display *dpy;
extern Window wnd;
extern Atom WMStateAtom;

extern int logMode;		// 0 - none, 1 - error, 2 - all

extern int logConsole;

const char *SpecCodeNames[] = 
{
    "WndRotate",
    "WndClose",
#ifdef PLUGIN_SUPPORT
    "KillPlugin",
#endif    
#ifdef OSD_SUPPORT
    "Version",
#endif    
     "Quit"
};

const int SpecCodeCount = sizeof(SpecCodeNames) / sizeof(char *);


#ifdef PLUGIN_SUPPORT
#include "xhkeys_plugin.h"
// resources.c
extern PluginData *Plugins;
extern int PluginCount;
#endif


//========================================================================
void log_message(Bool error, const char *msg)
{
  if (strlen(msg) > 0 &&
     (logMode == 2 || logMode == -1 || (logMode == 1 && error))) {
#ifdef USE_SYSLOG
	if (logConsole == 0)
  	  syslog (error ? LOG_ERR : LOG_INFO, "%s\n", msg);

	if (logConsole || error)
#endif  	
	  fprintf (error ? stderr : stdout, "%s\n", msg);
  }	  
}

// Dir:  >=0 bottom to top,  < 0 top to bottom

Window lookupWindow(Window cur, int dir, LookupWindowHandler handler, const void *par)
{ 
  Window tmpwnd, *children;
  unsigned int nchildren;
  int i, j;

  if ((*handler)(cur, par)) return cur;
  

  if (XQueryTree(dpy, cur, &tmpwnd, &tmpwnd, &children, &nchildren) == False)
  {	 children = NULL; nchildren = 0; }

  cur = None;
  for (i=0; i<nchildren; i++) {
    j = (dir >= 0) ? i : nchildren-i-1;
    if ((cur = lookupWindow(children[j], dir, handler, par)) != None)
		return cur;
  }  	

  if (children)	XFree(children);

  return cur;
}


//====================================================================

static Bool WndByWMClass(Window cur, const XClassHint *xch_needed)
{
  XClassHint   xch;
  
  return XGetClassHint(dpy, cur, &xch) &&
		 (strcmp(xch.res_name, xch_needed->res_name) == 0) &&	  
		 (strcmp(xch.res_class, xch_needed->res_class) == 0);
}


Window findWindowByWMClass(Window main_wnd,
                           const char *app_name, const char *class_name)
{
  XClassHint xch;
  
  xch.res_name = (char *) app_name;
  xch.res_class = (char *) class_name;
  
  return lookupWindow(main_wnd, -1, (LookupWindowHandler) WndByWMClass, &xch);

}

static int wmClassCount;

static Bool WndCountWMClass(Window cur, const XClassHint *xch_needed)
{
  XClassHint   xch;
  
  if ( XGetClassHint(dpy, cur, &xch) &&
	   (strcmp(xch.res_name, xch_needed->res_name) == 0) &&	  
	   (strcmp(xch.res_class, xch_needed->res_class) == 0)
	 ) wmClassCount++;
		 
  return False;
}


int countWindowsByWMClass(Window main_wnd,
                           const char *app_name, const char *class_name)
{
  XClassHint xch;
  
  xch.res_name = (char *) app_name;
  xch.res_class = (char *) class_name;
  wmClassCount = 0;  

  lookupWindow(main_wnd, -1, (LookupWindowHandler) WndCountWMClass, &xch);
  return wmClassCount;
}

//====================================================================
static Bool WndById(Window cur, Window id_needed)
{
  return (cur == id_needed);
}


Window findWindowById(Window main_wnd, Window id_needed)
{
  return lookupWindow(main_wnd, -1, (LookupWindowHandler) WndById, (const void *)id_needed);

}

//====================================================================
static Bool WndByTitle(Window cur, const char *title_needed)
{
  XTextProperty xtp;
  char **list;
  int list_count;
  Bool result = False;
  
  		
  if (XGetWMName(dpy, cur, &xtp) &&
	  XTextPropertyToStringList(&xtp, &list, &list_count)) {
	  result = (list_count > 0 && strcmp(list[0], title_needed) == 0); 
	  XFreeStringList(list);		
  }  	 

  return result;
}


Window findWindowByTitle(Window main_wnd,  const char *title)
{
  return lookupWindow(main_wnd, -1, (LookupWindowHandler) WndByTitle, title);

}
//====================================================================

Bool  GetWindowState(Window window, long *state)
{
    Atom actual_type;
    int actual_format;
    unsigned long nitems, bytes_after;
    unsigned char *data;
    Bool success;

    XGetWindowProperty(dpy, window, WMStateAtom, 0,
                             2, False, WMStateAtom,
                             &actual_type,  &actual_format,
                             &nitems, &bytes_after, &data);

    success = (actual_type == WMStateAtom && nitems > 0);
//	success = (actual_type != None && nitems > 0);
    if (success) *state = *(long *)data; 
    XFree(data);
    return success;
}

static Bool IsVisibleWindow(Window cur, const void *dummy )
{
  long state;
  
  return GetWindowState(cur, &state) && (state >= 0);
  
}

// Find top window having WM_STATE = state
Window findVisibleWindow(Window main_wnd, int dir)
{
  long state;
  return lookupWindow(main_wnd, dir, IsVisibleWindow, &state);

}


static Bool IsClientWindow(Window cur, const void *dummy )
{
  long state;
  return GetWindowState(cur, &state);
}


// Find top window that has WM_STATE property
Window findClientWindow(Window main_wnd, int dir)
{
  return lookupWindow(main_wnd, dir, IsClientWindow, NULL);

}


// return 0 - error, 1 - ok, 2 - timeout
short getMouseEvent(XEvent *xevPtr, unsigned int timeout, char **errMsg)
{
  time_t timeCurrent, timeToStop;
  XEvent xev;
  Bool rc = False;


  if(XGrabPointer(dpy, wnd, True, ButtonPressMask, GrabModeAsync, 
                 GrabModeAsync, None, None, CurrentTime) != 0) {
	*errMsg = "Grab pinter error";
	return 0;
  }

//  XSelectInput(dpy, wnd, ButtonPressMask);
  timeCurrent = clock();
  timeToStop = timeCurrent + timeout * CLOCKS_PER_SEC;
  rc = 2;  		// Awaiting timeout

  for(;timeCurrent < timeToStop; timeCurrent = clock())
  {
	if (XPending(dpy) == 0) {
	  sleep(0);
	  continue;
	}  

	XNextEvent(dpy, &xev);
	if (xev.type == ButtonPress) {
	  memcpy(xevPtr, &xev, sizeof(XKeyEvent));
	  rc = 1;
	  break;
	}  
  }

  XUngrabPointer(dpy,  CurrentTime);
  XSync(dpy, True);
  return rc;
}


#ifndef HAVE_STRSEP
char *strsep(char **stringp, const char *delim)
{
  char *start, *sep;;

  start = *stringp;
  sep = strpbrk(start, delim);
  if (sep != NULL) *sep++ = '\0';
  *stringp = sep;

  return start;
}
#endif

#ifdef PLUGIN_SUPPORT
const char *PluginDirs[] = {".", PLUGINDIR};
//	{".", "/usr/local/lib/xhkeys", "/usr/X11R6/lib/xhkeys",
//	      "/usr/lib/xhkeys", "/lib/xhkeys", NULL};

#if defined(CDROM_SUPPORT) || defined(MIXER_SUPPORT)
const char *DefaultPluginNames[] =
{
# ifdef CDROM_SUPPORT
	"xhkeys_cdaudio", "cda",
# endif		
# ifdef MIXER_SUPPORT    
	"xhkeys_mixer",  "vol",
# endif		
	NULL    
};
#endif

int FindPluginByLineNo(unsigned int lineNo)
{
    int i;
    
    for (i=0; i<PluginCount && Plugins[i].lineno != lineNo;  i++);
    if (i >= PluginCount) i= -1;

    return i;
}

int FindPluginByName(const char *pluginName)
{
    int i;
    const char *name;

    if (*pluginName == '\0') return -1;

    // First we look up the alias    
    for (i=0; i<PluginCount; i++) {
	if (Plugins[i].name == NULL) continue;
	
	name = Plugins[i].shortName;    
	if (name != NULL && strcasecmp(name, pluginName) == 0)
	    return i;
    }

    // If failed, look up plugin name    
    for (i=0; i<PluginCount; i++) {
	name = Plugins[i].name;
	if (name != NULL && strcmp(name, pluginName) == 0)
	    return i;
    }	    

    return -1;
}    



Bool validatePlugin(const char *fileName, PLUGIN_HANDLE handle, char *msg, int msgLen)
{
    static const char *requiredFuncNames[] = 
            {"process", "getopcount", "getopdata", NULL};

    const char **funcNamePtr, *funcName;

    funcNamePtr = requiredFuncNames;
    
    while((funcName = *funcNamePtr++) != NULL) {
	if (dlsym(handle, funcName) == NULL) {
	    sprintf(msg, "Plugin '%s' doesn't have '%s' function", fileName, funcName);
	    return False;
	}
    }

    return True;
}

Bool GetPluginOpnameInfo(const PluginData *plData,
                          const char *opName, int *opCodePtr, int *opFlagsPtr)
{
    PLUGIN_HANDLE handle;
    XHKEYS_PLUGIN_GETOPDATA *getopdataPtr;
    XHKEYS_PLUGIN_GETOPCOUNT *getopcountPtr;
    int opCount, opCode, opFlags;    
    char opNameTmp[XHKEYS_PLUGIN_OPNAME_LENGTH];

    handle = plData->handle;    
    getopcountPtr = (XHKEYS_PLUGIN_GETOPCOUNT *) dlsym(handle, "getopcount");
    getopdataPtr = (XHKEYS_PLUGIN_GETOPDATA *) dlsym(handle, "getopdata");
    if (getopcountPtr == NULL || getopdataPtr == NULL)  return False;
        
    opCount = (*getopcountPtr)();

    opFlags = 0;

    for (opCode = 0; opCode < opCount; opCode++)
    {
	if ((*getopdataPtr)(opCode, opNameTmp, &opFlags) &&
	   strcasecmp(opName, opNameTmp) == 0) break;
    }

    if (opCode >= opCount) return False;
    
    if (opCodePtr != NULL) *opCodePtr = opCode;
    if (opFlagsPtr != NULL) *opFlagsPtr = opFlags;
    return True;
}

#endif

