/*
 * SPDX-FileCopyrightText: 2020 Stalwart Labs Ltd <hello@stalw.art>
 *
 * SPDX-License-Identifier: AGPL-3.0-only OR LicenseRef-SEL
 */

use serde::{Deserialize, Serialize};

use crate::compiler::{
    grammar::{instruction::CompilerState, Capability, Comparator},
    lexer::{word::Word, Token},
    CompileError, Value,
};

use crate::compiler::grammar::{test::Test, MatchType};

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub(crate) struct TestNotifyMethodCapability {
    pub comparator: Comparator,
    pub match_type: MatchType,
    pub notification_uri: Value,
    pub notification_capability: Value,
    pub key_list: Vec<Value>,
    pub is_not: bool,
}

#[derive(Debug, Clone, PartialEq, Eq, Serialize, Deserialize)]
pub(crate) struct TestValidNotifyMethod {
    pub notification_uris: Vec<Value>,
    pub is_not: bool,
}

impl CompilerState<'_> {
    pub(crate) fn parse_test_valid_notify_method(&mut self) -> Result<Test, CompileError> {
        Ok(Test::ValidNotifyMethod(TestValidNotifyMethod {
            notification_uris: self.parse_strings(false)?,
            is_not: false,
        }))
    }

    pub(crate) fn parse_test_notify_method_capability(&mut self) -> Result<Test, CompileError> {
        let mut match_type = MatchType::Is;
        let mut comparator = Comparator::AsciiCaseMap;
        let mut notification_uri = None;
        let mut notification_capability = None;
        let mut key_list;

        loop {
            let token_info = self.tokens.unwrap_next()?;
            match token_info.token {
                Token::Tag(
                    word @ (Word::Is
                    | Word::Contains
                    | Word::Matches
                    | Word::Value
                    | Word::Count
                    | Word::Regex),
                ) => {
                    self.validate_argument(
                        1,
                        match word {
                            Word::Value | Word::Count => Capability::Relational.into(),
                            Word::Regex => Capability::Regex.into(),
                            Word::List => Capability::ExtLists.into(),
                            _ => None,
                        },
                        token_info.line_num,
                        token_info.line_pos,
                    )?;

                    match_type = self.parse_match_type(word)?;
                }
                Token::Tag(Word::Comparator) => {
                    self.validate_argument(2, None, token_info.line_num, token_info.line_pos)?;
                    comparator = self.parse_comparator()?;
                }
                _ => {
                    if notification_uri.is_none() {
                        notification_uri = self.parse_string_token(token_info)?.into();
                    } else if notification_capability.is_none() {
                        notification_capability = self.parse_string_token(token_info)?.into();
                    } else {
                        key_list = self.parse_strings_token(token_info)?;
                        break;
                    }
                }
            }
        }
        self.validate_match(&match_type, &mut key_list)?;

        Ok(Test::NotifyMethodCapability(TestNotifyMethodCapability {
            key_list,
            match_type,
            comparator,
            notification_uri: notification_uri.unwrap(),
            notification_capability: notification_capability.unwrap(),
            is_not: false,
        }))
    }
}
