#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <lufs/proto.h>
#include <lufs/fs.h>

#include "gnet_lib.h"
#include "gnet_xfer.h"
#include "gnet_search.h"

static inline int
set_nb(int fd){
    unsigned long flags;

    fcntl(fd, F_GETFL, &flags);
    flags |= O_NONBLOCK;
    fcntl(fd, F_SETFL, &flags);

    if((fcntl(fd, F_GETFL, &flags) < 0) || !(flags & O_NONBLOCK)){
	WARN("could not set non-blocking mode: %s", strerror(errno));
	return -1;
    }

    return 0;
}

static inline int
get_sock_error(int fd){
    int res;
    socklen_t len = sizeof(int);

    if(getsockopt(fd, SOL_SOCKET, SO_ERROR, &res, &len) < 0){
	WARN("getsockopt failed: %s", strerror(errno));
	return -1;
    }
    return res;
}

void
gnet_xfer_shutdown(struct gnet_locator *loc){
    struct gnet_xfer *xfer = loc->xfer;

    if(xfer){
	close(xfer->sock);
	free(xfer);
    }
}

static struct gnet_xfer*
create_xfer(){
    struct gnet_xfer *xfer;

    TRACE("creating xfer");

    xfer = malloc(sizeof(struct gnet_xfer));

    if(xfer){
	memset(xfer, 0, sizeof(struct gnet_xfer));
    }

    return xfer;
}

static int
connect_direct(struct gnet *gnet, struct gnet_locator *loc, int time_out){
    struct gnet_xfer *xfer = loc->xfer;
    struct sockaddr_in addr;
    int res;

    TRACE("connecting to %s:%d", inet_ntoa(*(struct in_addr*)loc->ip), loc->port);

    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_port = htons(loc->port);
    memcpy(&addr.sin_addr.s_addr, loc->ip, 4);

    if(xfer->sock){
	close(xfer->sock);
	xfer->connected = 0;
    }

    if((xfer->sock = socket(PF_INET, SOCK_STREAM, 0)) < 0){
	WARN("socket call failed: %s", strerror(errno));
	goto fail;
    }

    if(set_nb(xfer->sock) < 0)
	goto fail_sock;

    if((connect(xfer->sock, (struct sockaddr*)&addr, sizeof(struct sockaddr_in)) < 0) && (errno != EINPROGRESS)){
	WARN("connect call failed: %s", strerror(errno));
	goto fail_sock;
    }

    if(lu_check_to(0, xfer->sock, time_out) < 0)
	goto fail_sock;

    if((res = get_sock_error(xfer->sock)) != 0){
	WARN("connect failed: %s", strerror(res));
	goto fail_sock;
    }

    TRACE("connected");

    return 0;

  fail_sock:
    close(xfer->sock);
    xfer->sock = 0;
  fail:
    return -1;
}

static int
connect_push(struct gnet *gnet, struct gnet_locator *loc, int time_out){
    struct sockaddr_in addr;
    struct gnet_xfer *xfer = loc->xfer;
    socklen_t len;
    int ssock, res = -1;
    int lfs = 0;
    char c;

    WARN("push-connecting...");

    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family = AF_INET;
    addr.sin_port = 0;
    addr.sin_addr.s_addr = INADDR_ANY;

    if(xfer->sock){
	close(xfer->sock);
	xfer->connected = 0;
    }

    if((ssock = socket(PF_INET, SOCK_STREAM, 0)) < 0){
	WARN("socket call failed: %s", strerror(errno));
	goto out;
    }

    if(set_nb(ssock) < 0)
	goto out_ssock;

    if(bind(ssock, (struct sockaddr*)&addr, sizeof(struct sockaddr_in)) < 0){
	WARN("bind call failed: %s", strerror(errno));
	goto out_ssock;
    }

    if(listen(ssock, 2) < 0){
	WARN("listen call failed: %s", strerror(errno));
	goto out_ssock;
    }

    len = sizeof(struct sockaddr_in);
    if(getsockname(ssock, (struct sockaddr*)&addr, &len) < 0){
	WARN("getsockname call failed: %s", strerror(errno));
	goto out_ssock;
    }

    TRACE("listening on %s:%hu", inet_ntoa(*(struct in_addr*)&addr.sin_addr.s_addr), ntohs(addr.sin_port));

    if(gnet_send_push(gnet, loc, ntohs(addr.sin_port)) < 0){
	TRACE("could not send push request!");
	goto out_ssock;
    }

    if(lu_check_to(ssock, 0, time_out) < 0)
	goto out_ssock;

    len = sizeof(struct sockaddr_in);
    if((xfer->sock = accept(ssock, (struct sockaddr*)&addr, &len)) < 0){
	WARN("accept call failed: %s", strerror(errno));
	goto out_ssock;
    }

    if(set_nb(xfer->sock) < 0)
	goto out_ssock;

    do{
	if(lu_atomic_read(xfer->sock, &c, 1, time_out) < 0)
	    break;

	if(c == '\n')
	    lfs++;
	else
	    lfs = 0;
    }while(lfs < 2);

    if(lfs < 2){
	WARN("could not read reply!");
	goto out_ssock;
    }

    TRACE("reply read");

    res = 0;

  out_ssock:
    close(ssock);
  out:
    return res;
}

static int
request_file(int sock, struct gnet_locator *loc, unsigned long offset, unsigned long count, int time_out, char *buf, int buflen){
    int len, i, res;
    
    if(snprintf(buf, buflen, 
		"GET /get/%lu/%s HTTP/1.1\r\nUser-Agent: Gnutella\r\nHost: %hhu.%hhu.%hhu.%hhu:%hu\r\nConnection: Keep-Alive\r\nRange: bytes=%lu-%lu\r\n\r\n",
		loc->index, loc->name,
		loc->ip[0], loc->ip[1], loc->ip[2], loc->ip[3],
		loc->port, offset, 
		(count != 0) ? (offset + count - 1) : (loc->size - 1)) >= buflen){
	WARN("buffer too small");
	return -E2BIG;
    }

    TRACE(buf);
    
    len = strlen(buf);

    if((res = lu_atomic_write(sock, buf, len, time_out)) != len)
	return res;
    
    i = 0;

    do{
	if((res = lu_atomic_read(sock, buf + i, 1, time_out)) < 1)
	    return (res < 0) ? res : -1;
	
	i++;

	if((i > 3) && !(strncmp(buf + i - 4, "\r\n\r\n", 4))){
	    buf[i] = 0;
	    break;
	}
    }while(i < buflen - 1);

    if(i >= buflen - 1){
	TRACE("reply too long");
	return -E2BIG;
    }

    if((sscanf(buf, "HTTP/1.%*c %d ", &i) != 1) && (sscanf(buf, "HTTP %d ", &i) != 1)){
	WARN("could not parse reply code: %s", buf);
	return -EPROTO;
    }

    if(i != 200){
	WARN("request rejected: %d", i);
	return -EBUSY;
    }

    TRACE("file data available");

    loc->xfer->connected = 1;
    loc->xfer->offset = offset;

    return 0;
}

static int
connect_xfer(struct gnet *gnet, struct gnet_locator *loc, int time_out){

    if(!loc->firewalled)
	if(connect_direct(gnet, loc, time_out) >= 0)
	    return 0;
    
    return connect_push(gnet, loc, time_out);
}

int
gnet_xfer_open(struct gnet *gnet, struct gnet_locator *loc, unsigned long offset, unsigned long count, int time_out){
    struct gnet_xfer *xfer = loc->xfer;
    char *buf;
    int res;
    
    if(!(buf = malloc(BUFFER_SIZE)))
	return -ENOMEM;

    if(!xfer){
	if(!(xfer = create_xfer())){
	    res = -ENOMEM;
	    goto out;
	}
	loc->xfer = xfer;
    }    

    
    if(!(xfer->connected)){
	TRACE("xfer not connected");

	if((res = connect_xfer(gnet, loc, time_out)) < 0)
	    goto out;

	if((res = request_file(xfer->sock, loc, offset, (xfer->type == TYPE_SERIAL) ? 0 : count, time_out, buf, BUFFER_SIZE)) < 0)
	    goto out;    

    }else{
	TRACE("xfer connected, %s, offset=%lu", (xfer->type == TYPE_SERIAL) ? "serial" : "random", xfer->offset);

	if(xfer->type == TYPE_SERIAL){
	    if(xfer->offset == offset){
		TRACE("keep-alive :):):)");
		xfer->last_rnd = 0;
		xfer->last_ser++;
	    }else{
		TRACE("missed, reconnecting :(");
		xfer->last_ser = 0;
		xfer->last_rnd++;

		if(xfer->last_rnd > MODE_SWITCH){
		    TRACE("switching to random mode");
		    xfer->type = TYPE_RANDOM;
		}

		if((res = connect_xfer(gnet, loc, time_out)) < 0)
		    goto out;

		if((res = request_file(xfer->sock, loc, offset, (xfer->type == TYPE_SERIAL) ? 0 : count, time_out, buf, BUFFER_SIZE)) < 0)
		    goto out;    		
	    }
	}else{
	    if(xfer->offset == offset){
		TRACE("serial hit");
		xfer->last_ser++;
		xfer->last_rnd = 0;

		if(xfer->last_ser > MODE_SWITCH){
		    TRACE("switching to serial mode");
		    xfer->type = TYPE_SERIAL;
		}
	    }else{
		TRACE("random hit");
		xfer->last_rnd++;
		xfer->last_ser = 0;
	    }
	    
	    if((res = request_file(xfer->sock, loc, offset, (xfer->type == TYPE_SERIAL) ? 0 : count, time_out, buf, BUFFER_SIZE)) < 0)
		goto out;    			    
	}
	
    }
       
       
    res = 0;

  out:
    free(buf);
    return res;
}


int
gnet_xfer_read(struct gnet *gnet, struct gnet_locator *loc, unsigned long offset, unsigned long count, char *buf){
    struct gnet_xfer *xfer;
    int res;

    if(offset >= loc->size){
	WARN("trying to seek past the end of file: %lu", offset);
	return 0;
    }

    if(offset + count >= loc->size){
	TRACE("trying to read past the end of file");
	count = loc->size - offset - 1;
    }

    if((res = gnet_xfer_open(gnet, loc, offset, count, TIME_OUT)) < 0)
	return res;

    xfer = loc->xfer;

    if((res = lu_atomic_read(xfer->sock, buf, count, TIME_OUT)) < 0)
	return res;

    xfer->offset += res;

    TRACE("read %lu bytes", count);

    return res;
}
