/* $Id: op_logdump.c,v 1.11 2004/04/03 19:57:32 andrewbaker Exp $ */
/* 
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#include "config.h"

#include <stdio.h>
#include <string.h>
#ifdef SOLARIS
    #include <strings.h>
#endif
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "barnyard.h"
#include "plugbase.h"
#include "op_plugbase.h"
#include "util.h"
#include "mstring.h"
#include "sid.h"
#include "classification.h"
#include "op_decode.h"
#include "input-plugins/dp_log.h"

typedef struct _LogDumpData
{
    char *filename;
    char *filepath;
    FILE *file;
} LogDumpData;

/* Output plug-in API functions */
static int OpLogDump_Setup(OutputPlugin *, char *);
static int OpLogDump_Exit(OutputPlugin *);
static int OpLogDump_Start(OutputPlugin *, void *);
static int OpLogDump_Stop(OutputPlugin *);
static int OpLogDump(void *, void *);
static int OpLogDump_LogConfig(OutputPlugin *);

/* Internal functions */
static void ParseLogDumpArgs(char *, OutputPlugin *);
static FILE *OpenLogFile(char *);
static int PrintIPHeader(FILE *, Packet *);
static void PrintTCPHeader(FILE *, Packet *);
static void PrintUDPHeader(FILE *, Packet *);
static void PrintICMPHeader(FILE *, Packet *);
static void PrintTcpOptions(FILE *, Packet *);
static void PrintIpOptions(FILE *, Packet *);

/* init routine makes this processor available for dataprocessor directives */
void OpLogDump_Init()
{
    OutputPlugin *outputPlugin;
    
    outputPlugin = RegisterOutputPlugin("log_dump", "log");
    outputPlugin->setupFunc = OpLogDump_Setup;
    outputPlugin->exitFunc = OpLogDump_Exit;
    outputPlugin->startFunc = OpLogDump_Start;
    outputPlugin->stopFunc = OpLogDump_Stop;
    outputPlugin->outputFunc = OpLogDump;
    outputPlugin->logConfigFunc = OpLogDump_LogConfig;
}

int OpLogDump_LogConfig(OutputPlugin *outputPlugin)
{
    LogDumpData *data = NULL;

    if(!outputPlugin || !outputPlugin->data)
        return -1;
    
    data = (LogDumpData *)outputPlugin->data;

    LogMessage("OpLogDump configured\n");
    LogMessage("  Filename: %s\n", data->filename);
    return 0;
}

/* link the output processor functions to an output function node */
int OpLogDump_Setup(OutputPlugin *outputPlugin, char *args)
{
    ParseLogDumpArgs(args, outputPlugin);

    return 0;
}

int OpLogDump_Exit(OutputPlugin *outputPlugin)
{
    LogDumpData *data = (LogDumpData *)outputPlugin->data;
    if(data != NULL && data->filename != NULL)
        free(data->filename);
    return 0;
}

int OpLogDump_Start(OutputPlugin *outputPlugin, void *spool_header)
{
    LogDumpData *data = (LogDumpData *)outputPlugin->data;
    
    if(data == NULL)
        FatalError("ERROR: Unable to find context for log dump startup!\n");

    if(pv.verbose >= 2)
        OpLogDump_LogConfig(outputPlugin);
    
    data->filepath = ProcessFileOption(data->filename);

    data->file = OpenLogFile(data->filepath);
    
    return 0;
}

int OpLogDump_Stop(OutputPlugin *outputPlugin)
{
    LogDumpData *data = (LogDumpData *)outputPlugin->data;

    if(data != NULL)
    {
        fflush(data->file);
        fclose(data->file);
    }
    if(data->filepath)
        free(data->filepath);
    data->filepath = NULL;

    return 0;
}


int OpLogDump(void *data, void *logdata)
{
    UnifiedLogRecord *ad = (UnifiedLogRecord *)logdata;
    LogDumpData *afd = (LogDumpData *)data;
    Packet p;
    Sid *tmp = NULL;
    ClassType *ct = NULL;

    tmp = GetSid(ad->log.event.sig_generator, ad->log.event.sig_id);
    ct = GetClassType(ad->log.event.classification);

    fprintf(afd->file, "[**] [%d:%d:%d] %s [**]\n[Classification: %s] "
            "[Priority: %d]\n", ad->log.event.sig_generator, 
            ad->log.event.sig_id, ad->log.event.sig_rev, 
            tmp != NULL?tmp->msg:"ALERT", 
            ct != NULL?ct->name:"Unknown", ad->log.event.priority);

    PrintXref(ad->log.event.sig_generator, ad->log.event.sig_id, afd->file);
    fprintf(afd->file, "Event ID: %lu     Event Reference: %lu\n", 
            (unsigned long) ad->log.event.event_id, 
            (unsigned long) ad->log.event.event_reference);

    if(ad->log.pkth.caplen != 0)
    {
        if(DecodePacket(&p, &ad->log.pkth, ad->pkt + 2) != 0)
        {
            fprintf(afd->file, "Linktype %i not decoded.  Raw packet dumped\n", 
                    linktype);
            PrintNetData(afd->file, ad->pkt + 2, ad->log.pkth.caplen);
            ClearDumpBuf();
        }
        else
        {
            if(p.iph != NULL)
            {
                if(PrintIPHeader(afd->file, &p) == -1)
                    goto exit;

                switch(p.iph->ip_proto)
                {
                    case IPPROTO_TCP:
                        PrintTCPHeader(afd->file, &p);
                        break;

                    case IPPROTO_UDP:
                        PrintUDPHeader(afd->file, &p);
                        break;

                    case IPPROTO_ICMP:
                        PrintICMPHeader(afd->file, &p);
                        break;

                    default:
                        break;
                }
            }
            if(p.dsize)
            {
                PrintNetData(afd->file, p.data, p.dsize);
                ClearDumpBuf();
            }
        }
    }
exit:

    fprintf(afd->file, "=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+\n\n");

    fflush(afd->file);

    return 0;
}

/* initialize the output processor for this particular instantiation */
void ParseLogDumpArgs(char *args, OutputPlugin *outputPlugin)
{
    char **toks;
    int num_toks;
    LogDumpData *data;

    data = (LogDumpData *)SafeAlloc(sizeof(LogDumpData));

    if(args == NULL)
    {
       data->filename = strdup("dump.log");
       outputPlugin->data = (LogDumpData *) data;
       return;
    }

    toks = mSplit(args, " ", 2, &num_toks, 0);
    
    data->filename = strdup(toks[0]);
    
    FreeToks(toks, num_toks);
    outputPlugin->data = (LogDumpData *) data;
    return;
}



int PrintIPHeader(FILE * fp, Packet * p)
{
    char timestamp[256];

    if(p->iph == NULL)
    {
        fprintf(fp, "IP header truncated\n");
        return -1;
    }
    
    if(RenderTimeval(&p->pkth->ts, timestamp, 256) == -1)
    {
        LogMessage("ERROR: OpLogDump failed to render timeval\n");
        return -1;
    }

    fprintf(fp, "%s ", timestamp);

    if(p->pkt_flags & PKT_FRAG_FLAG)
    {
        /* just print the straight IP header */
        fputs(inet_ntoa(p->iph->ip_src), fp);
        fwrite(" -> ", 4, 1, fp);
        fputs(inet_ntoa(p->iph->ip_dst), fp);
    }
    else
    {
        if(p->iph->ip_proto != IPPROTO_TCP && p->iph->ip_proto != IPPROTO_UDP)
        {
            /* just print the straight IP header */
            fputs(inet_ntoa(p->iph->ip_src), fp);
            fwrite(" -> ", 4, 1, fp);
            fputs(inet_ntoa(p->iph->ip_dst), fp);
        }
        else
        {
            /* print the header complete with port information */
            fputs(inet_ntoa(p->iph->ip_src), fp);
            fprintf(fp, ":%d -> ", p->sp);
            fputs(inet_ntoa(p->iph->ip_dst), fp);
            fprintf(fp, ":%d", p->dp);
        }
    }

    fputc('\n', fp);

    fprintf(fp, "%s TTL:%d TOS:0x%X ID:%d IpLen:%d DgmLen:%d",
            protocol_names[p->iph->ip_proto],
            p->iph->ip_ttl,
            p->iph->ip_tos,
            ntohs(p->iph->ip_id),
            IP_HLEN(p->iph) << 2, ntohs(p->iph->ip_len));

    /* print the reserved bit if it's set */
    if(p->pkt_flags & PKT_RB_FLAG)
    {
        fprintf(fp, " RB");
    }

    if(p->pkt_flags & PKT_DF_FLAG)
        fprintf(fp, " DF");

    if(p->pkt_flags & PKT_MF_FLAG)
        fprintf(fp, " MF");

    fputc('\n', fp);

    /* print IP options */
    if(p->ip_option_count != 0)
    {
        PrintIpOptions(fp, p);
    }

    /* print fragment info if necessary */
    if(p->pkt_flags & PKT_FRAG_FLAG)
    {
        fprintf(fp, "Frag Offset: 0x%X   Frag Size: 0x%X",
                (p->frag_offset & 0xFFFF), p->dsize);
        fputc('\n', fp);
    }
    return 0;
}


void PrintTCPHeader(FILE * fp, Packet * p)
{
    char tcpFlags[9];

    if(p->tcph == NULL)
    {
        fprintf(fp, "TCP header truncated\n");
        return;
    }
    /* print TCP flags */
    CreateTCPFlagString(p, tcpFlags);
    fwrite(tcpFlags, 8, 1, fp); /* We don't care about the NULL */

    /* print other TCP info */
    fprintf(fp, " Seq: 0x%lX  Ack: 0x%lX  Win: 0x%X  TcpLen: %d",
            (u_long) ntohl(p->tcph->th_seq),
            (u_long) ntohl(p->tcph->th_ack),
            ntohs(p->tcph->th_win), TCP_OFFSET(p->tcph) << 2);

    if((p->tcph->th_flags & TH_URG) != 0)
    {
        fprintf(fp, "  UrgPtr: 0x%X\n", (u_int16_t) ntohs(p->tcph->th_urp));
    }
    else
    {
        fputc((int) '\n', fp);
    }

    /* dump the TCP options */
    if(p->tcp_option_count != 0)
    {
        PrintTcpOptions(fp, p);
    }
}


void PrintUDPHeader(FILE * fp, Packet * p)
{

    if(p->udph == NULL)
    {
        fprintf(fp, "UDP header truncated\n");
        return;
    }
    /* not much to do here... */
    fprintf(fp, "Len: %d\n", ntohs(p->udph->uh_len));
}



void PrintICMPHeader(FILE * fp, Packet * p)
{

    if(p->icmph == NULL)
    {
        fprintf(fp, "ICMP header truncated\n");
        return;
    }

    fprintf(fp, "Type:%d  Code:%d  ", p->icmph->icmp_type, 
            p->icmph->icmp_code);

    switch(p->icmph->icmp_type)
    {
        case ICMP_ECHOREPLY:
            fprintf(fp, "ID:%d  Seq:%d  ", p->icmph->icmp_id, 
                    p->icmph->icmp_seq);
            fwrite("ECHO REPLY\n", 10, 1, fp);
            break;

        case ICMP_DEST_UNREACH:
            fwrite("DESTINATION UNREACHABLE: ", 25, 1, fp);
            switch(p->icmph->icmp_code)
            {
                case ICMP_NET_UNREACH:
                    fwrite("NET UNREACHABLE", 15, 1, fp);
                    break;

                case ICMP_HOST_UNREACH:
                    fwrite("HOST UNREACHABLE", 16, 1, fp);
                    break;

                case ICMP_PROT_UNREACH:
                    fwrite("PROTOCOL UNREACHABLE", 20, 1, fp);
                    break;

                case ICMP_PORT_UNREACH:
                    fwrite("PORT UNREACHABLE", 16, 1, fp);
                    break;

                case ICMP_FRAG_NEEDED:
                    fwrite("FRAGMENTATION NEEDED", 20, 1, fp);
                    break;

                case ICMP_SR_FAILED:
                    fwrite("SOURCE ROUTE FAILED", 19, 1, fp);
                    break;

                case ICMP_NET_UNKNOWN:
                    fwrite("NET UNKNOWN", 11, 1, fp);
                    break;

                case ICMP_HOST_UNKNOWN:
                    fwrite("HOST UNKNOWN", 12, 1, fp);
                    break;

                case ICMP_HOST_ISOLATED:
                    fwrite("HOST ISOLATED", 13, 1, fp);
                    break;

                case ICMP_NET_ANO:
                    fwrite("NET ANO", 7, 1, fp);
                    break;

                case ICMP_HOST_ANO:
                    fwrite("HOST ANO", 8, 1, fp);
                    break;

                case ICMP_NET_UNR_TOS:
                    fwrite("NET UNREACHABLE TOS", 19, 1, fp);
                    break;

                case ICMP_HOST_UNR_TOS:
                    fwrite("HOST UNREACHABLE TOS", 20, 1, fp);
                    break;

                case ICMP_PKT_FILTERED:
                    fwrite("PACKET FILTERED", 15, 1, fp);
                    break;

                case ICMP_PREC_VIOLATION:
                    fwrite("PREC VIOLATION", 14, 1, fp);
                    break;

                case ICMP_PREC_CUTOFF:
                    fwrite("PREC CUTOFF", 12, 1, fp);
                    break;

                default:
                    fwrite("UNKNOWN", 7, 1, fp);
                    break;

            }
/*            {
                Packet orig_p;
                int orig_iph_size;

                bzero((char *) &orig_p, sizeof(Packet));
                orig_p.iph = p->orig_iph;
                orig_p.tcph = p->orig_tcph;
                orig_p.udph = p->orig_udph;
                orig_p.sp = p->orig_sp;
                orig_p.dp = p->orig_dp;

                if(orig_p.iph != NULL)
                {
                    orig_iph_size = orig_p.iph->ip_hlen << 2;

                    fprintf(fp, "\n** ORIGINAL DATAGRAM DUMP:\n");
                    fprintf(fp, "** END OF DUMP");
                }
                else
                {
                    fprintf(fp, "\nORIGINAL DATAGRAM TRUNCATED");
                }
            }*/

            break;

        case ICMP_SOURCE_QUENCH:
            fwrite("SOURCE QUENCH", 13, 1, fp);
            break;

        case ICMP_REDIRECT:
            fwrite("REDIRECT", 8, 1, fp);
            break;
        case ICMP_ECHO:
            fprintf(fp, "ID:%d   Seq:%d  ", p->icmph->icmp_id, p->icmph->icmp_seq);
            fwrite("ECHO", 4, 1, fp);
            break;

        case ICMP_TIME_EXCEEDED:
            fwrite("TTL EXCEEDED", 12, 1, fp);
            break;

        case ICMP_PARAMETERPROB:
            fwrite("PARAMETER PROBLEM", 17, 1, fp);
            break;

        case ICMP_TIMESTAMP:
            fwrite("TIMESTAMP REQUEST", 17, 1, fp);
            break;

        case ICMP_TIMESTAMPREPLY:
            fwrite("TIMESTAMP REPLY", 15, 1, fp);
            break;

        case ICMP_INFO_REQUEST:
            fwrite("INFO REQUEST", 12, 1, fp);
            break;

        case ICMP_INFO_REPLY:
            fwrite("INFO REPLY", 10, 1, fp);
            break;

        case ICMP_ADDRESS:
            fwrite("ADDRESS REQUEST", 15, 1, fp);
            break;

        case ICMP_ADDRESSREPLY:
            fwrite("ADDRESS REPLY", 13, 1, fp);
            break;

        default:
            fwrite("UNKNOWN", 7, 1, fp);

            break;
    }

    putc('\n', fp);

}



void PrintIpOptions(FILE * fp, Packet * p)
{
    int i;
    int j;
    u_long init_offset;
    u_long print_offset;

    init_offset = ftell(fp);

    if(!p->ip_option_count || p->ip_option_count > 40)
        return;

    fprintf(fp, "IP Options (%d) => ", p->ip_option_count);

    for(i = 0; i < (int) p->ip_option_count; i++)
    {
        print_offset = ftell(fp);

        if((print_offset - init_offset) > 60)
        {
            fwrite("\nIP Options => ", 15, 1, fp);
            init_offset = ftell(fp);
        }
            
        switch(p->ip_options[i].code)
        {
            case IPOPT_RR:
                fwrite("RR ", 3, 1, fp);
                break;

            case IPOPT_EOL:
                fwrite("EOL ", 4, 1, fp);
                break;

            case IPOPT_NOP:
                fwrite("NOP ", 4, 1, fp);
                break;

            case IPOPT_TS:
                fwrite("TS ", 3, 1, fp);
                break;

            case IPOPT_SECURITY:
                fwrite("SEC ", 4, 1, fp);
                break;

            case IPOPT_LSRR:
            case IPOPT_LSRR_E:
                fwrite("LSRR ", 5, 1, fp);
                break;

            case IPOPT_SATID:
                fwrite("SID ", 4, 1, fp);
                break;

            case IPOPT_SSRR:
                fwrite("SSRR ", 5, 1, fp);
                break;

            case IPOPT_RTRALT:
                fwrite("RTRALT ", 7, 1, fp);
                break;    

            default:
                fprintf(fp, "Opt %d: ", p->ip_options[i].code);

                if(p->ip_options[i].len)
                {
                    for(j = 0; j < (int)(p->ip_options[i].len - 1); j += 2)
                    {
                        fprintf(fp, "%02X%02X ", p->ip_options[i].data[j], 
                                p->ip_options[i].data[j + 1]);
                    }
                }
                break;
        }
    }

    fwrite("\n", 1, 1, fp);
}


void PrintTcpOptions(FILE * fp, Packet * p)
{
    int i;
    int j;
    u_char tmp[5];
    u_long init_offset;
    u_long print_offset;

    init_offset = ftell(fp);

    fprintf(fp, "TCP Options (%d) => ", p->tcp_option_count);

    if(p->tcp_option_count > 40 || !p->tcp_option_count)
        return;

    for(i = 0; i < (int) p->tcp_option_count; i++)
    {
        print_offset = ftell(fp);

        if((print_offset - init_offset) > 60)
        {
            fwrite("\nTCP Options => ", 16, 1, fp);
            init_offset = ftell(fp);
        }
            
        switch(p->tcp_options[i].code)
        {
            case TCPOPT_MAXSEG:
                bzero((char *) tmp, 5);
                fwrite("MSS: ", 5, 1, fp);
                strncpy((char*)tmp, (char*) (p->tcp_options[i].data), 2);
                fprintf(fp, "%u ", EXTRACT_16BITS(tmp));
                break;

            case TCPOPT_EOL:
                fwrite("EOL ", 4, 1, fp);
                break;

            case TCPOPT_NOP:
                fwrite("NOP ", 4, 1, fp);
                break;

            case TCPOPT_WSCALE:
                fprintf(fp, "WS: %u ", p->tcp_options[i].data[0]);
                break;

            case TCPOPT_SACK:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 2);
                fprintf(fp, "Sack: %u@", EXTRACT_16BITS(tmp));
                bzero((char *) tmp, 5);
                memcpy(tmp, (p->tcp_options[i].data) + 2, 2);
                fprintf(fp, "%u ", EXTRACT_16BITS(tmp));
                break;

            case TCPOPT_SACKOK:
                fwrite("SackOK ", 7, 1, fp);
                break;

            case TCPOPT_ECHO:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "Echo: %u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_ECHOREPLY:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "Echo Rep: %u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_TIMESTAMP:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "TS: %u ", EXTRACT_32BITS(tmp));
                bzero((char *) tmp, 5);
                memcpy(tmp, (p->tcp_options[i].data) + 4, 4);
                fprintf(fp, "%u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_CC:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "CC %u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_CCNEW:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "CCNEW: %u ", EXTRACT_32BITS(tmp));
                break;

            case TCPOPT_CCECHO:
                bzero((char *) tmp, 5);
                memcpy(tmp, p->tcp_options[i].data, 4);
                fprintf(fp, "CCECHO: %u ", EXTRACT_32BITS(tmp));
                break;

            default:
                if(p->tcp_options[i].len > 2)
                {
                    fprintf(fp, "Opt %d (%d): ", p->tcp_options[i].code,
                            (int) p->tcp_options[i].len);

                    for(j = 0; j < (int) (p->tcp_options[i].len - 1); j += 2)
                    {
                        fprintf(fp, "%02X%02X ", p->tcp_options[i].data[j], 
                                p->tcp_options[i].data[j + 1]);
                    }
                }
                else
                {
                    fprintf(fp, "Opt %d ", p->tcp_options[i].code);
                }
                break;
        }
    }

    fwrite("\n", 1, 1, fp);
}





FILE *OpenLogFile(char *filename)
{
     FILE *tmp;   

    if((tmp = fopen(filename, "a+")) == NULL)
    {
        FatalError("ERROR => fopen(%s) failed: %s\n", filename,
                    strerror(errno));
    }
 
    return tmp;
}


