/* $Id: op_alert_csv.c,v 1.9 2004/03/16 04:18:20 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

/* 
 * BUGS:
 *
 *   Strings are not properly escaped.  (embedded '"' will cause bad things)
 * 
 * TODO:
 *
 * Allow multiple timestamp printing formats
 *
 * Suggestions?
 *
 *
 * Keyword list:
 *  sig_gen         - signature generator
 *  sig_id          - signature id
 *  sig_rev         - signatrue revision
 *  sid             - SID triplet
 *  class           - class id
 *  classname       - textual name of class
 *  priority        - priority id
 *  event_id        - event id
 *  event_reference - event reference
 *  ref_tv_sec      - reference seconds
 *  ref_tv_usec     - reference microseconds
 *  tv_sec          - event seconds
 *  tv_usec         - event microseconds
 *  timestamp       - prettified timestamp (2001-01-01 01:02:03) in UTC
 *  src             - src address as a u_int32_t
 *  srcip           - src address as a dotted quad
 *  dst             - dst address as a u_int32_t
 *  dstip           - dst address as a dotted quad
 *  sport_itype     - source port or ICMP type (or 0)
 *  sport           - source port (if UDP or TCP)
 *  itype           - ICMP type (if ICMP)
 *  dport_icode     - dest port or ICMP code (or 0)
 *  dport           - dest port
 *  icode           - ICMP code (if ICMP)
 *  proto           - protocol number
 *  protoname       - protocol name
 *  flags           - flags from UnifiedAlertRecord
 *  msg             - message text
 *  hostname        - hostname (from barnyard.conf)
 *  interface       - interface (from barnyard.conf)
 */ 
 

/*  I N C L U D E S  *****************************************************/
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>

#include "strlcpyu.h"
#include "ConfigFile.h"
#include "plugbase.h"
#include "op_plugbase.h"
#include "mstring.h"
#include "util.h"
#include "sid.h"
#include "classification.h"
#include "input-plugins/dp_alert.h"
#include "barnyard.h"


/* KEYWORD DEFINES */
#define CSV_SIG_GEN          1 
#define CSV_SIG_ID           2 
#define CSV_SIG_REV          3 
#define CSV_SID              4 
#define CSV_CLASS            5
#define CSV_CLASSNAME        6 
#define CSV_PRIORITY         7 
#define CSV_NONE             8
#define CSV_EVENT_ID         9 
#define CSV_EVENT_REFERENCE 10 
#define CSV_REF_TV_SEC      11 
#define CSV_REF_TV_USEC     12 
#define CSV_TV_SEC          13 
#define CSV_TV_USEC         14 
#define CSV_TIMESTAMP       15 
#define CSV_SRC             16 
#define CSV_SRCIP           17 
#define CSV_DST             18 
#define CSV_DSTIP           19 
#define CSV_SPORT_ITYPE     20 
#define CSV_SPORT           21 
#define CSV_ITYPE           22 
#define CSV_DPORT_ICODE     23 
#define CSV_DPORT           24 
#define CSV_ICODE           25 
#define CSV_PROTO           26 
#define CSV_PROTONAME       27 
#define CSV_FLAGS           28
#define CSV_MSG             29
#define CSV_HOSTNAME        30
#define CSV_INTERFACE       31
        
#define MAX_KEY_VALUE 31

struct keyword_value
{
    char *keyword;
    int value;
};

struct keyword_value format_keys[] =
{
    { "", 0 },
    { "sig_gen", CSV_SIG_GEN },
    { "sig_id", CSV_SIG_ID },
    { "sig_rev", CSV_SIG_REV },
    { "sid", CSV_SID },
    { "class", CSV_CLASS },
    { "classname", CSV_CLASSNAME },
    { "priority", CSV_PRIORITY },
    { "",        CSV_NONE },
    { "event_id",CSV_EVENT_ID },
    { "event_reference", CSV_EVENT_REFERENCE },
    { "ref_tv_sec", CSV_REF_TV_SEC },
    { "ref_tv_usec", CSV_REF_TV_USEC },
    { "tv_sec", CSV_TV_SEC },
    { "tv_usec", CSV_TV_USEC },
    { "timestamp", CSV_TIMESTAMP },
    { "src", CSV_SRC },
    { "srcip", CSV_SRCIP },
    { "dst", CSV_DST },
    { "dstip", CSV_DSTIP },
    { "sport_itype", CSV_SPORT_ITYPE },
    { "sport", CSV_SPORT },
    { "itype", CSV_ITYPE },
    { "dport_icode", CSV_DPORT_ICODE },
    { "dport", CSV_DPORT },
    { "icode", CSV_ICODE },
    { "proto", CSV_PROTO },
    { "protoname", CSV_PROTONAME },
    { "flags", CSV_FLAGS },
    { "msg", CSV_MSG },
    { "hostname", CSV_HOSTNAME },
    { "interface", CSV_INTERFACE },
    { NULL, -1 },
};

/*  D A T A   S T R U C T U R E S  **************************************/
typedef struct _OpAlertCSV_Data 
{
    char *filepath;
    FILE *file;
    int num_entries;
    u_int32_t *entry_defs;
} OpAlertCSV_Data;


/* Output Plugin API */
static int OpAlertCSV_Setup(OutputPlugin *, char *args);
static int OpAlertCSV_Exit(OutputPlugin *);
static int OpAlertCSV_Start(OutputPlugin *, void *);
static int OpAlertCSV_Stop(OutputPlugin *);
static int OpAlertCSV(void *, void *);
static int OpAlertCSV_LogConfig(OutputPlugin *);

static OpAlertCSV_Data *OpAlertCSV_ParseArgs(char *);
static void OpAlertCSV_ParseCustomFormat(OpAlertCSV_Data *data, char *format);
static char *CSVEscape(char *);

/* init routine makes this processor available for dataprocessor directives */
void OpAlertCSV_Init()
{
    OutputPlugin *outputPlugin;
    
    outputPlugin = RegisterOutputPlugin("alert_csv", "alert");

    outputPlugin->setupFunc = OpAlertCSV_Setup;
    outputPlugin->exitFunc = OpAlertCSV_Exit;
    outputPlugin->startFunc = OpAlertCSV_Start;
    outputPlugin->stopFunc = OpAlertCSV_Stop;
    outputPlugin->outputFunc = OpAlertCSV;
    outputPlugin->logConfigFunc = OpAlertCSV_LogConfig;
}
    

static char csv_format_string[8192];

static char *CreateFormatString(OpAlertCSV_Data *data)
{
    int i = 0;
    char *offset = csv_format_string;
    int space_left = 8192;
    char *format_string;
    int used = 0;

    if(!data)
        return NULL;

    if(data->num_entries == 0)
    {
        return "sig_gen,sig_id,sig_rev,class,priority,event_id,tv_sec,"
            "tv_usec,src,dst,sport_itype,dport_icode,protocol";
    }

    memset(csv_format_string, 0, 8192);

    /* Create the format string */
    for(i = 0; i < data->num_entries; i++)
    {
        if(i > 0)
        {
            format_string = ", %s";
        }
        else
        {
            format_string = "%s";
        }
        if(data->entry_defs[i] <= 0 || data->entry_defs[i] >= MAX_KEY_VALUE)
        {
            used = snprintf(offset, space_left, "NULL");
        }
        else
        {
            used = snprintf(offset, space_left, format_string, 
                    format_keys[data->entry_defs[i]]);
        }
        if(used >= space_left)
            return csv_format_string;

        space_left -= used;
        offset += used;
    }
    return csv_format_string;
}

static int OpAlertCSV_LogConfig(OutputPlugin *outputPlugin)
{
    OpAlertCSV_Data *data = NULL;

    if(!outputPlugin || !outputPlugin->data)
        return -1;

    data = (OpAlertCSV_Data *)outputPlugin->data;

    LogMessage("OpAlertCSV configured\n");
    LogMessage("  Filepath: %s\n", data->filepath);
    LogMessage("  Format: %s\n", CreateFormatString(data));
    
    return 0;
}


/* Setup the output plugin, process any arguments, link the functions to
 * the output functional node
 */
static int OpAlertCSV_Setup(OutputPlugin *outputPlugin, char *args)
{
    /* setup the run time context for this output plugin */
    outputPlugin->data = OpAlertCSV_ParseArgs(args);

    return 0;
}

/* Inverse of the setup function, free memory allocated in Setup 
 * can't free the outputPlugin since it is also the list node itself
 */
static int OpAlertCSV_Exit(OutputPlugin *outputPlugin)
{
    OpAlertCSV_Data *data = (OpAlertCSV_Data *)outputPlugin->data;
    
    if(data != NULL)
    {
        if(data->filepath != NULL)
            free(data->filepath);
        if(data->entry_defs != NULL)
            free(data->entry_defs);
    }
    
    return 0;
}

/* 
 * this function gets called at start time, you should open any output files
 * or establish DB connections, etc, here
 */
static int OpAlertCSV_Start(OutputPlugin *outputPlugin, void *spool_header)
{
    OpAlertCSV_Data *data = (OpAlertCSV_Data *)outputPlugin->data;

    if(data == NULL)
        FatalError("ERROR: Unable to find context for AlertCSV startup!\n");

    if(pv.verbose >= 2)
        OpAlertCSV_LogConfig(outputPlugin);
    
    /* Open file */
    if((data->file = fopen(data->filepath, "a")) == NULL)
    {
        FatalError("ERROR: Unable to open '%s' (%s)\n", data->filepath, 
                strerror(errno));
    }
    
    return 0;
}

static int OpAlertCSV_Stop(OutputPlugin *outputPlugin)
{
    OpAlertCSV_Data *data = (OpAlertCSV_Data *)outputPlugin->data;

    if(data == NULL)
        FatalError("ERROR: Unable to find context for AlertCSV startup!\n");
    
    /* close file */
    fclose(data->file);

    return 0;
}

/* 
 * this is the primary output function for the plugin, this is what gets called
 * for every record read 
 */
static int OpAlertCSV(void *context, void *data)
{
    int i = 0;
    Sid *sid = NULL;
    ClassType *class_type = NULL;
    char timestamp[TIMEBUF_SIZE];
    UnifiedAlertRecord *record = (UnifiedAlertRecord *)data;
    OpAlertCSV_Data *op_data = (OpAlertCSV_Data *)context;
    FILE *file = op_data->file;
    char *escaped_string;

    if(op_data->num_entries == 0)
    {    
        /* default output mode */
        fprintf(op_data->file, "%u,%u,%u,%u,%u,%u,%lu,%lu,%u,%u,%u,%u,%u,%u\n", 
                record->event.sig_generator,
                record->event.sig_id, record->event.sig_rev, 
                record->event.classification, record->event.priority,
                record->event.event_id, record->ts.tv_sec, record->ts.tv_usec,
                record->sip, record->dip, record->sp, record->sp, record->dp,
                record->protocol);
    }
    for(i = 0; i < op_data->num_entries; ++i)
    {
        switch(op_data->entry_defs[i])
        {
            case CSV_SIG_GEN:
                fprintf(file, "%u", record->event.sig_generator);
                break;
            case CSV_SIG_ID:
                fprintf(file, "%u", record->event.sig_id);
                break;
            case CSV_SIG_REV:
                fprintf(file, "%u", record->event.sig_rev);
                break;
            case CSV_SID:
                fprintf(file, "%u:%u:%u", record->event.sig_generator, 
                        record->event.sig_id, record->event.sig_rev);
                break;
            case CSV_CLASS:
                fprintf(file, "%u", record->event.classification);
                break;
            case CSV_CLASSNAME:
                class_type = GetClassType(record->event.classification);
                fprintf(file, "\"%s\"", 
                        class_type != NULL ? class_type->name : "Unknown");
                break;
            case CSV_PRIORITY:
                fprintf(file, "%u", record->event.priority);
                break;
            case CSV_EVENT_ID:
                fprintf(file, "%u", record->event.event_id);
                break;
            case CSV_EVENT_REFERENCE:
                fprintf(file, "%u", record->event.event_reference);
                break;
            case CSV_REF_TV_SEC:
                fprintf(file, "%lu", record->event.ref_time.tv_sec);
                break;
            case CSV_REF_TV_USEC:
                fprintf(file, "%lu", record->event.ref_time.tv_usec);
                break;
            case CSV_TV_SEC:
                fprintf(file, "%lu", record->ts.tv_sec);
                break;
            case CSV_TV_USEC:
                fprintf(file, "%lu", record->ts.tv_usec);
                break;
            case CSV_TIMESTAMP:
                RenderTimestamp(record->ts.tv_sec, timestamp, TIMEBUF_SIZE);
                fprintf(file, "\"%s\"", timestamp);
                break;
            case CSV_SRC:
                fprintf(file, "%u", record->sip);
                break;
            case CSV_SRCIP:
                fprintf(file, "%u.%u.%u.%u", 
                        (record->sip & 0xff000000) >> 24,
                        (record->sip & 0x00ff0000) >> 16,
                        (record->sip & 0x0000ff00) >> 8,
                        record->sip & 0x000000ff);
                break;
            case CSV_DST:
                fprintf(file, "%u", record->dip);
                break;
            case CSV_DSTIP:
                fprintf(file, "%u.%u.%u.%u", 
                        (record->dip & 0xff000000) >> 24,
                        (record->dip & 0x00ff0000) >> 16,
                        (record->dip & 0x0000ff00) >> 8,
                        record->dip & 0x000000ff);
                break;
            case CSV_SPORT_ITYPE:
                fprintf(file, "%u", record->sp);
                break;
            case CSV_SPORT:
                if((record->protocol == 6) || (record->protocol == 17))
                    fprintf(file, "%u", record->sp);
                break;
            case CSV_ITYPE:
                if(record->protocol == 1)
                    fprintf(file, "%u", record->sp);
                break;
            case CSV_DPORT_ICODE:
                fprintf(file, "%u", record->dp);
                break;
            case CSV_DPORT:
                if((record->protocol == 6) || (record->protocol == 17))
                    fprintf(file, "%u", record->dp);
                break;
            case CSV_ICODE:
                if(record->protocol == 1)
                    fprintf(file, "%u", record->dp);
                break;
            case CSV_PROTO:
                fprintf(file, "%u", record->protocol);
                break;
            case CSV_PROTONAME:
                fprintf(file, "\"%s\"", protocol_names[record->protocol]);
                break;
            case CSV_FLAGS:
                fprintf(file, "%u", record->flags);
                break;
            case CSV_MSG:
                sid = GetSid(record->event.sig_generator, record->event.sig_id);
                if(sid != NULL && sid->CSVmsg == NULL)
                    sid->CSVmsg = CSVEscape(sid->msg);
                fprintf(file, "%s", sid != NULL ? sid->CSVmsg : "Snort Alert");
                break;
            case CSV_HOSTNAME:
                escaped_string = CSVEscape(pv.hostname);
                fprintf(file, "%s", pv.hostname != NULL ? escaped_string : "");
                free(escaped_string);
                break;
            case CSV_INTERFACE:
                escaped_string = CSVEscape(pv.interface);
                fprintf(file, "%s", pv.interface != NULL ? escaped_string : "");
                free(escaped_string);
                break;
        }
        if(i < op_data->num_entries - 1)
            fprintf(file, ",");
        else
            fprintf(file, "\n");
    }
    fflush(file);        
    return 0;
}

/* initialize the output processor for this particular instantiation */
OpAlertCSV_Data *OpAlertCSV_ParseArgs(char *args)
{
    OpAlertCSV_Data *data;

    data = (OpAlertCSV_Data *)SafeAlloc(sizeof(OpAlertCSV_Data));

    if(args != NULL)
    {
        char **toks;
        int num_toks;
        /* parse out your args */
        toks = mSplit(args, " ", 2, &num_toks, 0);
        switch(num_toks)
        {
            case 2:
                OpAlertCSV_ParseCustomFormat(data, toks[1]);
            case 1:
                data->filepath = strdup(toks[0]);
                break;
            case 0:
                data->filepath = strdup("csv.out");
                break;
            default:
                FatalError("ERROR %s (%d) => Invalid arguments for AlertCSV "
                        "plugin: %s\n", file_name, file_line, args);
        }       
        /* free your mSplit tokens */
        FreeToks(toks, num_toks);
    }
    else    
    {
        data->filepath = strdup("csv.out");
    }

    return data;
}


void OpAlertCSV_ParseCustomFormat(OpAlertCSV_Data *data, char *format)
{
    char **toks;
    int num_toks;
    int i;
    toks = mSplit(format, ",", 128, &num_toks, 0);
    data->num_entries = num_toks;
    data->entry_defs = (u_int32_t *)calloc(num_toks, sizeof(u_int32_t));
    for(i = 0; i < num_toks; ++i)
    {
        if(strcasecmp("sig_gen", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_SIG_GEN;
        }
        else if(strcasecmp("sig_id", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_SIG_ID;
        }
        else if(strcasecmp("sig_rev", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_SIG_REV;
        }
        else if(strcasecmp("sid", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_SID;
        }
        else if(strcasecmp("class", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_CLASS;
        }
        else if(strcasecmp("classname", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_CLASSNAME;
        }
        else if(strcasecmp("priority", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_PRIORITY;
        }
        else if(strcasecmp("event_id", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_EVENT_ID;
        }
        else if(strcasecmp("event_reference", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_EVENT_REFERENCE;
        }
        else if(strcasecmp("ref_tv_sec", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_REF_TV_SEC;
        }
        else if(strcasecmp("ref_tv_usec", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_REF_TV_USEC;
        }
        else if(strcasecmp("tv_sec", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_TV_SEC;
        }
        else if(strcasecmp("tv_usec", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_TV_USEC;
        }
        else if(strcasecmp("timestamp", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_TIMESTAMP;
        }
        else if(strcasecmp("src", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_SRC;
        }
        else if(strcasecmp("srcip", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_SRCIP;
        }
        else if(strcasecmp("dst", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_DST;
        }
        else if(strcasecmp("dstip", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_DSTIP;
        }
        else if(strcasecmp("sport_itype", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_SPORT_ITYPE;
        }
        else if(strcasecmp("sport", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_SPORT;
        }
        else if(strcasecmp("itype", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_ITYPE;
        }
        else if(strcasecmp("dport_icode", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_DPORT_ICODE;
        }
        else if(strcasecmp("dport", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_DPORT;
        }
        else if(strcasecmp("icode", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_ICODE;
        }
        else if(strcasecmp("proto", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_PROTO;
        }
        else if(strcasecmp("protoname", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_PROTONAME;
        }
        else if(strcasecmp("flags", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_FLAGS;
        }
        else if(strcasecmp("msg", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_MSG;
        }
        else if(strcasecmp("hostname", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_HOSTNAME;
        }
        else if(strcasecmp("interface", toks[i]) == 0)
        {
            data->entry_defs[i] = CSV_INTERFACE;
        }
        else
        {
            fprintf(stderr, "WARNING %s (%u) => Unrecognized keyword in "
                    "AlertCSV: %s\n", file_name, file_line, toks[i]);
        }
    }
    FreeToks(toks, num_toks);
}

char *CSVEscape(char *input)
{
    size_t strLen;
    char *buffer;
    char *current;
    if((strchr(input, ',') == NULL) && (strchr(input, '"') == NULL))
        return strdup(input);
    /* max size of escaped string is 2*size + 3, so we allocate that much */
    strLen = strlen(input);
    buffer = (char *)SafeAlloc((strLen * 2) + 3);
    current = buffer;
    *current = '"';
    ++current;
    while(*input != '\0')
    {
        switch(*input)
        {
            case '"':
                *current = '\\';
                ++current;
                *current = '"';
                ++current;
                break;
            case '\\':
                *current = '\\';
                ++current;
                *current = '\\';
                ++current;
                break;
            default:
                *current = *input;
                ++current;
                break;
        }
        ++input;
    }
    *current = '"';
    return buffer;
}
