/* $Id: classification.c,v 1.5 2003/05/03 03:20:54 andrewbaker Exp $ */
/*
** Copyright (C) 2002 Andrew R. Baker <andrewb@sourcefire.com>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#ifdef SOLARIS
    #include <strings.h>
#endif

#include "classification.h"
#include "mstring.h"
#include "util.h"
#include "barnyard.h"

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define BUF_SIZE  1024

ClassType *ClassList;

extern char *file_name;
extern int file_line;

void ReadClassFile(FILE *);
void ParseClassLine(char *);
static void ClassTypeDestroy(ClassType *);

void InitClassData(char *file)
{
    FILE *classfile;

    file_name = file;
    
    if((classfile = fopen(file, "r")) == NULL)
    {
        LogMessage("ERROR => Unable to open Classification file \"%s\": %s\n", 
                file, strerror(errno));
        
        return;
    }

    ReadClassFile(classfile);
}

void FreeClassData()
{
    ClassType *next;
    while(ClassList)
    {
        next = ClassList->next;
        ClassTypeDestroy(ClassList);
        ClassList = next;
    }
    ClassList = NULL;
}

static void ClassTypeDestroy(ClassType *classType)
{
    if(!classType)
        return;

    if(classType->type)
        free(classType->type);
    if(classType->name)
        free(classType->name);

    free(classType);
}

void ReadClassFile(FILE *fp)
{
    char buf[BUF_SIZE];
    char *index;
    char **toks;
    int num_toks;
    
    bzero(buf, BUF_SIZE);
    
    while(fgets(buf, BUF_SIZE, fp) != NULL)
    {
        file_line++;

        index = buf;

        /* advance through any whitespace at the beginning of the line */
        while(*index == ' ' || *index == '\t')
            index++;

        /* if it's not a comment or a <CR>, send it to the parser */
        if((*index != '#') && (*index != 0x0a) && (index != NULL))
        {
            toks = mSplit(index, ":", 2, &num_toks, 0);
            
            if(num_toks > 1)
            {
                ParseClassLine(toks[1]);
            }

            FreeToks(toks, num_toks);
        }
    }
}


void ParseClassLine(char *args)
{
    char **ctoks;
    int num_ctoks;
    char *data = NULL;
    int class_id = 0;
    ClassType *newNode;
    ClassType *current = ClassList;

    ctoks = mSplit(args, ",",3, &num_ctoks, '\\');

    if(num_ctoks < 1)
    {
        LogMessage("WARNING %s(%d): You must supply at least ONE"
                " classification argument\n", file_name, file_line);
        LogMessage("WARNING %s(%d): Ignoring configuration directive (%s: %d)"
                "\"config classification: %s\"\n", file_name, file_line, args);
        return;
    }

    data = ctoks[0];

    while(isspace((int)*data)) data++;

    while(current != NULL)
    {
        if(!strncasecmp(current->type, data, strlen(current->type)))
        {
            LogMessage("WARNING %s(%d): Duplicate classification \"%s\""
                    "found, ignoring this line\n", file_name, file_line, data);
            return;
        }

        class_id = current->id;
        current = current->next;
    }

    /* Create the new node */
    newNode = (ClassType *) SafeAlloc(sizeof(ClassType));

    newNode->type = strdup(ctoks[0]);
    newNode->id = class_id + 1;

    if(num_ctoks == 2)
    {
        data = ctoks[1];
        while (isspace((int)*data)) data++;
        newNode->priority = atoi(data);

        data = ctoks[0];
        while (isspace((int)*data)) data++;
        newNode->name = strdup(data);
    }
    else
    {
        data = ctoks[1];
        while (isspace((int)*data)) data++;
        newNode->name = strdup(data);


        data = ctoks[2];
        while (isspace((int)*data)) data++;
        newNode->priority = atoi(data);
    }


    /* Add the node to the list */
    if(ClassList == NULL)
    {
        ClassList = newNode;
    }
    else
    {
        current = ClassList;

        while(current->next != NULL)
            current = current->next;

        current->next = newNode;
    }

    FreeToks(ctoks, num_ctoks);


#ifdef DEBUG
    printf("classification list:\n");
    current = ClassList;
    while(current != NULL)
    {
        printf("Node %d   type: %s   name: %s   pri: %d\n", current->id, 
                current->type, current->name, current->priority);
        current = current->next;
    }
#endif

    return;
}



ClassType *GetClassType(int id)
{
    ClassType *idx;  /* indexing pointer... */

    idx = ClassList;

    if(ClassList == NULL)
        return NULL;

    while(idx != NULL)
    {
        if(idx->id == id)
        {
            return idx;
        }

        idx = idx->next;
    }
    
    return NULL;
}

