import { Widget } from '@lumino/widgets';
/**
 * Any object is "printable" if it implements the `IPrintable` interface.
 *
 * To do this it, it must have a method called `Printing.symbol` which returns either a function
 * to print the object or null if it cannot be printed.
 *
 * One way of printing is to use the `printWidget` function, which creates a hidden iframe
 * and copies the DOM nodes from your widget to that iframe and printing just that iframe.
 *
 * Another way to print is to use the `printURL` function, which takes a URL and prints that page.
 */
export declare namespace Printing {
    /**
     * Function that takes no arguments and when invoked prints out some object or null if printing is not defined.
     */
    type OptionalAsyncThunk = (() => Promise<void>) | null;
    /**
     * Symbol to use for a method that returns a function to print an object.
     */
    const symbol: unique symbol;
    /**
     * Objects who provide a custom way of printing themselves
     * should implement this interface.
     */
    interface IPrintable {
        /**
         * Returns a function to print this object or null if it cannot be printed.
         */
        [symbol]: () => OptionalAsyncThunk;
    }
    /**
     * Returns whether an object implements a print method.
     */
    function isPrintable(a: unknown): a is IPrintable;
    /**
     * Returns the print function for an object, or null if it does not provide a handler.
     */
    function getPrintFunction(val: unknown): OptionalAsyncThunk;
    /**
     * Prints a widget by copying it's DOM node
     * to a hidden iframe and printing that iframe.
     */
    function printWidget(widget: Widget): Promise<void>;
    /**
     * Prints a URL by loading it into an iframe.
     *
     * @param url URL to load into an iframe.
     */
    function printURL(url: string): Promise<void>;
}
