/*
 * QueueTemplate.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 2002-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): suhler.
 *
 * Version:  2.7
 * Created by suhler on 02/02/27
 * Last modified by suhler on 04/12/30 12:38:14
 */

package sunlabs.brazil.template;

import java.util.Vector;
import java.util.StringTokenizer;
import java.util.Enumeration;
import sunlabs.brazil.session.SessionManager;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.Glob;

/**
 * Template class for Managing simple Queues, allowing text communication
 * among sessions.
 * The "name" attribute names the Q.  When enqueueing messages, "name"
 * is a white space separated list of queue recipients. When Dequeueing
 * Messages, "name" is the recipient Queue.
 * <p>
 * The following tags are recognized.
 * <ul>
 * <li><code>&lt;enqueue name="recipients ..." meta="anything" message="msg"
 *                       from="sender" [nocreate="true|false"] &gt;</code>
 * <p>
 * <li><code>&lt;dequeue name="name" prepend="props prefix"
 *     timelimit="sec"&gt;</code>
 * <li><code>&lt;queueinfo name="q_name" prepend="props prefix" clear&gt;</code>
 * </ul>
 *
 * @author		Stephen Uhler
 * @version		 @(#)QueueTemplate.java	2.7 2.7
 */

public class QueueTemplate extends Template {
    public static final String Q_ID = "q";	// session table key

    /**
     * Add a text message onto a named queue.
     * Attributes:<br>
     * - name:	the name of the queue (required)<br>
     * - from:  the sender name<br>
     * - message: the text message<br>
     * - meta:	  application specific meta-data<br>
     * - nocreate: If set, The item will not be Queued if the Q for that
     *	 recipient does not already exist.<br>
     * - force:	  put to the queue even if its closed<br>
     * <p>
     * The property "count" contains the number of recipients for which
     * the message was successfully Queued.  The property "error.name"
     * will contain an error message if Queueing failed. In both cases, 
     * the template prefix will be prepended.  It is not considered an
     * error condition for a message not to be delivered to a non existent
     * Queue if "nocreate" is set.
     */

    public void
    tag_enqueue(RewriteContext hr) {
	String names = hr.get("name");
        String glob = hr.get("glob");	// specify names as a glob pattern in properties
	String from = hr.get("from", "anonymous");
	String meta = hr.get("meta");
	String message = hr.get("message", "Hello");
	boolean noCreate = hr.isTrue("nocreate");
	boolean force = hr.isTrue("force");
	hr.killToken();
	debug(hr);

	/* supply names as a glob pattern - experimental */

        if (names==null && glob != null) {
	   StringBuffer sb = new StringBuffer();
	   Enumeration e = hr.request.props.propertyNames();
	   String[] substr = new String[1];
	   boolean ok = false;
	   while(e.hasMoreElements()) {
	       if (Glob.match(glob, (String) e.nextElement(), substr)) {
		   sb.append(" ").append(substr[0]);
		   ok=true;
	       }
	   }
	   if (ok) {
	       names = sb.toString();
	       hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
		       "enqueue " + glob + "=(" + names + ")");
	   }
        }

        if (names==null) {
           debug(hr, "Missing Q name");
           return;
	}
	QueueItem item = new QueueItem(from, message, meta);
	StringTokenizer st = new StringTokenizer(names);
	Class exists = noCreate ? null : Queue.class;
	int count = 0;
	while (st.hasMoreTokens()) {
	    String name = st.nextToken();
	    Queue q = (Queue) SessionManager.getSession(name, Q_ID, exists);
	    if (q != null && !q.put(item, force)) {
	        hr.request.props.put(hr.prefix + "error." + name, " Q full");
	    } else {
		
		count++;
	    }
	}
	hr.request.props.put(hr.prefix + "count", "" + count);
    }

    /**
     * Allow another data to be enqueued from java code.
     * @param to:	The queue name
     * @param from:	The sender of the data
     * @param message:	The message to enqueue
     * @param meta:	The meta data, if any
     * @param noCreate:	If true, don't create a Q if it doesn'a already exist
     * @param force:	Force item onto Q even if it is closed
     * @return		True, if the data was enqueued
     */

    public static boolean
    enqueue(String to, String from, String message, 
		String meta, boolean noCreate, boolean force) {
	QueueItem item = new QueueItem(from, message, meta);
	Queue q = (Queue) SessionManager.getSession(to, Q_ID,
		noCreate ? null : Queue.class);
	return (q != null && q.put(item, force));
    }

    /**
     * Remove an item from the queue, and generate the appropriate properties.
     * Attributes:<br>
     * <dl>
     * <dt>name<dd>The name of the queue to examine
     * <dt>prepend<dd>The prefix in the properties table to use to
     * set the results.  Defaults to our prefix.
     * <dt>timelimit<dd>how long to wait for an item to appear, in sec.
     * Defaults to 30.
     * </dl>
     * If an item is retrieved, the following request properties
     * are set (preceded by prepend):
     * <dl class=props>
     * <dt>message<dd>Set The text of the next message in the queue.
     * <dt>age<dd>Set how long the message has been q's (in seconds)
     * <dt>sent<dd>Set the timestamp (in sec) of when the item was q'd.
     * <dt>items<dd>Set the number of Q'd items.
     * <dt>from<dd>Set the (unauthenticated) sender.
     * <dt>error<dd>Something went wrong.  Set an error message.
     * </dl>
     * Note: this tag blocks until an item is received in the Queue
     * (or a timelimit expires).  As template processing is synchronized
     * based on sessions, care should be taken to avoid blocking
     * other (unrelated) session based requests while waiting on
     * the queue.
     */

    public void
    tag_dequeue(RewriteContext hr) {
	String name = hr.get("name");
	String prepend = hr.get("prepend", hr.prefix);
	String timelimit = hr.get("timelimit", "30");
        int timer = 30;
	try {
	    timer = Integer.decode(timelimit).intValue();
	} catch (Exception e) {}

	hr.killToken();
	debug(hr);
        if (name == null) {
           debug(hr, "Missing Q name");
           return;
	}
	Queue q = (Queue) SessionManager.getSession(name, Q_ID, Queue.class);
	hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
		name + ": Q.get(" + timer + ")...");
	QueueItem data = (QueueItem) q.get(timer);
	if (data != null) { 
	    hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
		    name + ": Q.get(" + timer + ") " +
		    data.message + " (" + data.meta + ")");
	    hr.request.props.put(prepend + "message", data.message);
	    hr.request.props.put(prepend + "from", data.from);
	    if (data.meta != null) {
	        hr.request.props.put(prepend + "meta", data.meta);
	    }
	    hr.request.props.put(prepend + "age", "" +
		     (System.currentTimeMillis() - data.timestamp)/1000);
	    hr.request.props.put(prepend + "sent", "" + (data.timestamp/1000));
	} else {
	    hr.request.log(Server.LOG_DIAGNOSTIC, hr.prefix,
		    name + ": Q.get() Timeout");
	    hr.request.props.put(prepend + "error", "timelimit");
	}
        return;
    }

    /**
     * Program access to the Q.
     * @param name;	The name of the Q.  A new Q will be created if
     *			it doesn't already exist.
     * @param timelimit: how long (in ms) to wait before returning
     * @return		The Queue item, or Null if no item is available.
     */

    public static QueueItem
    dequeue(String name, int timelimit) {
	Queue q = (Queue) SessionManager.getSession(name, Q_ID, Queue.class);
	return (QueueItem) q.get(timelimit);
    }

    /**
     * Return info about the Q, and optionally clear it.
     * Attributes:<br>
     * <dl>
     * <dt>name<dd>The name of the queue to examine
     * <dt>prepend<dd>The prefix in the properties table to use to
     * set the results.  Defaults to our prefix.
     * <dt>clear<dd>If set, then clear the queue
     * <dt>remove<dd>If set, then remove the queue
     * <dt>closed=[true|false]<dd>set the closed state of the Q
     * </dl>
     * The following request properties
     * are set (preceded by prepend):
     * <dl>
     * <dt>lastIn<dd>The timestamp (ms) of the last Q insert attempt.
     * <dt>lastOut<dd>The timestamp (ms) of the last Q retrieval attempt.
     * <dt>size<dd>The number of items in the Q.
     * <dt>count<dd>The number of items inserted into the Q
     * </dl>
     */

    public void
    tag_queueinfo(RewriteContext hr) {
	String name = hr.get("name");
	String prepend = hr.get("prepend", hr.prefix);
	boolean clear = hr.isTrue("clear");
	boolean remove = hr.isTrue("remove");
	boolean notify = hr.isTrue("notify");
	boolean closed = (hr.get("closed") != null);

	hr.killToken();
	debug(hr);
        if (name == null) {
           debug(hr, "Missing Q name");
           return;
	}
	Queue q = (Queue) SessionManager.getSession(name, Q_ID, Queue.class);
	hr.request.props.put(prepend + "lastIn", "" + q.lastIn());
	hr.request.props.put(prepend + "lastOut", "" + q.lastOut());
	hr.request.props.put(prepend + "size", "" + q.size());
	hr.request.props.put(prepend + "count", "" + q.count());
	hr.request.props.put(prepend + "closed", "" + q.isClosed());
	if (closed) {
	    q.setClosed(hr.isTrue("closed"));
	}
	if (clear) {
	    q.clear();
	}
	if (notify) {
	    q.kick();
	}

	if (remove) {
	    SessionManager.remove(name, Q_ID);
	}
    }

    /**
     * Return a Q
     */

    public static Queue
    getQ(String name) {
	return (Queue) SessionManager.getSession(name, Q_ID, Queue.class);
    }

    /**
     * A bag of items to keep on the Q.
     * We could add other stuff later.
     */

    public static class QueueItem {
        public long timestamp;	// when the item was queued 
	public String from;	// Who sent it
	public String message;	// What is the message
	public String meta;	// Arbitrary meta info about the message

	/**
	 * Add an item to the Q.
	 * "meta" is application specific "meta" data associated with 
	 * this Q item.
	 */

        public QueueItem(String from, String message, String meta) {
	    timestamp = System.currentTimeMillis();
	    this.from=from;
	    this.message = message;
	    this.meta = meta;
	}

        public String toString() {
            return from + ": " + message + " (" + timestamp + ")";
        }
    }

    /**
     * Create an object queue.  "Getters" wait 'till something
     * appears in the queue.
     */

    public static class Queue {
	public static int max=100;  
	Vector q;	// where to hold the items
	Object mutex = new Object();
	long lastGet;	// timestamp of last get
	long lastPut;	// timestamp of last put
	int maxItems;	// max q size
	int count;	// number of items Q'd
	boolean closed; // The queue is closed
	// long start=0;   // temporary

	/**
	 * Create a new Q of a maximum possible size
	 */

	public Queue() {
	   lastGet = lastPut = -1;
	   count = 0;
	   maxItems = max;
	   q = new Vector();
	   closed = false;
	   // start = System.currentTimeMillis();
        }

	/**
	 * Return the next item on the queue, waiting for up to
	 * "timeout" seconds or for an interrupt.
	 * @return the top of the Q, or null.
	 */

	public Object
	get(int timeout) {
	    try {
		String name = Thread.currentThread().getName();
	        synchronized (mutex) {
		    if (q.size() == 0) {
		        mutex.wait(timeout==0 ? 1 : timeout*1000);
		    }
		    lastGet = System.currentTimeMillis();
		    if (q.size() > 0) {
		        Object data = q.firstElement();
		        q.removeElementAt(0);
			return data;
                    }
	        }
	    } catch (InterruptedException e) {}
	    return null;
	}

	/**
	 * Put an item on the queue if it's open and not full.
	 */

	public boolean
	put(Object item) {
	    return put(item, false);
	}

	/**
	 * Put an item on the queue if it's not full.
	 * If "force" is true, override the "closed" flag.
	 */

	public boolean
	put(Object item, boolean force) {
	    if (closed && !force) {
		return false;
	    }
	    boolean result;
	    String name = Thread.currentThread().getName();
            synchronized (mutex) {
		lastPut = System.currentTimeMillis();
                if (q.size() < maxItems) {
		    count++;
	            q.addElement(item);
	            mutex.notifyAll();
                    result = true;
	        } else {
		    result =  false;
		}
	    }
	    return result;
	}

	/**
	 * How many items are queue'd.
	 */

	public int
	size() {
	    return q.size();
	}

	/**
	 * Send a notify: for debugging
	 */

	public void kick() {
            synchronized (mutex) {
	        // System.out.println("*NOTIFY* " + mutex);
	        mutex.notifyAll();
	    }
	}

	/**
	 * Return the last time a Q insertion was attempted.
	 * -1 if no attempts were made.
	 */

	public long
	lastIn() {
	    return lastPut;
	}

	/**
	 * Return the last time a Q removal was attempted.
	 * -1 if no attempts were made.
	 */

	public long
	lastOut() {
	    return lastGet;
	}

	/**
	 * Return the total number of items Q'd.
	 */

	public long
	count() {
	    return count;
	}

	/**
	 * Clear the queue.
	 */

	public void
	clear() {
	   q.removeAllElements();
	   lastGet = lastPut = -1;
	   count = 0;
	}

	/**
	 * set the closed state
	 */

	public boolean setClosed(boolean closed) {
	    boolean result = this.closed;
	    this.closed = closed;
	    return result;
	}

	/**
	 * get the closed state
	 */

	public boolean isClosed() {
	   return this.closed;
	}
    }
}
