/*
 * PlainFilter.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2004 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): suhler.
 *
 * Version:  2.2
 * Created by suhler on 99/07/29
 * Last modified by suhler on 04/11/30 15:19:39
 */

package sunlabs.brazil.filter;

import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.util.http.MimeHeaders;
import sunlabs.brazil.util.http.HttpUtil;

/**
 * Filter to turn text/plain into html.  This allows plain text to 
 * be processed by other filters that only deal with html.
 * <p>
 * The following server properties are used:
 * <dl class=props>
 * <dt>template	<dd> The string to use as an html template.  The string
 *		     should contain a single "%", which is replaced by the 
 *		     text/plain content.  The default stuff the content
 *		     between &lt;pre&gt;...&lt;/pre&gt;.
 * </dl>
 *
 * @author		Stephen Uhler
 * @version		2.2
 */

public class PlainFilter implements Filter {
    String template1;		// html wrapper template (prefix)
    String template2;		// html wrapper template (postfix)

    public boolean
    init(Server server, String prefix) {
	String template = server.props.getProperty(prefix + "template", 
		"<title>text document</title><body bgcolor=white><pre>" +
		"%</pre></body>");
	int index = template.indexOf("%");
	if (index > 0) {
	    template1 = template.substring(0,index);
	    template2 = template.substring(index+1);
	    return true;
	} else {
	    return false;
	}
    }

    /**
     * This is the request object before the content was fetched
     */

    public boolean respond(Request request) {
	return false;
    }

    /**
     * Only filter text/plain documents
     */

    public boolean
    shouldFilter(Request request, MimeHeaders headers) {
	String type = headers.get("content-type");
	return (type != null && type.toLowerCase().startsWith("text/plain"));
    }

    /**
     * Wrap html around text/plain, converting it to html.
     * Change the content-type to text/html.
     */

    public byte[]
    filter(Request request, MimeHeaders headers, byte[] content) {
	String result =  template1 + HttpUtil.htmlEncode(new String(content)) +
		template2;
	headers.put("content-type", "text/html");
	return result.getBytes();
    }
}
