/*
 *      CUEgen Version 1.2.0
 *
 *      A FLAC-compatible audio CD cuesheet generator.
 *
 *      (C) Paul Slavin 2005
 *      slavinp@cs.man.ac.uk
 *
 *      Licensed under the GNU GPL
 */


#include <stdio.h>
#include <stdlib.h>
#include <linux/cdrom.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>

#include "cuegen.h"


char cd[32] = "/dev/cdrom";
char outfile[128], dummyfile[64] = "cdda.wav", perf[128] = "", title[128] = "";
const char version[] = "1.2.0";
struct cdrom_tochdr toch;
struct cdrom_tocentry *tocent;
struct track *tracks;
int cdfp, first, last;
FILE *fp = NULL;


int main(int argc, char *argv[]){

	int option;
	const char* const short_options = "hio:f:d:t:p:";

	fp = stdout;

	do{
		option = getopt(argc, argv, short_options);

		switch(option){

		case 'h':	print_usage(stdout, 0);
				break;

		case 'i':	printinfo();
				break;

		case 'o':	if(optarg){
					fp = fopen(optarg, "w");
					if(!fp){
						fprintf(stderr, "Unable to create %s\n", optarg);
						exit(1);
					}
				}

				break;

		case 'f':	if(optarg){
					strncpy(dummyfile, optarg, 63);
					dummyfile[63] = '\0';
				}

				break;

		case 'd':	strncpy(cd, optarg, 31);
				cd[31] = '\0';

				break;

		case 't':	strncpy(title, optarg, 127);
				title[127] = '\0';

				break;

		case 'p':	strncpy(perf, optarg, 127);
				perf[127] = '\0';

				break;

		case '?':
		case ':':	print_usage(stderr, 1);
				break;

		default:	break;
		}

	} while(option != -1);
					
	interrogate();	
	maketracks(tocent);
	printcue(fp, last);

	
	tidy();	
	return(0);
}


void tidy(void){

	close(cdfp);
	fclose(fp);
	free(tocent);
	free(tracks);

	return;

}


void maketracks(struct cdrom_tocentry *tr){

	int i, offset_frame, offset_second;
	int minute, second, frame;
	
	offset_second = tr[0].cdte_addr.msf.second;
	offset_frame = tr[0].cdte_addr.msf.frame;


	tracks = (struct track *)malloc(last * sizeof(struct track));
	if(!tracks){
		fprintf(stderr, "Unable to allocate tracks[].\n");
		exit(1);
	}

	for(i=first; i<=last; i++){

		minute = tr[i].cdte_addr.msf.minute;
		second = tr[i].cdte_addr.msf.second;
		frame  = tr[i].cdte_addr.msf.frame;

		if(frame < offset_frame){
			second--;
			frame = 75 + frame - offset_frame;
		}else{
			frame-= offset_frame;
		}

		if(second < offset_second){
			minute--;
			second = 60 + second - offset_second;
		}else{
			second -= offset_second;
		}

		tracks[i].start.minute = minute;
		tracks[i].start.second = second;
		tracks[i].start.frame  = frame;
	}

}


void printcue(FILE *stream, int track_count){

	int i;

	if( strcmp(title, "") )
		fprintf(stream, "TITLE \"%s\"\n",title);

	if( strcmp(perf, "") )
		fprintf(stream, "PERFORMER \"%s\"\n", perf);

	fprintf(stream, "FILE \"%s\" WAVE\n", dummyfile);
	
	for(i=0; i<track_count; i++){
		fprintf(stream, "  TRACK %.2d AUDIO\n", i+1);
		fprintf(stream, "    INDEX 01 %.2d:%.2d:%.2d\n", tracks[i].start.minute, tracks[i].start.second, tracks[i].start.frame);
	}
}


void interrogate(void){

	int i;

        if ( (cdfp = open(cd, O_RDONLY)) == -1){
                fprintf(stderr, "Open of %s failed.\n", cd);
                exit(1);
        }

        if( ioctl(cdfp, CDROMREADTOCHDR, &toch) == -1){
                fprintf(stderr, "Unable to read CD header.\n");
                exit(1);
        }

        if( toch.cdth_trk0 < 1){
                fprintf(stderr, "No audio tracks on CD.\n");
                exit(1);
        }

        first = toch.cdth_trk0;
        last = toch.cdth_trk1;


        tocent = (struct cdrom_tocentry *)malloc((toch.cdth_trk1 + 1)* sizeof(struct cdrom_tocentry));
	if(!tocent){
		fprintf(stderr, "Unable to allocate tocent[].\n");
		exit(1);
	}

        for(i=first; i<=last; i++){

                tocent[i-1].cdte_track = i;
                tocent[i-1].cdte_format = CDROM_MSF;

                if( ioctl(cdfp, CDROMREADTOCENTRY, &tocent[i-1]) == -1){
                        fprintf(stderr, "Unable to read track info for track %d.\n", i);
                        exit(1);
                }
        }

	tocent[last].cdte_track = CDROM_LEADOUT;
        tocent[last].cdte_format = CDROM_MSF;

        if( ioctl(cdfp, CDROMREADTOCENTRY, &tocent[last]) == -1){
                fprintf(stderr, "unable to read CD leadout.\n");
                exit(1);
        }

	return;
}


void print_usage(FILE *stream, int excode){

	fprintf(stream,"CUEgen Version %s\n", version);
	fprintf(stream, "Paul Slavin 2005\n\n");
        fprintf(stream, "Usage:   cuegen [options] \n");
        fprintf(stream,
                "    -h                          Print this usage information.\n"
                "    -i                          Display detailed CD information in human-readable format.\n"
                "    -o  <output file>           Write cuesheet to specified file. Default is stdout.\n"
                "    -f  <dummy filename>        Dummy filename for inclusion in cuesheet. Default is cdda.wav.\n"
		"    -d  <device name>           Specify a CD device to interrogate. Default is /dev/cdrom.\n"
		"    -t  <title>                 Specify a title to be stored in the cuesheet's TITLE field.\n"
		"    -p  <performer>             Specify a performer to be stored in the cuesheet's PERFORMER field.\n");

        exit(excode);
}


void printinfo(void){

	int i, dmins=0, dsecs=0, dframes=0;

	interrogate();
	maketracks(tocent);

	
	printf("\nTrack\tStart\t\tEnd\t\tDuration\n");
	puts("=========================================================");

	for(i=0; i<last; i++){
		dmins = dsecs = dframes = 0;

		dmins = tracks[i+1].start.minute - tracks[i].start.minute;
		dsecs = tracks[i+1].start.second - tracks[i].start.second;
		dframes = tracks[i+1].start.frame - tracks[i].start.frame;

		if(dframes < 0){
			dsecs--;
			dframes = 75 + dframes;
		}

		if(dsecs < 0){
			dmins--;
			dsecs = 60 + dsecs;
		}

		printf("%.2d\t%.2d:%.2d:%.2d", i+1, tracks[i].start.minute, tracks[i].start.second, tracks[i].start.frame);

		printf("\t%.2d:%.2d:%.2d", tracks[i+1].start.minute, tracks[i+1].start.second, tracks[i+1].start.frame);
		printf("\t%.2d:%.2d:%.2d\n", dmins, dsecs, dframes);

	}
	
	puts("=========================================================");
	printf("\nTracks: %d", last); 

	printf("\t\tDuration: %.2dm %.2ds\n\n", tracks[last].start.minute, tracks[last].start.second);

	tidy();
	exit(0);
}
