;;;
;;; cdb.scm - cdb interface
;;;
;;;   Copyright (c) 2003-2004 Kimura Fuyuki, All rights reserved.
;;;
;;;   Redistribution and use in source and binary forms, with or without
;;;   modification, are permitted provided that the following conditions
;;;   are met:
;;;
;;;   1. Redistributions of source code must retain the above copyright
;;;      notice, this list of conditions and the following disclaimer.
;;;
;;;   2. Redistributions in binary form must reproduce the above copyright
;;;      notice, this list of conditions and the following disclaimer in the
;;;      documentation and/or other materials provided with the distribution.
;;;
;;;   3. Neither the name of the authors nor the names of its contributors
;;;      may be used to endorse or promote products derived from this
;;;      software without specific prior written permission.
;;;
;;;   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;;;   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;;;   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;;;   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
;;;   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;;;   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
;;;   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;;   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;;   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;;   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;;   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
;;;
;;;  $Id: cdb.scm,v 1.4 2004/02/25 02:39:57 fuyuki Exp $
;;;

(define-module dbm.cdb
  (extend dbm)
  (use srfi-2)
  (export <cdb> cdb-file-of
          cdb-open cdb-close cdb-closed?
          cdb-find cdb-datapos cdb-datalen cdb-read cdb-findinit cdb-findnext
          cdb-make-open cdb-make-add cdb-make-exists cdb-make-put
          |CDB_PUT_ADD| |CDB_PUT_REPLACE| |CDB_PUT_INSERT| |CDB_PUT_WARN|))
(select-module dbm.cdb)
(dynamic-load "cdb")

;;
;; Initialize
;;

(define-class <cdb-meta> (<dbm-meta>)
  ())

(define-class <cdb> (<dbm>)
  ((cdb-file :accessor cdb-file-of :init-value #f))
  :metaclass <cdb-meta>)

(define-method dbm-open ((self <cdb>))
  (next-method)
  (unless (slot-bound? self 'path)
    (error "path must be set to open cdb database"))
  (when (cdb-file-of self)
    (errorf "cdb ~S already opened" cdb))
  (let ((path (slot-ref self 'path))
        (rwmode (slot-ref self 'rw-mode)))
    (case rwmode
      ((:read)
       (slot-set! self 'cdb-file (cdb-open path)))
      ((:write)
       (errorf "cdb doesn't support rw-mode ~s" rwmode))
      ((:create)
       (slot-set! self 'cdb-file
                  (cdb-make-open path (slot-ref self 'file-mode)))))
    self))

;;
;; close operation
;;

(define-method dbm-close ((self <cdb>))
  (let ((cdb (cdb-file-of self)))
    (and cdb (cdb-close cdb))))

(define-method dbm-closed? ((self <cdb>))
  (let ((cdb (cdb-file-of self)))
    (or (not cdb) (cdb-closed? cdb))))

;;
;; accessors
;;

(define-method dbm-put! ((self <cdb>) key value)
  (next-method)
  (let ((rwmode (slot-ref self 'rw-mode)))
    (unless (eq? rwmode :create)
      (errorf "dbm-put! doesn't work in ~s mode" rwmode)))

  (when (negative? (cdb-make-add (cdb-file-of self)
                                 (%dbm-k2s self key)
                                 (%dbm-v2s self value)))
    (error "dbm-put! failed" self)))

(define-method dbm-get ((self <cdb>) key . args)
  (next-method)
  (let ((rwmode (slot-ref self 'rw-mode)))
    (unless (eq? rwmode :read)
      (errorf "dbm-get doesn't work in ~s mode" rwmode)))

  (let* ((cdb (cdb-file-of self))
         (r (cdb-find cdb (%dbm-k2s self key))))
    (cond ((positive? r)
           (%dbm-s2v self (cdb-read cdb (cdb-datalen cdb) (cdb-datapos cdb))))
          ((pair? args) (car args))     ;fall-back value
          (else (errorf "cdb: no data for key ~s in database ~s" key cdb)))))

(define-method dbm-exists? ((self <cdb>) key)
  (next-method)
  (let ((cdb (cdb-file-of self)))
    (case (slot-ref self 'rw-mode)
      ((:read) (positive? (cdb-find cdb (%dbm-k2s self key))))
      ((:create) (positive? (cdb-make-exists cdb (%dbm-k2s self key)))))))

;;
;; Iterations
;;

(define-method dbm-fold ((self <cdb>) proc knil)
  (let ((rwmode (slot-ref self 'rw-mode)))
    (unless (eq? rwmode :read)
      (errorf "dbm-fold doesn't work in ~s mode" rwmode)))

  (call-with-input-file (slot-ref self 'path)
    (lambda (inp)
      (with-port-locking inp
        (lambda ()
          (define read-len cdb-unpack)

          (and-let* ((buf (read-block 2048 inp))
                     ((not (eof-object? buf)))
                     (eod (read-len buf)))

            (define (read-block-safe len)
              (cond ((> (+ (port-tell inp) len) eod) #f)
                    ((read-block len inp)
                     => (lambda (buf) (and (not (eof-object? buf)) buf)))))
            (define (read-len-safe)
              (cond ((read-block-safe 4) => read-len)
                    (else #f)))

            (let loop ((r knil))
              (let ((klen (read-len-safe))
                    (vlen (read-len-safe)))
                (if (and klen vlen)
                    (let ((key (read-block-safe klen))
                          (val (read-block-safe vlen)))
                      (if (and key val)
                          (let ((key (string-incomplete->complete key))
                                (val (string-incomplete->complete val)))
                            (loop (proc (%dbm-s2k self key)
                                        (%dbm-s2v self val)
                                        r)))
                          r))
                    r)))))))
    :buffering :full))

;;
;; Metaoperations
;;

(autoload file.util copy-file move-file)

(define (%with-cdb-locking path thunk)
  (let1 db (cdb-open path) ;; put read-lock
    (with-error-handler
        (lambda (e) (cdb-close db) (raise e))
      (lambda () (thunk) (cdb-close db)))))

(define-method dbm-db-exists? ((class <cdb-meta>) name)
  (file-exists? name))

(define-method dbm-db-remove ((class <cdb-meta>) name)
  (sys-unlink name))

(define-method dbm-db-copy ((class <cdb-meta>) from to . keys)
  (%with-cdb-locking
   (lambda () (apply copy-file from to :safe #t keys))))

(define-method dbm-db-rename ((class <cdb-meta>) from to . keys)
  (%with-cdb-locking
   (lambda () (apply move-file from to :safe #t keys))))

(provide "dbm/cdb")
