/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 20 2003)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef OPTURTREE_H
#define OPTURTREE_H

#include "exndarr.h"
#include "inndarr.h"
#include "brarr.h"
#include "libr.h"
#include "lind.h"
#include "librarr.h"
#include "vec.h"
#include "dmat20.h"
#include "brent.h"

const bool PARALLEL_LOGLI = true;
const int DO_PARALLEL_LOGLI_FUNC = 1;
const int DO_PARALLEL_OTHER_FUNC = 2;

/**

	Class OptUtTree: this is one of the most important class, defining NNI operations and 
	optimizing branch lengths of an unrooted tree 
*/


class OptUrTree : public Brent {
public:
	friend class LiNd;
	friend class LiBr;

	/**
		the constructor function
	*/
	OptUrTree();

	/**
	use this function for constructing this class
	*/
	void doConstructor ();

	/**
		swap the position of chiNdNo and dNdNo
		@param parNdNo parent of sNdNo and dNdNo
		@param sNdNo first child of parNdNo 
		@param dNdNo second child of parNdNo 
		@param chiNdNo a child of sNdNo 
	*/
	void changeNd (int parNdno, int sNdno, int chiNdNo, int dNdNo);

	/**
		calculate the score of each branch,
		which is the increased log-likelihood
		if NNI is applied on this branch
	*/
	void createScoreBr ();
#ifdef PARALLEL
	/**
		parallel calculate the score of each branch,
		which is the increased log-likelihood
		if NNI is applied on this branch
	*/
	void par_createScoreBr ();
#endif // PARALLEL

	/**
		reoptimize the length of branch brNo and
		set the length to the new value
		@param brNo the branch Nr.
	*/
	double optLocalBr (int brNo);

	//int optTopology (int parNdNo, int sNdNo, int dNdNo);

	/**
		sort the branch scores (for NNI) by the
		descending order of increased likelihood
		on each branch
	*/
	void sortScoreBr ();
	/**
		sort the branch scores (for NNI) using Quicksort
		by the
		descending order of increased likelihood
		on each branch
		@param left the left most branch
		@param right the right most branch
	*/
	void sortScoreBr (int left, int right);

	/**
		Find the set of NNI operations by scanning
		down the sorted list and looking for
		swap which can be applied simultaneously
		(meaning two NNI having non-adjacent branches) 
		@param maxNSwapBr the maximum number of wanted NNIs
		@return the number of found NNIs
	*/
	int findNPosSwapBr (int maxNSwapBr);

	/**
		apply the NNIs found simultaneously
		@param lamda a parameter to modify the remaining branchs
		which was not applied with NNI
	*/	
	void trySwapBr (double lamda);

	/**
		main function to do NNI! loop through 
		several phases to find simultaneous NNIs, 
		and do it until no likelihood improvement found.
		@return the number of NNI operations applied	
	*/
	int swapBr ();

        /**
		swap back the NNI if the new likelihood
		is worse, then divide lamda by 2
	*/
	void swapBrBack ();

	// Unused
	double cmpSwapLogLi ();

	/**
		after doing NNI, reoptimize all branches once
		and recompute the log-likelihood
		@return the new log-likelihood
	*/
	double optSwapLogLi ();

	/**
		optimize the tree topology with NNI
		@param topo_changed (OUT) true if topology is changed
		@return the new log-likelihood
	*/
	double optTopology (bool &topo_changed);

	/**
		draw the tree
	*/
	void draw ();

	/**
		original branch lengths
	*/
	double *oriLenArr_;//[MAX_NUM_SEQ * 2];

	/**
		the posible branch lengths after NNI
	*/
	double *canLenArr_; //[MAX_NUM_SEQ * 2];


	/**
		the posible branch lengths after simultaneous NNI
	*/
	double *optedLenArr_; //[MAX_NUM_SEQ * 2];

	/**
		set out group node no for this tree
		@param outGrpNdNo the index of the leaf node
	*/
	void setOutGrpNd (int outGrpNdNo);

	/**
		set the exNdArr for this class, construct the structure of this unrooted tree
		@param exNdArr array of external nodes
	*/
	void setExNd (ExNdArr &exNdArr);

	/**
		set the inNdArr for this class, construct the structure of this unrooted tree
		@param inNdArr array of internal nodes
	*/
	void setInNd (InNdArr &inNdArr);

	/**
		set the brArr for this class, construct the structure of this unrooted tree
		@Param brArr array of branches
	*/
	void setBr (BrArr<double> &brArr);

	/**
		get the brArr of this tree after optimizing
		@param brArr (OUT) array of branches
	*/
	void getBr (BrArr<double> &brArr);

	/**
		get the exNdArr of this tree after optimizing
		@param exNdArr array of external nodes
	*/
	void getExNd (ExNdArr &exNdArr);

	/**
		get the InNdArr of this tree after optimizing
		@param inNdArr array of internal nodes
	*/
	void getInNd (InNdArr &inNdArr);


	/**
		get the branch length
		@param brNo index of branch
		@return the length
	*/
	double getBrLen (int brNo);

	/**
		 optimize all branch lengths of the tree
		@param nIt maximum number of iterations
		@return log-likehood afterwards
	*/
	double optBranches(int nIt = MAX_IT_UR_BR);


	/**
		opt all childrent bres of this node
		@param ndNo node index
		@param bChangedBr (OUT) number of branches changed
	*/
	void optBranches(int ndNo, int &nChangedBr);


	/**
		clean all content of this class
	*/
	void clean();

	/**
		everything is initialized here
	*/
	void init ();

	/**
		@return the log-likelihood of this tree.
	*/
	double getLogLi ();

	/**
		write all information of this optimal unrooted tree into the screen
	*/
	void write ();

	/**
		update the log-likelihood
		@param logLi new log-likelihood
	*/
	void updateBestLogLi (double logLi);

	/**
		Allocate memory for some variables
	*/
	void allocMemory();

	/**
		release all memmory for this class
    */
	void release ();

	/**
		the destructor of this class
	*/
	virtual ~OptUrTree();

	/***********************************************************************************
	***********************************************************************************
	***********************************************************************************
	 
	***********************************************************************************
	***********************************************************************************/

	//private :


	/**
		vector storing score of branches for NNI
	*/
	Vec<int> inBrNoLs_;

	/**
		computed likelihood of tree
	*/
	double bestLogLi_;

	/**
		the current number of inNds in this tree
	*/
	int curNInNd_;

	/**
		the current number of ExNds in this tree
	*/
	int curNExNd_;

	/**
		the current number of bres in this tree
	*/
	int curNBr_;

	/**
		the maximum number of inNds in this tree
	*/
	int maxNInNd_;

	/**
		the maximum number of ExNds in this tree
	*/
	int maxNExNd_;

	/**
		the maximum number of bres in this tree
	*/
	int maxNBr_;

	/**
		the outGrpNdNo if this unrooted tree
	*/
	int outGrpNdNo_;


	/**
		the outGrpBrNo if this unrooted tree
	*/
	int outGrpBrNo_;


	/**
		the artificial internal root node if this unrooted tree
	*/
	int artInRootNo_;

	/**
		the array of all internal nds of this UnrootTree
	*/
	InNdArr inNdArr_;

	/**
		the array of all ex nds of this UnrootTree
	*/
	ExNdArr exNdArr_;

	/**
		the array of all bres of this UnrootTree
	*/
	LiBrArr liBrArr_;

	/**
		isLiCmped = 0, it means that the likelihood of this tree has NO been computed
	*/
	int isLiCmped_;


	/**
		isBrLenEsted = 1, it means that we have estimated the len of all branches
	*/
	int isBrLenEsted_;

	/**
		contains the branch no which is changed
	*/
	int actBrNo_;

	/**
		it contains the genetic distance between any two sequences
	*/
	Mat<double> *genDisMat_;

	/**
		the number of calling function cmpNegLogLiBr
	*/
	int nCmpNegLogLiBr_;

	/**
		the number of compute likehood of an internal node
	*/
	int nCmpLi_;

	int nCmpExLogLi_;
	int nCmpInLogLi_;

	/**
		the number of calling cmpLogLi
	*/
	int nCmpLogLi_;

	/**
		@param ndNo node index
		@return true if this is an internal node
	*/
	int isInNd (int ndNo);

	/**
		check whether or not it is a external node
		@param ndNo node index
		@return true if this is an external node
	*/
	int isExNd (int ndNo);

	/**
		create external descendant nodes for all 
		internal nodes of this tree
   */
	void createExDesLiNd ();

	/**
		create pattern for all internal node of this tree
	*/
	void createPtnLiNd ();

	/**
		create the list of all external nodes 
		which belong to the same one root 
	*/
	void createGrpExNd (const int parGrpNo, const int grpNo, Vec<int> &grpExNdNoLs);


	/**
		compute the distance between two groups, group 1 and group 2.
		@param grpNo1 is a root of subtree 1
		@param grpNo2 is a root of subtree 2 
		@param parGrpNo is the parent of grpNo1, and grpNo2
		@return the distance between two groups
	*/
	double cmpGrpDis (const int parGrpNo, const int grpNo1, const int grpNo2);
	
	double cmpGrpDis (Vec<int> &grpExNdNoLs1, Vec<int> &grpExNdNoLs2);
	
	double cmpGrpDis (int exNdNo, Vec<int> &grpExNdNoLs);


	/**
		using Rzhetsky and Nei method to estimate 
		the branch of this tree as
		a starting point for maximum likelihood
	*/
	double estExBrLen (Br<double> &exBr);

	/**
		using Rzhetsky and Nei method to estimate 
		the branch of this tree as
		a starting point for maximum likelihood
	*/
	double estInBrLen (Br<double> &inBr);

	/**
		using Rzhetsky and Nei method to estimate 
		the branch of this tree as
		a starting point for maximum likelihood
	*/
	void estBrLen ();


	/**
		assign values for all LiNd_ of all LiNd
	*/
	void cmpLiNd ();

	/**
		assign values for all LiNd_ of all LiNd
	*/
	void cmpHeadLiNd ();

	/**
		compute the likelihood at a given pattern site
	*/
	LDOUBLE cmpPtnLi (int ptnNo_);


	/**
		cmp the log likelihood of this tree
	*/
	double cmpLogLi ();


	/**
		serve for cmpLiBr and cmpLogLiBr
	*/
	void changeBr (double len);

	//cmp the negative log likelihood of this tree when only this br changes
	//using for optimizing by brent method
	virtual double cmpNegLogLi (double len);

	double cmpLogLiDerivatives (double brLen, double &logli_derv1, double &logli_derv2, bool calc_logli);



	//optimize the br whose name is brNo, return the length of br
	double optBr (int brNo);


	/**
		Newton-Raphson method
	*/
	double doNewtonRaphson(double guess, double &logli);

#ifdef PARALLEL
	void turnOnParallelPhase(bool par_logli);
	void turnOffParallelPhase();
	bool isParallelPhase(bool par_logli);
#endif // PARALLEL

private:
	/**
		sum of score in topology optimization
	*/
	double score_sum, original_logli, upper_bound_logli;


	int do_parallel_phase;

}
; //end of class optimizing unrooted tree

extern OptUrTree opt_urtree;

#endif // OPTIMALUNROOTTREE


