/***************************************************************************
 *                                                                         *
 *   basem.h     (begin: Mar 05 2003)                                      *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifndef BASEM_H
#define BASEM_H

#include "constant.h"
#include "mat.h"
#include "vec.h"
#include "dvec20.h"
#include "dmat20.h"
#include "brent.h"

const double SCALE_RATE = 0.01;
const double SCALE_RATE_CODON = 1.0;

/**
	abstract substitution model, all other models inherit from BaseM
*/
class BaseM : public Brent {
public:
	/**
		constructor function
	*/
	BaseM();

	/**
		set the model type
		@param modelType HKY85, TN93,...
	*/
	void setModel (MODEL modelType);

	/**
		find wether DNA model or not
		@return TRUE if model is a DNA model 
	*/
	int isDNA ();

	/**
		set the transition transvertion ratio
		@param tsTvRatio the transition transvertion ratio
	*/
	void setTsTvRatio (const double tsTvRatio);

	/**
		set the pyridimine/purine ratio
		@param pyPuRatio pyridimine/purine ratio
	*/
	void setPyPuRatio (const double pyPuRatio);
	
		
	/**
		set the state equilibrium frequencies 
		@param stateFrqArr the state equilibrium frequencies 
	*/
	void setStateFrq (DVec20 &stateFrqArr);

	/**
		set parameters for DNA Model
		@param tsAG transition frequency A<->G
		@param tsCT transition frequency C<->T
		@param tvAC transversion frequency A<->C
		@param tvAT transversion frequency A<->T
		@param tvCG transversion frequency C<->G
		@param tvGT transversion frequency G<->T
	*/
	void setGenPam (const double tsAG, const double tsCT,
	                const double tvAC, const double tvAT, const double tvCG, const double tvGT);

	/**
		get parameters for DNA Model
		@param tsAG (OUT) transition frequency A<->G
		@param tsCT (OUT) transition frequency C<->T
		@param tvAC (OUT) transversion frequency A<->C
		@param tvAT (OUT) transversion frequency A<->T
		@param tvCG (OUT) transversion frequency C<->G
		@param tvGT (OUT) transversion frequency G<->T
	*/
	void getGenPam (double &tsAG,  double &tsCT,
	                double &tvAC,  double &tvAT, double &tvCG, double &tvGT);

	/**
		get the state frequencies of this alignment
		@param stateFrqArr (OUT) the state equilibrium frequencies 
	*/
	void getStateFrq (DVec20 &stateFrqArr);

	/**
		get the Transition/Transversion Ratio
		@return the Transition/Transversion Ratio
	*/
	virtual double getTsTvRatio ();
	
	/**
		get the pyridimine/purine ratio
		@return the pyridimine/purine ratio
	*/
	virtual double getPyPuRatio ();

	/**
		compute the probability of changing from stateNo1 into stateNo2 after a period time of time
		@param stateNo1 the first state
		@param stateNo2 the second state
		@param time the period of time
		@return the mutation probability
	*/
	virtual double cmpProbChange (const int stateNo1, const int stateNo2, const double time);
	
	/**
		compute the probability changes matrix
		@param time the period of time
		@param probMat (OUT) the probability changes matrix
	*/
	virtual void cmpProbChange (const double time, DMat20 &probMat);
	
	/**
		compute the probability change matrix, for codon-based Nielsen Yang 98 Model
		@param time the period of time
		@param classNo the NY98 class index
		@param probMat (OUT) the probability changes matrix
	*/
	virtual void cmpProbChange (const double time, const int classNo, DMat20 &probMat);
	
	/**
		compute the 1st and 2nd derivatives of log likelihood
		@param brLen the period of time
		@param probMat (OUT) the probability changes matrix
		@param derv1 (OUT) 1st derivative of the probability changes matrix
		@param derv2 (OUT) 2nd derivative of the probability changes matrix
	*/
    virtual void cmpProbChangeDerivatives (const double brLen, 
		DMat20 &probMat, DMat20 &derv1, DMat20 &derv2);
	
	
	/**
		compute the 1st and 2nd derivatives of log likelihood, for codon-based Nielsen Yang 98 Model
		@param brLen the period of time
		@param classNo the NY98 class index
		@param probMat (OUT) the probability changes matrix
		@param derv1 (OUT) 1st derivative of the probability changes matrix
		@param derv2 (OUT) 2nd derivative of the probability changes matrix
	*/
    virtual void cmpProbChangeDerivatives (const double brLen, const int classNo, 
		DMat20 &probMat, DMat20 &derv1, DMat20 &derv2);
	
	/**
		all things are inited here
	*/
	virtual void init ();

	/**
		optimize parameters, dedicated only to TN93 model
		@return TRUE of some parameters changed
	*/
	virtual bool optPam (const PAM_TYPE tsTvRatioType, const PAM_TYPE pyPuRatioType);
	
	/**
		optimize parameters
		@return TRUE of some parameters changed
	*/
	virtual bool optPam ();

	/* The following are internal functions, some are taken
		from the book Numerical Recipes, Press et al. (1992)
	*/

	/**
		convert a non-symmetric matrix into Hessenberg form with zeros everywhere
		below the diagonal except for the first sub-diagonal row
		@param a (IN-OUT) the matrix
		@param ordr (OUT) the order of columns
		@param n (IN) size of matrix 
	*/
	void elmhes(DMat20 a, Vec<int> &ordr, int n);

	/*
		something here
	*/
	void eltran(DMat20 a, DMat20 zz, Vec<int> &ordr, int n);

	/*
		something here
	*/
	void mcdiv(double ar, double ai, double br, double bi,
	           double *cr, double *ci);

	/**
		QR algorithm for non-symmetric matrix to calculate eigenvectors and eigenvalues
		of a Hessenberg matrix (should be preceded by elmhes function)
		@param n (IN) size of matrix 
	*/
	void hqr2(int n, int low, int hgh, DMat20 h,
	          DMat20 zz, DVec20 &wr, DVec20 &wi);

	/**
		normalize the rate matrix
		@param a (IN-OUT) the rate matrix
		@param stateFrq (IN) the state frequencies
		@param num_state the number of state
	*/			  
	void onepamratematrix(DMat20 a, DVec20 stateFrq, int num_state);

	/**
		EigenSystem for non-symmetric matrix
		@param eval (OUT) eigenvalues
		@param evec (OUT) eigenvectors
		@param num_state (IN) number of states
	*/
	void eigensystem(DVec20 &eval, DMat20 &evec, int num_state);

	/**
		compute the inverse of a square matrix
		@param inmat (IN) the matrix
		@param imtrx (OUT) the inverse of the input matrix
		@param size the size of matrix
	*/
	void luinverse(DMat20 inmat, DMat20 imtrx, int size);

	/**
		check the correctness of the inverse operation
	*/
	void checkevector(DMat20 evec, DMat20 ivec, int nn);
	
// added by BQM 2006-08-11
	
	/**
		convert the rate matrix into symmetric form
		@param a (IN-OUT) the rate matrix
		@param stateFrq (IN) the state frequencies
		@param stateFrq_sqrt (OUT) square root of state frequencies
		@param num_state the number of states
	*/
	void transformRateMatrix(DMat20 a, DVec20 stateFrq, DVec20 stateFrq_sqrt, int num_state);

	void eliminateZero(DMat20 &mat, DVec20 &forg, int num, 
		DMat20 &new_mat, DVec20 &new_forg, int &new_num);
	
	/**
		EigenSystem for symmetric matrix
		@param eval (OUT) eigenvalues
		@param evec (OUT) eigenvectors
		@param inv_evec (OUT) inverse matrix of eigenvectors
		@param num_state (IN) number of states
	*/
	void eigensystem_new(DVec20 &eval, DMat20 &evec, DMat20 &inv_evec, int num_state);

// end BQM 2006-08-11
	
	/**
		get the rate parameters and frequency vector
		@param a[0..n-1] (OUT) rate matrix
		@param forg[0..n-1] (OUT) frequency vector
		@param num_state (IN) number of state, 4 for DNA, 20 for protein, 61 for codon
	*/
	virtual void rtfdata (DMat20 &a, DVec20 &forg, int num_state);

	/**
		write matrix
	*/
	void writeMat (DMat20 &rate, int num_state);
	/**
		write int vector
	*/
	void writeInt (Vec<int> &vec, int num_state);
	/**
		write double vector
	*/
	void writeDouble (DVec20 &vec, int num_state);
	
	/**
		calculate eigen coefficients
		@param num_state the number of states
	*/
	void calcEigenCoeff(int num_state);

	/**
		compute information to efficiently calculate the prob matrix
		@param num_state the number of states
	*/	
	void tranprobmat(int num_state);

	/**
		older version to compute information to efficiently calculate the prob matrix
		using eigensystem for non-symmetric matrix
	*/
	void tranprobmat_old(int num_state);
	
	/**
		release all memmory for this class
	*/
	void release ();

	/**
		the destructor function
	*/
	virtual ~BaseM();


	/************************************************************************************
	************************************************************************************/
	/**
		the total number of substitutions per unit time
	*/
	double total_num_subst;
	
protected :

	/**
		model type
	*/
	MODEL modelType_;

	/**
		number of states, 4 for DNA, 20 for Amino acid and 61 for Codon
	*/
	int nState_;
	
	/**
		active parameter, used in optimization phase
	*/
	MODEL_PAM_TYPE isActPam_;
	
	/**
		the transition transvertion ratio
	*/
	double tsTvRatio_;
	
	/**
		the pyridimine/purine ratio
	*/
	double pyPuRatio_;

	/**
		A -> G ratio
	*/
	double tsAG_;
	
	/**
		C -> T ratio
	*/
	double tsCT_;

	/**
		A -> C ratio
	*/
	double tvAC_;
	
	/**
		A -> T ratio
	*/
	double tvAT_;
	
	/**
		C -> G ratio
	*/
	double tvCG_;
	
	/**
		G -> T ratio
	*/
	double tvGT_;

	/**
		the state frequency list
	*/
	DVec20 stateFrqArr_;
	
	//double tsTvRatioError_;

	/**
		unit rate matrix
	*/
	DMat20 unitRateMat_;
	
	/**
		eigenvectors matrix
	*/
	DMat20 eigVecMat_;
	
	/**
		inversion of eigenvectors matrix
	*/
	DMat20 invEigVecMat_;
	
	/**
		eigenvalues 
	*/
	DVec20 eigValArr_;
	
	/**
		eigen coefficient, for 1st derivative, for 2nd derivative
	*/
	DMat20 eigen_coeff[MAX_STATE_NUM], eigen_coeff_derv1[MAX_STATE_NUM], 
		eigen_coeff_derv2[MAX_STATE_NUM];
	
	DCube20 eigen_coeff_vec[MAX_NUM_CLASS], eigen_coeff_derv1_vec[MAX_NUM_CLASS],
		eigen_coeff_derv2_vec[MAX_NUM_CLASS];
	DVec20 eigValArr_vec[MAX_NUM_CLASS];
	
	/**
		compute the substitution rate per unit time for this model
		@return total substitution rate
	*/
	double cmpSubRate ();

	/**
		cmp the state frequencies for this alignment
	*/
	virtual void cmpStateFrq ();

	void printProbInfo(const double brLen, const int classNo, DMat20 &probMat);
	void printProbInfo(const double brLen, DMat20 &probMat);

// BQM 2006-08-10
// the following were added to calculate eigenvalues, eigenvectors
// using Householder + QL method for real symmetric matrix
// adopted from Numerical Recipes chapter 11

	/**
		Householder transformation of symmetric matrix A
		to tridiagonal form 
		@param a the input matrix, must be symmetric. On output,
			a is replaced by the orthogonal matrix effecting the transformation
		@param  n the size of matrix a
		@param d [0..n-1] returned the diagonal elements of the tridiagonal matrix
		@param e [0..n-1] returned the off-diagonal elements with e[0]=0
	*/
	void tred2(DMat20 &a, int n, DVec20 &d, DVec20 &e);

	/**
		QL algorithm with implicit shifts to determine eigenvalues and
		eigenvectors of a real tridiagonal symmetric matrix.
		@param d [0..n-1] diagonal elements of the tridiagonal matrix. 
			On output d return the eigenvalues.
		@param e [0..n-1] off-diagonal elements of the tridiagonal matrix, e[0] arbitrary.
			On output e is destroyed.
		@param z must be input as the matrix returned by tred2
			z[k] return the normalized eigenvector corresponding to d[k]
	*/
	void tqli(DVec20 &d, DVec20 &e, int n, DMat20 &z);

	double pythag(double a, double b);

}
; //end of stateM class

#endif
