//
// jja: swiss army knife for chess file formats
// src/lib.rs: Common utility functions
//
// Copyright (c) 2023 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0-or-later

// We like clean and simple code with documentation.
// Keep in sync with main.rs.
// See: https://rust-lang.github.io/rust-clippy/master/index.html for documentation.
#![deny(missing_docs)]
#![deny(clippy::allow_attributes_without_reason)]
// TODO: Make the code free of arithmetic side effects.
// #![deny(clippy::arithmetic_side_effects)]
#![deny(clippy::as_ptr_cast_mut)]
#![deny(clippy::as_underscore)]
#![deny(clippy::assertions_on_result_states)]
#![deny(clippy::borrow_as_ptr)]
#![deny(clippy::branches_sharing_code)]
#![deny(clippy::case_sensitive_file_extension_comparisons)]
#![deny(clippy::cast_lossless)]
//#![deny(clippy::cast_possible_truncation)]
//#![deny(clippy::cast_possible_wrap)]
#![deny(clippy::cast_precision_loss)]
#![deny(clippy::cast_ptr_alignment)]
//#![deny(clippy::cast_sign_loss)]
#![deny(clippy::checked_conversions)]
#![deny(clippy::clear_with_drain)]
#![deny(clippy::clone_on_ref_ptr)]
#![deny(clippy::cloned_instead_of_copied)]
#![deny(clippy::cognitive_complexity)]
#![deny(clippy::collection_is_never_read)]
#![deny(clippy::copy_iterator)]
#![deny(clippy::create_dir)]
#![deny(clippy::dbg_macro)]
#![deny(clippy::debug_assert_with_mut_call)]
#![deny(clippy::decimal_literal_representation)]
#![deny(clippy::default_trait_access)]
#![deny(clippy::default_union_representation)]
#![deny(clippy::derive_partial_eq_without_eq)]
#![deny(clippy::doc_link_with_quotes)]
#![deny(clippy::doc_markdown)]
#![deny(clippy::explicit_into_iter_loop)]
#![deny(clippy::explicit_iter_loop)]
// TODO: #![deny(clippy::fallible_impl_from)]
#![deny(clippy::missing_safety_doc)]
#![deny(clippy::undocumented_unsafe_blocks)]

//! The main module contains various sub-modules related to chess file formats, opening databases,
//! ECO classification, text editing, and other utilities.

// I18n
// When the "i18n" feature is enabled, use the tr macro from the tr crate
#[cfg(feature = "i18n")]
pub use tr::tr;

/// When the "i18n" feature is disabled, define the tr macro to perform formatting without translation
#[cfg(not(feature = "i18n"))]
#[macro_export]
macro_rules! tr {
    ($s:expr) => {
        $s
    };
    ($s:expr, $($arg:expr),* $(,)?) => {
        format!($s, $($arg),*)
    };
}

/// Arena book constants and utilities
pub mod abk;
/// Interface to Arena book files (aka 'abk')
pub mod abkbook;
/// Brainlearn experience file constants and utilities
pub mod brainlearn;
/// Brainlearn experience file interface
pub mod brainlearnfile;
/// Chess logic
pub mod chess;
/// CTG constants and utilities
pub mod ctg;
/// CTG book file interface
pub mod ctgbook;
/// ECO classification
pub mod eco;
/// Error handling
pub mod error;
/// Utilities for binary file I/O
pub mod file;
/// Utilities for Zobrist Hashing
pub mod hash;
/// Lichess evaluations export file constants and utilities
pub mod lieval;
/// Polyglot Book Merging
pub mod merge;
/// Chessmaster book constants and utilities
pub mod obk;
/// Interface to Chessmaster book files (aka 'obk')
pub mod obkbook;
/// Portable Game Notation utilities
pub mod pgn;
/// Portable Game Notation parsing and import
pub mod pgnbook;
/// Portable Game Notation header filtering
pub mod pgnfilt;
/// Polyglot constants and utilities
pub mod polyglot;
/// Polyglot book file interface
pub mod polyglotbook;
/// Chess Quotes
pub mod quote;
/// Random playouts using books
pub mod random;
/// Reexecuting Self for fun and profit
pub mod reexec;
/// Stockfish compatible Zobrist hashing
pub mod stockfish;
/// External system interface (tty detection, invoke editor, ...)
pub mod system;

/// A module containing build-time information.
pub mod built_info {
    // The file has been placed there by the build script.
    include!(concat!(env!("OUT_DIR"), "/built.rs"));
}
