// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package users

import (
	"bytes"
	"context"
	"database/sql"
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/drone/drone/core"
	"github.com/drone/drone/handler/api/errors"
	"github.com/drone/drone/mock"

	"github.com/go-chi/chi"
	"github.com/golang/mock/gomock"
	"github.com/google/go-cmp/cmp"
)

func TestUpdate(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	admin := true
	userInput := &userInput{
		Admin: &admin,
	}
	user := &core.User{
		Login: "octocat",
		Admin: false,
	}

	users := mock.NewMockUserStore(controller)
	users.EXPECT().FindLogin(gomock.Any(), user.Login).Return(user, nil)
	users.EXPECT().Update(gomock.Any(), user)

	transferer := mock.NewMockTransferer(controller)
	transferer.EXPECT().Transfer(gomock.Any(), user).Return(nil)

	c := new(chi.Context)
	c.URLParams.Add("user", "octocat")

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(userInput)
	w := httptest.NewRecorder()
	r := httptest.NewRequest("PATCH", "/", in)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(users, transferer)(w, r)
	if got, want := w.Code, 200; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	if got, want := user.Admin, true; got != want {
		t.Errorf("Want user admin %v, got %v", want, got)
	}

	got, want := new(core.User), user
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) > 0 {
		t.Errorf(diff)
	}
}

func TestUpdate_BadRequest(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	users := mock.NewMockUserStore(controller)

	c := new(chi.Context)
	c.URLParams.Add("user", "octocat")

	in := new(bytes.Buffer)
	w := httptest.NewRecorder()
	r := httptest.NewRequest("PATCH", "/", in)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(users, nil)(w, r)
	if got, want := w.Code, 400; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), &errors.Error{Message: "EOF"}
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) > 0 {
		t.Errorf(diff)
	}
}

func TestUpdate_NotFound(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	users := mock.NewMockUserStore(controller)
	users.EXPECT().FindLogin(gomock.Any(), mockUser.Login).Return(nil, sql.ErrNoRows)

	c := new(chi.Context)
	c.URLParams.Add("user", "octocat")

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(mockUser)
	w := httptest.NewRecorder()
	r := httptest.NewRequest("PATCH", "/", in)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(users, nil)(w, r)
	if got, want := w.Code, 404; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), &errors.Error{Message: "sql: no rows in result set"}
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) > 0 {
		t.Errorf(diff)
	}
}

func TestUpdate_UpdateFailed(t *testing.T) {
	controller := gomock.NewController(t)
	defer controller.Finish()

	userInput := &core.User{
		Login: "octocat",
		Admin: true,
	}
	user := &core.User{
		Login: "octocat",
		Admin: false,
	}

	users := mock.NewMockUserStore(controller)
	users.EXPECT().FindLogin(gomock.Any(), userInput.Login).Return(user, nil)
	users.EXPECT().Update(gomock.Any(), user).Return(errors.ErrNotFound)

	c := new(chi.Context)
	c.URLParams.Add("user", "octocat")

	in := new(bytes.Buffer)
	json.NewEncoder(in).Encode(mockUser)
	w := httptest.NewRecorder()
	r := httptest.NewRequest("PATCH", "/", in)
	r = r.WithContext(
		context.WithValue(context.Background(), chi.RouteCtxKey, c),
	)

	HandleUpdate(users, nil)(w, r)
	if got, want := w.Code, http.StatusInternalServerError; want != got {
		t.Errorf("Want response code %d, got %d", want, got)
	}

	got, want := new(errors.Error), errors.ErrNotFound
	json.NewDecoder(w.Body).Decode(got)
	if diff := cmp.Diff(got, want); len(diff) > 0 {
		t.Errorf(diff)
	}
}
