require 'common/plugin.rb'

#
# Plugins for the removal of users and domains.
#
module RmPlugin

  # Absorb the subclass run() magic from the Plugin::Run module.
  extend Plugin::Run

  # The runner class associated with removal plugins.
  #
  # @return [Class] the {RmRunner} class.
  #
  def self.runner()
    return RmRunner
  end


  # The "dummy" runner class associated with removal plugins.
  #
  # @return [Class] the {RmDummyRunner} class.
  #
  def self.dummy_runner()
    return RmDummyRunner
  end


  # Remove the *target* domain or user. This is a generic version of
  # the {#remove_user} and {#remove_domain} operations that will
  # dispatch based on the type of *target*.
  #
  # @param target [User,Domain] the user or domain to remove.
  #
  def remove(target)
    # A generic version of remove_user/remove_domain that
    # dispatches base on the class of the target.
    if target.is_a?(User)
      return remove_user(target)
    elsif target.is_a?(Domain)
      return remove_domain(target)
    else
      raise NotImplementedError
    end
  end


  # Remove the given *domain*. Some plugins don't have a concept of
  # domains, so the default implementation here removes all users that
  # look like they belong to *domain*. Subclasses can be smarter.
  #
  # @param domain [Domain] the domain to remove.
  #
  def remove_domain(domain)
    users = list_domains_users([domain])

    # It's possible for a domain to exist with no users, but this
    # default implementation is based on the assumption that it should
    # work for plugins having no "domain" concept.
    raise NonexistentDomainError.new(domain.to_s()) if users.empty?

    users.each do |u|
      remove_user(u)
    end
  end


  # The interface for the "remove a user" operation. Subclasses
  # need to implement this method so that it removes *user*.
  #
  # @param user [User] the user to remove.
  #
  def remove_user(user)
    raise NotImplementedError
  end

end
