require 'pg'

require 'common/postfixadmin_plugin'
require 'mv/mv_plugin'


# Handle moving (renaming) of users in the Postfixadmin database.
#
class PostfixadminMv

  include PostfixadminPlugin
  include MvPlugin


  # Move the user *src* to *dst* within the Postfixadmin
  # database. This should "rename" him in _every_ table where he is
  # referenced. Unfortunately that must be done manually.
  #
  # This can fail is *src* does not exist, or if *dst* already exists
  # before the move. It will also fail if the domain associated with
  # the user *dst* does not exist.
  #
  # @param src [User] the source user to be moved.
  #
  # @param dst [User] the destination user being moved to.
  #
  def mv_user(src, dst)
    raise NonexistentUserError.new(src.to_s()) if not user_exists(src)

    if not domain_exists(dst.domain())
      raise NonexistentDomainError.new(dst.domain.to_s())
    end

    raise UserAlreadyExistsError.new(dst.to_s()) if user_exists(dst)

    mailbox_query  = 'UPDATE mailbox SET '
    mailbox_query += '  username=$1,'
    mailbox_query += '  domain=$2,'
    mailbox_query += "  maildir=CONCAT($2, '/', $3, '/'),"
    mailbox_query += '  local_part=$3 '
    mailbox_query += 'WHERE username=$4;'

    alias_query1  = 'UPDATE alias SET '
    alias_query1 += '  address=$1,'
    alias_query1 += '  domain=$2,'
    alias_query1 += '  goto=REPLACE(goto, $4, $1) '
    alias_query1 += 'WHERE address=$4;'

    alias_query2  = 'UPDATE alias SET '
    alias_query2 += 'goto=REPLACE(goto, $4, $1);'

    sql_queries = [mailbox_query, alias_query1, alias_query2]

    connection = PG::Connection.new(@db_hash)
    begin
      sql_queries.each do |sql_query|
        varchar = 1043 # from pg_type.h
        params = [{:value => dst.to_s(), :type => varchar},
                  {:value => dst.domainpart(), :type => varchar},
                  {:value => dst.localpart(), :type => varchar},
                  {:value => src.to_s(), :type => varchar}]
        connection.sync_exec_params(sql_query, params)
      end
    ensure
      # Make sure the connection gets closed even if a query explodes.
      connection.close()
    end
  end

end
