require 'common/domain'
require 'common/errors'
require 'common/runner'

# Perform the moving (renaming) of users/domains using {MvPlugin}s.
#
class MvRunner
  include Runner

  # Run *plugin* to move the user *src* to *dst*. The method
  # signature includes the unused *cfg* for consistency with the
  # runners that do need a {Configuration}.
  #
  # @param cfg [Configuration] unused.
  #
  # @param plugin [Class] plugin class that will perform the move.
  #
  # @param src [User] the source user to be moved.
  #
  # @param dst [User] the destination user being moved to.
  #
  def run(cfg, plugin, src, dst)

    if src.is_a?(Domain) or dst.is_a?(Domain) then
      msg = 'only users can be moved'
      raise NotImplementedError.new(msg)
    end

    begin
      src_description = plugin.describe(src)
      plugin.mv_user(src, dst)
      dst_description = plugin.describe(dst)

      msg  = "Moved user "
      msg += add_description(src, src_description)
      msg += " to "
      msg += add_description(dst, dst_description)
      msg += "."
      report(plugin, msg)

    rescue NonexistentUserError
      # This means that the SOURCE user didn't exist, since a
      # nonexistent destination user is perfectly expected.
      report(plugin, "Source user #{src.to_s()} not found.")
    rescue NonexistentDomainError
      # This could mean that the source domain doesn't exist, but in
      # that case, we just report that the source user doesn't
      # exist. So a nonexistent domain refers to a nonexistent
      # DESTINATION domain.
      report(plugin, "Destination domain #{dst.domainpart()} not found.")
    rescue UserAlreadyExistsError
      report(plugin, "Destination user #{dst.to_s()} already exists.")
    end
  end

end
