require 'common/plugin'
require 'common/user'

# Code that all Roundcube plugins ({RoundcubePrune}, {RoundcubeRm},
# {RoundcubeMv}) share.
#
module RoundcubePlugin

  # We implement the Plugin "interface."
  include Plugin


  # Initialize this Roundcube {Plugin} with values in *cfg*.
  #
  # @param cfg [Configuration] the configuration for this plugin.
  #
  def initialize(cfg)
    @db_hash = {
      :host     => cfg.roundcube_dbhost,
      :port     => cfg.roundcube_dbport,
      :options  => cfg.roundcube_dbopts,
      :dbname   => cfg.roundcube_dbname,
      :user     => cfg.roundcube_dbuser,
      :password => cfg.roundcube_dbpass }
  end


  # Describe the given Roundcube *user*.
  #
  # @param user [User] the user whose description we want.
  #
  # @return [String] a string containing the Roundcube "User ID"
  #   associated with *user*.
  #
  def describe_user(user)
    user_id = self.get_user_id(user)
    return "User ID: #{user_id}"
  end


  # Return a list of Roundcube users.
  #
  # @return [Array<User>] a list of users contained in the
  #   Roundcube database.
  #
  def list_users()
    usernames = []

    connection = PG::Connection.new(@db_hash)

    sql_query = 'SELECT username FROM users;'

    begin
      connection.sync_exec(sql_query) do |result|
        usernames = result.field_values('username')
      end
    ensure
      # Make sure the connection gets closed even if the query explodes.
      connection.close()
    end

    return usernames.map{ |u| User.new(u) }
  end

  protected;


  # Find the Roundcube "User ID" associated with the given *user*.
  #
  # @param user [User] the user whose Roundcube "User ID" we want.
  #
  # @return [Fixnum] the Roundcube "User ID" for *user*.
  #
  def get_user_id(user)
    user_id = nil

    connection = PG::Connection.new(@db_hash)
    sql_query = 'SELECT user_id FROM users WHERE username = $1;'

    begin
      connection.sync_exec_params(sql_query, [user.to_s()]) do |result|
        if result.num_tuples > 0
          user_id = result[0]['user_id']
        end
      end
    ensure
      # Make sure the connection gets closed even if the query explodes.
      connection.close()
    end

    return user_id
  end


end
