require 'yaml'

# A configuration object that knows how to read options out of a file
# in <tt>$XDG_CONFIG_HOME/mailshears/mailshears.conf.yml</tt>. The
# configuration options can be accessed via methods even though the
# internal representation is a hash.
#
# === Examples
#
#   >> cfg = Configuration.new()
#   >> cfg.i_mean_business()
#   => true
#
class Configuration

  # The hash structure in which we store our configuration options
  # internally.
  @dict = {}

  # Initialize a {Configuration} object with the config file at *path*.
  #
  # @param path [String] the path to the configuration file to
  #   load. We check for a file named "mailshears.conf.yml" in the
  #   user's XDG configuration directory by default.
  #
  def initialize(path = nil)
    if path.nil? then
        # The default path to the user's configuration file.
      path = ENV['HOME'] + '/.config'
      if ENV.has_key?('XDG_CONFIG_HOME') then
        path = ENV['XDG_CONFIG_HOME']
      end
      path += '/mailshears/mailshears.conf.yml'
    end

    cfg = default_configuration()

    # Now, load the user configuration which will override the
    # variables defined above.
    begin
      user_config = YAML.load(File.open(path))

      # Write our own update() method for Ruby 1.8.
      user_config.each do |key, value|
        cfg[key] = value
      end
    rescue Errno::ENOENT
      # If the user config file doesn't exist, whatever.
    end

    # Convert all of the keys to symbols.
    cfg = cfg.inject({}){|memo,(k,v)| memo[k.to_sym] = v; memo}

    @dict = cfg
  end


  # Replace all missing method calls with hash lookups. This lets us
  # retrieve the values in our option hash by using methods named
  # after the associated keys.
  #
  # @param sym [Symbol] the method that was called.
  #
  # @return [Object] the config file value associated with *sym*.
  #
  def method_missing(sym, *args)
    return @dict[sym]
  end


  private;


  # A default config hash.
  #
  # @return [Hash] sensible default configuration values.
  #
  def default_configuration()
    d = {}

    d['i_mean_business'] = false
    d['plugins'] = ['postfixadmin']

    d['davical_dbhost'] = 'localhost'
    d['davical_dbport'] = 5432
    d['davical_dbopts'] = ''
    d['davical_dbuser'] = 'postgres'
    d['davical_dbpass'] = ''
    d['davical_dbname'] = 'davical'

    d['dovecot_mail_root'] = '/tmp/mailshears-test'

    d['postfixadmin_dbhost'] = 'localhost'
    d['postfixadmin_dbport'] = 5432
    d['postfixadmin_dbopts'] = ''
    d['postfixadmin_dbuser'] = 'postgres'
    d['postfixadmin_dbpass'] = ''
    d['postfixadmin_dbname'] = 'postfixadmin'

    d['roundcube_dbhost'] = 'localhost'
    d['roundcube_dbport'] = 5432
    d['roundcube_dbopts'] = ''
    d['roundcube_dbuser'] = 'postgres'
    d['roundcube_dbpass'] = ''
    d['roundcube_dbname'] = 'roundcube'

    return d
  end

end
