/***************************************************************************
Copyright (c) 2021, The OpenBLAS Project
All rights reserved.
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:
1. Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in
the documentation and/or other materials provided with the
distribution.
3. Neither the name of the OpenBLAS project nor the names of
its contributors may be used to endorse or promote products
derived from this software without specific prior written permission.
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE OPENBLAS PROJECT OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*****************************************************************************/

#include "common.h"

int CNAME(int transa, int transb, BLASLONG M, BLASLONG N, BLASLONG K, FLOAT alpha, FLOAT beta)
{
  double MNK = (double) M * (double) N * (double) K;

#if defined(DOUBLE) // dgemm

  // gcc11 (minor <= 2) has an issue when multiple assemble_pairs are used. This
  // issue affects both dgemm_nn and dgemm_tn.
#if (defined(__GNUC__) && (__GNUC__ == 11 && __GNUC_MINOR__ <= 2))
  if (!transb)
    return 0;
#endif

  if (MNK <= 54.0*54.0*54.0)
    return 1;

#else // sgemm

#if defined(__GNUC__) && defined(__clang__)
  // clang generates code with register spilling for the region of code with
  // packing, thus, we had to disable this optimization for clang. Given that
  // the packing on-demand used in this work is one of the reasons that lead the
  // small kernels to outperform the normal flow (when MNK increases), with it
  // disabled we had to reduce the MNK inputs used by the code generated by clang.
  if (MNK > 84.0*84.0*84.0)
    return 0;

  if (transa && !transb) {
    // sgemm_tn works better when packing on-demand is used
    if (MNK <= 64.0*64.0*64.0 && K >= 4)
      return 1;
    else
      return 0;
  }
  
#else // gcc

   if (MNK > 100.0*100.0*100.0)
     return 0;

#endif

#ifdef SMP
  // Multi-threading execution outperforms (or approaches) the execution of the
  // small kernel.
  if (num_cpu_avail(3) > 1) {
    if (MNK <= 64.0*64.0*64.0)
      return 1;
  } else {
    return 1;
  }
#else
  return 1;
#endif

#endif

  return 0;
}
