#! /bin/sh
# A script to clean or rebuild files created by autoconf, automake,
# libtool, gettext and gnulib. Run autogen.sh --help for usage.
#
# This software is copyrighted work licensed under the terms of the
# GNU General Public License. Please consult the file `COPYING' for
# details.
#
# Author: Oskar Liljeblad <oskar@osk.mine.nu>
# Version: 2

set -e

GNULIBTOOL=/usr/share/gnulib/gnulib-tool
GNULIBPATH=/usr/share/gnulib
GETTEXTPATH=/usr/share/gettext

function m4extract () {
  local file="$1"
  local macro="$2"
  local arg="$3"

  local res
  res="`m4 --trace=$macro -de -D$macro=$arg $file 2>&1 >/dev/null`"
  res="`echo $res | sed 's/^.* -> \(.*\)$/\1/;s/^\[//;s/\]$//;p;d'`"
  echo "$res"
}

function contains () {
  local find="$1"
  shift
  for str in "$@"; do
    if [ "$str" = "$find" ]; then
      return 0
    fi
  done
  return 1
}

function expand_module_dependencies () {
  local knownlist="$@"
  local checklist="$@"

  while [ -n "$checklist" ]; do
    newlist=`$GNULIBTOOL --extract-dependencies $checklist`
    checklist=""
    for module in $newlist; do
      if ! contains $module $knownlist; then
         knownlist="$knownlist $module"
         checklist="$checklist $module"
      fi
    done
  done

  echo $knownlist
}

if contains --help "$@"; then
  echo "Usage: $0 --clean|--generate|--help"
  exit
fi

package="`m4extract configure.ac AC_INIT '$1'`"
if [ "$package" = "" ]; then
  echo "Cannot get package name from configure.ac!"
  exit 1
fi

modulelist=modules.gnulib
if [ -f $modulelist ]; then
  modules=`cat modules.gnulib | egrep -v '^[[:space:]]*#'`
  modules=`expand_module_dependencies $modules`
  filelist=`$GNULIBTOOL --extract-filelist $modules | sort | uniq`
else
  modules=""
  filelist=""
fi

if contains --clean "$@"; then
  cleanup=1
  if contains --generate "$@"; then
    regenerate=1
  else
    regenerate=0
  fi
else
  cleanup=0
  regenerate=1
fi

if [ "$cleanup" = "1" ]; then
  if [ -f Makefile ]; then
    make maintainer-clean || :
  fi
  echo Removing generated files
  # Generated by autoheader
  rm -f config.h.in config.h.in~
  # Generated by aclocal
  rm -f aclocal.m4
  # Added by automake -a
  rm -f COPYING INSTALL compile config.guess config.sub depcomp install-sh missing
  # Generated by automake
  for file in `m4extract configure.ac AC_CONFIG_FILES '$1'`; do
    if [ -f "$file.am" ]; then
      rm -f "$file.in"
    fi
  done
  # Generated by autoconf
  rm -f configure
  # Added by libtoolize
  rm -f ltmain.sh
  # Added by gettextize
  rm -f ABOUT-NLS config.rpath mkinstalldirs
  if [ "`m4extract configure.ac AM_GNU_GETTEXT '$1'`" != "" ]; then
    if [ -d po ]; then
      for file in /usr/share/gettext/po/*; do
        rm -f po/`basename $file`
      done
    fi
    if [ -d intl ]; then
      rm -rf intl
    fi
    # Generated by make at build time
    if [ -d po ]; then
      rm -f po/$package.pot
    fi
  fi
  # Added by gnulib
  if [ -f $modulelist ]; then
    rm -f m4/Makefile.am
    rm -f lib/Makefile.am
    echo Removing gnulib files
    for file in $filelist; do
       if [ "${file:0:3}" = "m4/" -o "${file:0:4}" = "lib/" ]; then
        rm -f "$file"
      fi
    done
  fi
fi

if [ "$regenerate" = "1" ]; then
  if [ -f $modulelist ]; then
    echo "Adding symbolic links to gnulib files"
    mkdir -p m4 lib
    for file in $filelist; do
      if [ "${file:0:3}" = "m4/" -o "${file:0:4}" = "lib/" ]; then
        rm -f $file
        ln -s $GNULIBPATH/$file $file
      fi
    done

    echo "Creating m4/Makefile.am"
    (cat m4/Makefile.am.head &&
     for file in $filelist; do
       if [ "${file:0:3}" = "m4/" ]; then
         echo "EXTRA_DIST += ${file:3}"
       fi
     done) > m4/Makefile.am

    echo "Creating lib/Makefile.am"
    (cat lib/Makefile.am.head &&
     $GNULIBTOOL --extract-automake-snippet --lib libgnu $modules |
     sed -e "s,lib_SOURCES,libgnu_la_SOURCES,g" -e "s,lib_OBJECTS,libgnu_la_OBJECTS,g") > lib/Makefile.am
  fi

  if [ "`m4extract configure.ac AM_GNU_GETTEXT '$1'`" != "" ]; then
    #echo "Running gettextize"
    #gettextize --no-changelog
    mkdir -p po
    echo "Installing gettext files"
    for file in mkinstalldirs config.rpath ABOUT-NLS; do
      rm -f $file
      ln -s $GETTEXTPATH/$file $file
    done
    for file in $GETTEXTPATH/po/*; do
      base=`basename "$file"`
      rm -f po/$base
      ln -s $file po/$base
    done
    mkdir -p intl
    for file in $GETTEXTPATH/intl/*; do
      base=`basename "$file"`
      if [ "$base" = "plural.c" ]; then
        cp $file intl/$base
        touch intl/$base
      elif [ "$base" != "COPYING.LIB-2.0" -a "$base" != "COPYING.LIB-2.1" ]; then
        rm -f intl/$base
        ln -s $file intl/$base
      fi
    done
    if [ -d lib ]; then
      echo "Updating po/POTFILES.in"
      (cat po/POTFILES.in \
        | grep -v '^lib/'; \
       ls lib/*.[ch]) > po/POTFILES.in.tmp && mv po/POTFILES.in.tmp po/POTFILES.in
    fi
    #echo "Updating AM_GNU_GETTEXT_VERSION in configure.ac"
    #newver=`sed 's/.* gettext-\([0-9].*\)/\1/' < $GETTEXTPATH/intl/VERSION`
    #cat configure.ac |
    #  sed "s/^AM_GNU_GETTEXT_VERSION(.*)/AM_GNU_GETTEXT_VERSION($newver)/" > configure.ac.tmp \
    # && mv configure.ac.tmp configure.ac
  fi

  if [ -f $modulelist ]; then
    echo "Updating gnulib commands in configure.ac"
    (state=0
     cat configure.ac | while IFS="" read line; do
       if [ $state = 0 ]; then
         echo "$line"
         if [ "$line" = "# Gnulib" ]; then
           echo gl_USE_SYSTEM_EXTENSIONS
           $GNULIBTOOL --extract-autoconf-snippet --lib libgnu $modules | \
  	   grep -v '^[ \t]*$' | \
  	   grep -v '^AM_GNU_GETTEXT(\[external\])$'
           state=1
         fi
       elif [ $state = 1 ]; then
         if [ "$line" = "" ]; then
           echo
           state=0
         fi
       fi
     done) > configure.ac.tmp \
      && mv configure.ac.tmp configure.ac
  fi

  echo "Running autoreconf -is"
  autoreconf -is

  #echo "Running autoheader"
  #autoheader
  #echo "Running aclocal"
  #aclocal -I m4
  #echo "Running automake"
  #automake -a
  #echo "Running autoconf"
  #autoconf
fi
