// Copyright (c) Microsoft Corporation.
// Licensed under the MIT License.

using System.Runtime.Serialization;

namespace System.Management.Automation
{
    /// <summary>
    /// This is a wrapper for exception class
    /// <see cref="System.ArgumentNullException"/>
    /// which provides additional information via
    /// <see cref="System.Management.Automation.IContainsErrorRecord"/>.
    /// </summary>
    /// <remarks>
    /// Instances of this exception class are usually generated by the
    /// PowerShell Engine.  It is unusual for code outside the PowerShell Engine
    /// to create an instance of this class.
    /// </remarks>
    public class PSArgumentNullException
            : ArgumentNullException, IContainsErrorRecord
    {
        #region ctor
        /// <summary>
        /// Initializes a new instance of the PSArgumentNullException class.
        /// </summary>
        /// <returns>Constructed object.</returns>
        public PSArgumentNullException()
            : base()
        {
        }

        /// <summary>
        /// Initializes a new instance of the PSArgumentNullException class.
        /// </summary>
        /// <param name="paramName"></param>
        /// <returns>Constructed object.</returns>
        /// <remarks>
        /// Per MSDN, the parameter is paramName and not message.
        /// I confirm this experimentally as well.
        /// </remarks>
        public PSArgumentNullException(string paramName)
            : base(paramName)
        {
        }

        /// <summary>
        /// Initializes a new instance of the PSArgumentNullException class.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="innerException"></param>
        /// <returns>Constructed object.</returns>
        public PSArgumentNullException(string message, Exception innerException)
            : base(message, innerException)
        {
            _message = message;
        }

        /// <summary>
        /// Initializes a new instance of the PSArgumentNullException class.
        /// </summary>
        /// <param name="paramName"></param>
        /// <param name="message"></param>
        /// <returns>Constructed object.</returns>
        /// <remarks>
        /// ArgumentNullException has this ctor form and we imitate it here.
        /// </remarks>
        public PSArgumentNullException(string paramName, string message)
            : base(paramName, message)
        {
            _message = message;
        }

        #region Serialization
        /// <summary>
        /// Initializes a new instance of the PSArgumentNullException class
        /// using data serialized via
        /// <see cref="System.Runtime.Serialization.ISerializable"/>
        /// </summary>
        /// <param name="info">Serialization information.</param>
        /// <param name="context">Streaming context.</param>
        /// <returns>Constructed object.</returns>
        [Obsolete("Legacy serialization support is deprecated since .NET 8", DiagnosticId = "SYSLIB0051")] 
        protected PSArgumentNullException(SerializationInfo info,
                           StreamingContext context)
        {
            throw new NotSupportedException();
        }        
        #endregion Serialization
        #endregion ctor

        /// <summary>
        /// Additional information about the error.
        /// </summary>
        /// <value></value>
        /// <remarks>
        /// Note that ErrorRecord.Exception is
        /// <see cref="System.Management.Automation.ParentContainsErrorRecordException"/>.
        /// </remarks>
        public ErrorRecord ErrorRecord
        {
            get
            {
                _errorRecord ??= new ErrorRecord(
                    new ParentContainsErrorRecordException(this),
                    _errorId,
                    ErrorCategory.InvalidArgument,
                    null);

                return _errorRecord;
            }
        }

        private ErrorRecord _errorRecord;
        private readonly string _errorId = "ArgumentNull";

        /// <summary>
        /// See <see cref="System.Exception.Message"/>
        /// </summary>
        /// <remarks>
        /// Exception.Message is get-only, but you can effectively
        /// set it in a subclass by overriding this virtual property.
        /// </remarks>
        /// <value></value>
        public override string Message
        {
            get { return string.IsNullOrEmpty(_message) ? base.Message : _message; }
        }

        private readonly string _message;
    }
}
