package static

import (
	"bytes"
	"net/http"
	"os"
	"strings"
	"time"
)

type fileSystem struct {
	files map[string]file
}

func (fs *fileSystem) Open(name string) (http.File, error) {
	name = strings.Replace(name, "//", "/", -1)
	f, ok := fs.files[name]
	if ok {
		return newHTTPFile(f, false), nil
	}
	index := strings.Replace(name+"/index.html", "//", "/", -1)
	f, ok = fs.files[index]
	if !ok {
		return nil, os.ErrNotExist
	}
	return newHTTPFile(f, true), nil
}

type file struct {
	os.FileInfo
	data []byte
}

type fileInfo struct {
	name    string
	size    int64
	mode    os.FileMode
	modTime time.Time
	isDir   bool

	files []os.FileInfo
}

func (f *fileInfo) Name() string {
	return f.name
}

func (f *fileInfo) Size() int64 {
	return f.size
}

func (f *fileInfo) Mode() os.FileMode {
	return f.mode
}

func (f *fileInfo) ModTime() time.Time {
	return f.modTime
}

func (f *fileInfo) IsDir() bool {
	return f.isDir
}

func (f *fileInfo) Readdir(count int) ([]os.FileInfo, error) {
	return make([]os.FileInfo, 0), nil
}

func (f *fileInfo) Sys() interface{} {
	return nil
}

func newHTTPFile(file file, isDir bool) *httpFile {
	return &httpFile{
		file:   file,
		reader: bytes.NewReader(file.data),
		isDir:  isDir,
	}
}

type httpFile struct {
	file

	reader *bytes.Reader
	isDir  bool
}

func (f *httpFile) Read(p []byte) (n int, err error) {
	return f.reader.Read(p)
}

func (f *httpFile) Seek(offset int64, whence int) (ret int64, err error) {
	return f.reader.Seek(offset, whence)
}

func (f *httpFile) Stat() (os.FileInfo, error) {
	return f, nil
}

func (f *httpFile) IsDir() bool {
	return f.isDir
}

func (f *httpFile) Readdir(count int) ([]os.FileInfo, error) {
	return make([]os.FileInfo, 0), nil
}

func (f *httpFile) Close() error {
	return nil
}

// New returns an embedded http.FileSystem
func New() http.FileSystem {
	return &fileSystem{
		files: files,
	}
}

// Lookup returns the file at the specified path
func Lookup(path string) ([]byte, error) {
	f, ok := files[path]
	if !ok {
		return nil, os.ErrNotExist
	}
	return f.data, nil
}

// MustLookup returns the file at the specified path
// and panics if the file is not found.
func MustLookup(path string) []byte {
	d, err := Lookup(path)
	if err != nil {
		panic(err)
	}
	return d
}

// Index of all files
var files = map[string]file{
	"/index.html": {
		data: file0,
		FileInfo: &fileInfo{
			name:    "index.html",
			size:    3539,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/reset.css": {
		data: file1,
		FileInfo: &fileInfo{
			name:    "reset.css",
			size:    990,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/timeago.js": {
		data: file2,
		FileInfo: &fileInfo{
			name:    "timeago.js",
			size:    3258,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/favicon.png": {
		data: file3,
		FileInfo: &fileInfo{
			name:    "favicon.png",
			size:    1804,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/icons/arrow-right.svg": {
		data: file4,
		FileInfo: &fileInfo{
			name:    "arrow-right.svg",
			size:    381,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/icons/skipped.svg": {
		data: file5,
		FileInfo: &fileInfo{
			name:    "skipped.svg",
			size:    430,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/icons/pending.svg": {
		data: file6,
		FileInfo: &fileInfo{
			name:    "pending.svg",
			size:    1093,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/icons/sleeping.svg": {
		data: file7,
		FileInfo: &fileInfo{
			name:    "sleeping.svg",
			size:    2524,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/icons/running.svg": {
		data: file8,
		FileInfo: &fileInfo{
			name:    "running.svg",
			size:    690,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/icons/failure.svg": {
		data: file9,
		FileInfo: &fileInfo{
			name:    "failure.svg",
			size:    551,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/icons/success.svg": {
		data: file10,
		FileInfo: &fileInfo{
			name:    "success.svg",
			size:    472,
			modTime: time.Unix(1567652579, 0),
		},
	},
	"/style.css": {
		data: file11,
		FileInfo: &fileInfo{
			name:    "style.css",
			size:    8564,
			modTime: time.Unix(1572549830, 0),
		},
	},
	"/logs.html": {
		data: file12,
		FileInfo: &fileInfo{
			name:    "logs.html",
			size:    3853,
			modTime: time.Unix(1567652579, 0),
		},
	},
}

//
// embedded files.
//

// /index.html
var file0 = []byte(`<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<title>Drone Runner Dashboard</title>
<link rel="stylesheet" type="text/css" href="reset.css">
<link rel="stylesheet" type="text/css" href="style.css">
<script src="timeago.js" type="text/javascript"></script>
</head>
<body>

<header class="navbar">
    <div class="logo">
        <svg viewBox="0 0 60 60" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><defs><path d="M12.086 5.814l-.257.258 10.514 10.514C20.856 18.906 20 21.757 20 25c0 9.014 6.618 15 15 15 3.132 0 6.018-.836 8.404-2.353l10.568 10.568C48.497 55.447 39.796 60 30 60 13.434 60 0 46.978 0 30 0 19.903 4.751 11.206 12.086 5.814zm5.002-2.97C20.998 1.015 25.378 0 30 0c16.566 0 30 13.022 30 30 0 4.67-1.016 9.04-2.835 12.923l-9.508-9.509C49.144 31.094 50 28.243 50 25c0-9.014-6.618-15-15-15-3.132 0-6.018.836-8.404 2.353l-9.508-9.508zM35 34c-5.03 0-9-3.591-9-9s3.97-9 9-9c5.03 0 9 3.591 9 9s-3.97 9-9 9z" id="a"></path></defs><use fill="#FFF" xlink:href="#a" fill-rule="evenodd"></use></svg>
    </div>
    <nav class="inline-nav">
        <ul>
            <li><a href="/">Dashboard</a></li>
            <li><a href="/logs">Logging</a></li>
        </ul>
    </nav>
</header>

<main>
    <section>
        <header>
            <h1>Dashboard</h1>
        </header>
        <div class="alert sleeping">
            <p>There is no recent activity to display.</p>
        </div>
        <article class="cards stages">
            <div class="card stage">
                <h2>{{ .Repo.Slug }}</h2>
                <img src="https://avatars0.githubusercontent.com/u/817538" />
                <span class="connector"></span>
                <span class="status {{ .Stage.Status }}"></span>
                <span class="desc">processing the <em>{{ .Stage.Name }}</em> stage for build <em>#{{ .Build.Number }}</em></span>
                <span class="time" datetime="{{ if .Stage.Updated }}{{ .Stage.Updated }}{{ else }}{{ .Stage.Created }}{{ end }}"></span>
            </div>
            <div class="card stage">
                <h2>octocat/hello-world</h2>
                <img src="https://avatars0.githubusercontent.com/u/817538" />
                <span class="connector"></span>
                <span class="status pending"></span>
                <span class="desc">processing the <em>default</em> stage for build <em>#25</em></span>
                <span class="time" datetime="2016-07-07T09:24:17Z"></span>
            </div>
            <div class="card stage">
                <h2>octocat/hello-world</h2>
                <img src="https://avatars0.githubusercontent.com/u/817538" />
                <span class="connector"></span>
                <span class="status running"></span>
                <span class="desc">processing the <em>default</em> stage for build <em>#25</em></span>
                <span class="time" datetime="2016-07-07T09:24:17Z"></span>
            </div>
            <div class="card stage">
                <h2>octocat/hello-world</h2>
                <img src="https://avatars0.githubusercontent.com/u/817538" />
                <span class="connector"></span>
                <span class="status success"></span>
                <span class="desc">processing the <em>default</em> stage for build <em>#25</em></span>
                <span class="time" datetime="2016-07-07T09:24:17Z"></span>
            </div>
        </article>
    </section>
</main>

<footer></footer>

<script>
timeago.render(document.querySelectorAll('.time'));
</script>
</body>
</html>`)

// /reset.css
var file1 = []byte(`html, body, div, span, applet, object, iframe,
h1, h2, h3, h4, h5, h6, p, blockquote, pre,
a, abbr, acronym, address, big, cite, code,
del, dfn, em, img, ins, kbd, q, s, samp,
small, strike, strong, sub, sup, tt, var,
b, u, i, center,
dl, dt, dd, ol, ul, li,
fieldset, form, label, legend,
table, caption, tbody, tfoot, thead, tr, th, td,
article, aside, canvas, details, embed, 
figure, figcaption, footer, header, hgroup, 
menu, nav, output, ruby, section, summary,
time, mark, audio, video {
	margin: 0;
	padding: 0;
	border: 0;
	font-size: 100%;
	font: inherit;
	vertical-align: baseline;
}
/* HTML5 display-role reset for older browsers */
article, aside, details, figcaption, figure, 
footer, header, hgroup, menu, nav, section {
	display: block;
}
body {
	line-height: 1;
}
ol, ul {
	list-style: none;
}
blockquote, q {
	quotes: none;
}
blockquote:before, blockquote:after,
q:before, q:after {
	content: '';
	content: none;
}
table {
	border-collapse: collapse;
	border-spacing: 0;
}`)

// /timeago.js
var file2 = []byte(`!function(t, e) {
    "object" == typeof exports && "undefined" != typeof module ? e(exports) : "function" == typeof define && define.amd ? define(["exports"], e) : e(t.timeago = {})
}(this, function(t) {
    "use strict";
    var f = [60, 60, 24, 7, 365 / 7 / 12, 12]
      , o = function(t) {
        return parseInt(t)
    }
      , n = function(t) {
        return t instanceof Date ? t : !isNaN(t) || /^\d+$/.test(t) ? new Date(o(t)) : (t = (t || "").trim().replace(/\.\d+/, "").replace(/-/, "/").replace(/-/, "/").replace(/(\d)T(\d)/, "$1 $2").replace(/Z/, " UTC").replace(/([\+\-]\d\d)\:?(\d\d)/, " $1$2"),
        new Date(t))
    }
      , s = function(t, e) {
        for (var n = 0, r = t < 0 ? 1 : 0, a = t = Math.abs(t); f[n] <= t && n < f.length; n++)
            t /= f[n];
        return (0 === (n *= 2) ? 9 : 1) < (t = o(t)) && (n += 1),
        e(t, n, a)[r].replace("%s", t)
    }
      , d = function(t, e) {
        return ((e = e ? n(e) : new Date) - n(t)) / 1e3
    }
      , r = "second_minute_hour_day_week_month_year".split("_")
      , a = "秒_分钟_小时_天_周_个月_年".split("_")
      , e = function(t, e) {
        if (0 === e)
            return ["just now", "right now"];
        var n = r[parseInt(e / 2)];
        return 1 < t && (n += "s"),
        ["".concat(t, " ").concat(n, " ago"), "in ".concat(t, " ").concat(n)]
    }
      , i = {
        en_US: e,
        zh_CN: function(t, e) {
            if (0 === e)
                return ["刚刚", "片刻后"];
            var n = a[parseInt(e / 2)];
            return ["".concat(t, " ").concat(n, "前"), "".concat(t, " ").concat(n, "后")]
        }
    }
      , c = function(t) {
        return i[t] || e
    }
      , l = "timeago-tid"
      , u = function(t, e) {
        return t.getAttribute ? t.getAttribute(e) : t.attr ? t.attr(e) : void 0
    }
      , p = function(t) {
        return u(t, l)
    }
      , _ = {}
      , v = function(t) {
        clearTimeout(t),
        delete _[t]
    }
      , h = function t(e, n, r, a) {
        v(p(e));
        var o = d(n, a);
        e.innerHTML = s(o, r);
        var i, c, u = setTimeout(function() {
            t(e, n, r, a)
        }, 1e3 * function(t) {
            for (var e = 1, n = 0, r = Math.abs(t); f[n] <= t && n < f.length; n++)
                t /= f[n],
                e *= f[n];
            return r = (r %= e) ? e - r : e,
            Math.ceil(r)
        }(o), 2147483647);
        _[u] = 0,
        c = u,
        (i = e).setAttribute ? i.setAttribute(l, c) : i.attr && i.attr(l, c)
    };
    t.version = "4.0.0-beta.2",
    t.format = function(t, e, n) {
        var r = d(t, n);
        return s(r, c(e))
    }
    ,
    t.render = function(t, e, n) {
        var r;
        void 0 === t.length && (t = [t]);
        for (var a = 0; a < t.length; a++) {
            r = t[a];
            var o = u(r, "datetime")
              , i = c(e);
            h(r, o, i, n)
        }
        return t
    }
    ,
    t.cancel = function(t) {
        if (t)
            v(p(t));
        else
            for (var e in _)
                v(e)
    }
    ,
    t.register = function(t, e) {
        i[t] = e
    }
    ,
    Object.defineProperty(t, "__esModule", {
        value: !0
    })
});
`)

// /favicon.png
var file3 = []byte{
	0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a, 0x00, 0x00, 0x00, 0x0d,
	0x49, 0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20,
	0x08, 0x03, 0x00, 0x00, 0x00, 0x44, 0xa4, 0x8a, 0xc6, 0x00, 0x00, 0x00,
	0x04, 0x67, 0x41, 0x4d, 0x41, 0x00, 0x00, 0xb1, 0x8f, 0x0b, 0xfc, 0x61,
	0x05, 0x00, 0x00, 0x00, 0x01, 0x73, 0x52, 0x47, 0x42, 0x00, 0xae, 0xce,
	0x1c, 0xe9, 0x00, 0x00, 0x00, 0x20, 0x63, 0x48, 0x52, 0x4d, 0x00, 0x00,
	0x7a, 0x26, 0x00, 0x00, 0x80, 0x84, 0x00, 0x00, 0xfa, 0x00, 0x00, 0x00,
	0x80, 0xe8, 0x00, 0x00, 0x75, 0x30, 0x00, 0x00, 0xea, 0x60, 0x00, 0x00,
	0x3a, 0x98, 0x00, 0x00, 0x17, 0x70, 0x9c, 0xba, 0x51, 0x3c, 0x00, 0x00,
	0x01, 0xd1, 0x50, 0x4c, 0x54, 0x45, 0x00, 0x00, 0x00, 0x19, 0x2d, 0x46,
	0x0e, 0x21, 0x47, 0x1a, 0x2e, 0x47, 0x19, 0x2d, 0x45, 0x14, 0x28, 0x41,
	0x19, 0x2c, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x47, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46, 0x19, 0x2d, 0x46,
	0xff, 0xff, 0xff, 0x7a, 0xc6, 0x05, 0xa1, 0x00, 0x00, 0x00, 0x99, 0x74,
	0x52, 0x4e, 0x53, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x33,
	0x82, 0xb1, 0xd1, 0xf0, 0xfd, 0xef, 0xd0, 0xb0, 0x7f, 0x31, 0x04, 0x68,
	0xe4, 0xe0, 0x9f, 0x44, 0x01, 0x2c, 0xc1, 0xec, 0x96, 0x23, 0x4b, 0x75,
	0x0e, 0x29, 0xbf, 0xca, 0x42, 0x54, 0xe2, 0xfb, 0x8d, 0x0c, 0xde, 0x4d,
	0x49, 0xe1, 0xfa, 0x0d, 0xc0, 0xcd, 0x87, 0x64, 0x66, 0x79, 0xb8, 0xf4,
	0x41, 0x28, 0xce, 0x2a, 0x5b, 0x1a, 0xbb, 0xfe, 0xc9, 0x25, 0x99, 0xa4,
	0x92, 0x4a, 0x81, 0x02, 0x76, 0x39, 0xbd, 0xee, 0x43, 0xa2, 0x84, 0xd9,
	0x17, 0x4f, 0xf2, 0x97, 0x03, 0x35, 0xed, 0x3a, 0x8f, 0x0b, 0xb7, 0x34,
	0x83, 0xb9, 0x3b, 0x70, 0xa3, 0x02, 0x2b, 0xe8, 0x65, 0x73, 0xa1, 0x9c,
	0x37, 0x69, 0xaa, 0x20, 0xdc, 0x5d, 0x7d, 0xd2, 0x18, 0x9e, 0xd4, 0x9a,
	0xf8, 0x50, 0x1b, 0xcf, 0xa8, 0x30, 0xdd, 0xf1, 0xe7, 0xfc, 0xf9, 0x71,
	0x0a, 0x74, 0x80, 0x57, 0x3f, 0x94, 0x6f, 0xac, 0xa6, 0xc4, 0x32, 0xa0,
	0x2d, 0x46, 0x24, 0x3e, 0x48, 0x40, 0xc5, 0x91, 0x8c, 0x21, 0x3d, 0x6e,
	0x35, 0x08, 0xac, 0x2a, 0x00, 0x00, 0x00, 0x01, 0x62, 0x4b, 0x47, 0x44,
	0x9a, 0x98, 0xdf, 0x67, 0x12, 0x00, 0x00, 0x00, 0x09, 0x70, 0x48, 0x59,
	0x73, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00, 0x46, 0xc9,
	0x6b, 0x3e, 0x00, 0x00, 0x02, 0x02, 0x49, 0x44, 0x41, 0x54, 0x38, 0xcb,
	0x7d, 0x93, 0xf9, 0x43, 0x12, 0x41, 0x14, 0xc7, 0xe7, 0x9b, 0xd5, 0x22,
	0xb2, 0xa2, 0x0e, 0x66, 0x05, 0x86, 0x96, 0xb9, 0x88, 0x66, 0x98, 0x99,
	0x69, 0x42, 0x68, 0xd1, 0x25, 0xe6, 0x11, 0x66, 0x1e, 0xa5, 0xa1, 0x1d,
	0x9a, 0x78, 0x90, 0x28, 0x9d, 0xda, 0x69, 0x96, 0x65, 0xa9, 0x1d, 0xf3,
	0xdf, 0x3a, 0xb3, 0x17, 0x8b, 0x62, 0xef, 0x97, 0x99, 0x79, 0xef, 0xb3,
	0xf3, 0xde, 0xce, 0xf7, 0x3d, 0x42, 0x34, 0x03, 0x24, 0x5b, 0xbe, 0xbd,
	0xc0, 0x21, 0xcb, 0x85, 0xce, 0xa2, 0xe2, 0x12, 0x0a, 0xb2, 0xcb, 0x70,
	0xc0, 0x55, 0x2a, 0x33, 0xdd, 0xe4, 0x23, 0x65, 0x47, 0xb3, 0x10, 0xe0,
	0x18, 0x70, 0xdc, 0xcd, 0x2c, 0xe6, 0x29, 0x3f, 0x01, 0x64, 0xe2, 0xde,
	0x8a, 0x4a, 0xe0, 0xe4, 0x29, 0x2b, 0xc1, 0xaa, 0x4e, 0x1b, 0x04, 0x50,
	0xad, 0xf8, 0x6a, 0xfc, 0x7b, 0x88, 0xda, 0x3a, 0x8d, 0x00, 0xce, 0xd4,
	0x33, 0x76, 0xb6, 0x26, 0xc0, 0x89, 0x06, 0xf9, 0x5c, 0xa3, 0xeb, 0x7c,
	0xd3, 0x85, 0xe6, 0x8b, 0x2a, 0xd1, 0xa2, 0x12, 0x68, 0xbd, 0x24, 0x4e,
	0x2a, 0xd1, 0x16, 0x0c, 0xe5, 0x81, 0x7f, 0xe2, 0xf7, 0x5e, 0x0e, 0x73,
	0x5f, 0x7b, 0x07, 0x07, 0x40, 0xaf, 0x68, 0x37, 0x0a, 0x22, 0x53, 0x76,
	0xe8, 0xaa, 0xf0, 0x45, 0xf8, 0xbf, 0xe0, 0x5a, 0xa1, 0x9e, 0xd3, 0x77,
	0xfd, 0x06, 0x80, 0x0e, 0x57, 0xe3, 0xcd, 0x5b, 0x12, 0x10, 0xea, 0xe4,
	0xae, 0x68, 0x17, 0x08, 0xbd, 0x6d, 0x56, 0xd5, 0x7d, 0x10, 0xb4, 0xa9,
	0x87, 0xdf, 0xed, 0x89, 0xf4, 0x02, 0x7d, 0x77, 0xb8, 0x2b, 0xd6, 0x4f,
	0xee, 0x2a, 0x46, 0x7c, 0xe0, 0x1e, 0x10, 0xd4, 0xaf, 0x1b, 0x94, 0x30,
	0x34, 0xcc, 0x57, 0x65, 0x84, 0xdc, 0x0f, 0x1b, 0x80, 0x3b, 0x80, 0xa1,
	0x07, 0xc6, 0x3b, 0x8d, 0x02, 0x83, 0xe2, 0x4d, 0xc7, 0x88, 0xdd, 0xcc,
	0xf0, 0x30, 0x8e, 0xf1, 0x09, 0xe3, 0xf0, 0x08, 0x78, 0x2c, 0xd6, 0x27,
	0xa4, 0xc0, 0x04, 0x9e, 0x02, 0x93, 0x0e, 0xe3, 0x30, 0x05, 0x4c, 0x89,
	0xd5, 0x49, 0x4c, 0x17, 0x7b, 0x46, 0x31, 0x9d, 0x30, 0xe4, 0x9a, 0x01,
	0x66, 0xc5, 0xc6, 0x41, 0x4c, 0x09, 0xd9, 0xdc, 0x3c, 0xfa, 0x93, 0xfa,
	0xfe, 0x39, 0x97, 0x6a, 0x41, 0x25, 0x2d, 0x40, 0x6a, 0x11, 0x28, 0x51,
	0x5f, 0x95, 0x45, 0x83, 0xc0, 0x52, 0x5a, 0x03, 0x32, 0x29, 0xd8, 0x0b,
	0x2e, 0xd8, 0xcb, 0xd8, 0xab, 0xd7, 0x6f, 0xec, 0x5e, 0x0a, 0xb8, 0xde,
	0x6a, 0x29, 0x9c, 0x19, 0x60, 0x79, 0x25, 0x0e, 0xd0, 0xe9, 0xc9, 0xb6,
	0x77, 0x5c, 0x24, 0x8c, 0x37, 0x08, 0x5f, 0x9a, 0x14, 0x59, 0x04, 0x8e,
	0xbe, 0xf7, 0x6b, 0x72, 0x20, 0x4e, 0x75, 0xf5, 0x93, 0xe4, 0x43, 0xd8,
	0x42, 0xa4, 0xec, 0x1f, 0xe7, 0x29, 0xe2, 0x95, 0x9f, 0x3e, 0xe7, 0x07,
	0x54, 0x42, 0x5e, 0x25, 0xb6, 0xfa, 0xac, 0x2e, 0x49, 0xcd, 0x7d, 0x49,
	0xae, 0x7d, 0x75, 0xe8, 0xfd, 0xe1, 0x56, 0xbe, 0x11, 0xba, 0xce, 0x72,
	0x9a, 0x4a, 0x7c, 0x9f, 0x91, 0x08, 0x7e, 0x0c, 0xfc, 0x87, 0x10, 0xd5,
	0xd2, 0x18, 0xdb, 0x9f, 0x38, 0x24, 0x2a, 0xde, 0xa8, 0xda, 0x87, 0xf0,
	0x8d, 0x69, 0x3d, 0x89, 0xae, 0xda, 0xdc, 0x40, 0xe2, 0xa7, 0xd1, 0xd6,
	0x75, 0x39, 0x89, 0xc4, 0x2f, 0x73, 0x30, 0x0e, 0x6f, 0xb6, 0xef, 0x8d,
	0x6f, 0x8d, 0x5a, 0x47, 0x6b, 0x63, 0x3b, 0x9a, 0x1d, 0xf6, 0xfc, 0xfe,
	0x83, 0xac, 0xe9, 0x94, 0x5a, 0xca, 0x15, 0xcb, 0xf0, 0x46, 0xfe, 0x4a,
	0xbb, 0xe7, 0x1b, 0xd4, 0xb6, 0xba, 0xee, 0x14, 0xe3, 0x9f, 0x4e, 0xfe,
	0xb3, 0x8c, 0xff, 0x0e, 0x8e, 0xb4, 0x63, 0x2b, 0x88, 0xb0, 0x0f, 0x8b,
	0x00, 0x00, 0x00, 0x25, 0x74, 0x45, 0x58, 0x74, 0x64, 0x61, 0x74, 0x65,
	0x3a, 0x63, 0x72, 0x65, 0x61, 0x74, 0x65, 0x00, 0x32, 0x30, 0x31, 0x39,
	0x2d, 0x30, 0x32, 0x2d, 0x32, 0x37, 0x54, 0x31, 0x31, 0x3a, 0x35, 0x31,
	0x3a, 0x34, 0x34, 0x2b, 0x30, 0x30, 0x3a, 0x30, 0x30, 0x27, 0x6a, 0xa6,
	0x2a, 0x00, 0x00, 0x00, 0x25, 0x74, 0x45, 0x58, 0x74, 0x64, 0x61, 0x74,
	0x65, 0x3a, 0x6d, 0x6f, 0x64, 0x69, 0x66, 0x79, 0x00, 0x32, 0x30, 0x31,
	0x39, 0x2d, 0x30, 0x32, 0x2d, 0x32, 0x37, 0x54, 0x31, 0x31, 0x3a, 0x35,
	0x31, 0x3a, 0x34, 0x34, 0x2b, 0x30, 0x30, 0x3a, 0x30, 0x30, 0x56, 0x37,
	0x1e, 0x96, 0x00, 0x00, 0x00, 0x46, 0x74, 0x45, 0x58, 0x74, 0x73, 0x6f,
	0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x00, 0x49, 0x6d, 0x61, 0x67, 0x65,
	0x4d, 0x61, 0x67, 0x69, 0x63, 0x6b, 0x20, 0x36, 0x2e, 0x37, 0x2e, 0x38,
	0x2d, 0x39, 0x20, 0x32, 0x30, 0x31, 0x34, 0x2d, 0x30, 0x35, 0x2d, 0x31,
	0x32, 0x20, 0x51, 0x31, 0x36, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f,
	0x2f, 0x77, 0x77, 0x77, 0x2e, 0x69, 0x6d, 0x61, 0x67, 0x65, 0x6d, 0x61,
	0x67, 0x69, 0x63, 0x6b, 0x2e, 0x6f, 0x72, 0x67, 0xdc, 0x86, 0xed, 0x00,
	0x00, 0x00, 0x00, 0x18, 0x74, 0x45, 0x58, 0x74, 0x54, 0x68, 0x75, 0x6d,
	0x62, 0x3a, 0x3a, 0x44, 0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x3a,
	0x3a, 0x50, 0x61, 0x67, 0x65, 0x73, 0x00, 0x31, 0xa7, 0xff, 0xbb, 0x2f,
	0x00, 0x00, 0x00, 0x18, 0x74, 0x45, 0x58, 0x74, 0x54, 0x68, 0x75, 0x6d,
	0x62, 0x3a, 0x3a, 0x49, 0x6d, 0x61, 0x67, 0x65, 0x3a, 0x3a, 0x68, 0x65,
	0x69, 0x67, 0x68, 0x74, 0x00, 0x31, 0x39, 0x32, 0x0f, 0x00, 0x72, 0x85,
	0x00, 0x00, 0x00, 0x17, 0x74, 0x45, 0x58, 0x74, 0x54, 0x68, 0x75, 0x6d,
	0x62, 0x3a, 0x3a, 0x49, 0x6d, 0x61, 0x67, 0x65, 0x3a, 0x3a, 0x57, 0x69,
	0x64, 0x74, 0x68, 0x00, 0x31, 0x39, 0x32, 0xd3, 0xac, 0x21, 0x08, 0x00,
	0x00, 0x00, 0x19, 0x74, 0x45, 0x58, 0x74, 0x54, 0x68, 0x75, 0x6d, 0x62,
	0x3a, 0x3a, 0x4d, 0x69, 0x6d, 0x65, 0x74, 0x79, 0x70, 0x65, 0x00, 0x69,
	0x6d, 0x61, 0x67, 0x65, 0x2f, 0x70, 0x6e, 0x67, 0x3f, 0xb2, 0x56, 0x4e,
	0x00, 0x00, 0x00, 0x17, 0x74, 0x45, 0x58, 0x74, 0x54, 0x68, 0x75, 0x6d,
	0x62, 0x3a, 0x3a, 0x4d, 0x54, 0x69, 0x6d, 0x65, 0x00, 0x31, 0x35, 0x35,
	0x31, 0x32, 0x36, 0x38, 0x33, 0x30, 0x34, 0xd7, 0x02, 0x40, 0x72, 0x00,
	0x00, 0x00, 0x0f, 0x74, 0x45, 0x58, 0x74, 0x54, 0x68, 0x75, 0x6d, 0x62,
	0x3a, 0x3a, 0x53, 0x69, 0x7a, 0x65, 0x00, 0x30, 0x42, 0x42, 0x94, 0xa2,
	0x3e, 0xec, 0x00, 0x00, 0x00, 0x56, 0x74, 0x45, 0x58, 0x74, 0x54, 0x68,
	0x75, 0x6d, 0x62, 0x3a, 0x3a, 0x55, 0x52, 0x49, 0x00, 0x66, 0x69, 0x6c,
	0x65, 0x3a, 0x2f, 0x2f, 0x2f, 0x6d, 0x6e, 0x74, 0x6c, 0x6f, 0x67, 0x2f,
	0x66, 0x61, 0x76, 0x69, 0x63, 0x6f, 0x6e, 0x73, 0x2f, 0x32, 0x30, 0x31,
	0x39, 0x2d, 0x30, 0x32, 0x2d, 0x32, 0x37, 0x2f, 0x36, 0x34, 0x32, 0x39,
	0x36, 0x31, 0x34, 0x63, 0x37, 0x61, 0x38, 0x30, 0x66, 0x37, 0x62, 0x35,
	0x63, 0x39, 0x62, 0x62, 0x36, 0x64, 0x30, 0x35, 0x35, 0x61, 0x64, 0x61,
	0x62, 0x61, 0x65, 0x65, 0x2e, 0x69, 0x63, 0x6f, 0x2e, 0x70, 0x6e, 0x67,
	0x2f, 0x36, 0x62, 0x9b, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45, 0x4e, 0x44,
	0xae, 0x42, 0x60, 0x82,
}

// /icons/arrow-right.svg
var file4 = []byte{
	0x3c, 0x73, 0x76, 0x67, 0x20, 0x76, 0x69, 0x65, 0x77, 0x42, 0x6f, 0x78,
	0x3d, 0x22, 0x30, 0x20, 0x30, 0x20, 0x31, 0x30, 0x20, 0x31, 0x39, 0x22,
	0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73, 0x3d, 0x22, 0x68, 0x74, 0x74, 0x70,
	0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x33, 0x2e, 0x6f, 0x72,
	0x67, 0x2f, 0x32, 0x30, 0x30, 0x30, 0x2f, 0x73, 0x76, 0x67, 0x22, 0x3e,
	0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x67, 0x20, 0x66, 0x69, 0x6c, 0x6c,
	0x3d, 0x22, 0x23, 0x37, 0x33, 0x38, 0x34, 0x39, 0x61, 0x22, 0x20, 0x66,
	0x69, 0x6c, 0x6c, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d, 0x22, 0x6e, 0x6f,
	0x6e, 0x7a, 0x65, 0x72, 0x6f, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x20, 0x64, 0x61,
	0x74, 0x61, 0x2d, 0x76, 0x2d, 0x37, 0x37, 0x66, 0x61, 0x31, 0x31, 0x62,
	0x61, 0x3d, 0x22, 0x22, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x35, 0x2e, 0x36,
	0x35, 0x20, 0x31, 0x2e, 0x35, 0x61, 0x2e, 0x36, 0x35, 0x2e, 0x36, 0x35,
	0x20, 0x30, 0x20, 0x31, 0x20, 0x30, 0x2d, 0x31, 0x2e, 0x33, 0x20, 0x30,
	0x76, 0x31, 0x35, 0x61, 0x2e, 0x36, 0x35, 0x2e, 0x36, 0x35, 0x20, 0x30,
	0x20, 0x31, 0x20, 0x30, 0x20, 0x31, 0x2e, 0x33, 0x20, 0x30, 0x76, 0x2d,
	0x31, 0x35, 0x7a, 0x22, 0x3e, 0x3c, 0x2f, 0x70, 0x61, 0x74, 0x68, 0x3e,
	0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x70, 0x61,
	0x74, 0x68, 0x20, 0x64, 0x61, 0x74, 0x61, 0x2d, 0x76, 0x2d, 0x37, 0x37,
	0x66, 0x61, 0x31, 0x31, 0x62, 0x61, 0x3d, 0x22, 0x22, 0x20, 0x64, 0x3d,
	0x22, 0x4d, 0x31, 0x2e, 0x34, 0x36, 0x20, 0x31, 0x33, 0x2e, 0x30, 0x34,
	0x61, 0x2e, 0x36, 0x35, 0x2e, 0x36, 0x35, 0x20, 0x30, 0x20, 0x31, 0x20,
	0x30, 0x2d, 0x2e, 0x39, 0x32, 0x2e, 0x39, 0x32, 0x4c, 0x34, 0x2e, 0x34,
	0x20, 0x31, 0x37, 0x2e, 0x38, 0x31, 0x38, 0x61, 0x2e, 0x38, 0x35, 0x2e,
	0x38, 0x35, 0x20, 0x30, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x2e, 0x32,
	0x30, 0x32, 0x20, 0x30, 0x4c, 0x39, 0x2e, 0x34, 0x36, 0x20, 0x31, 0x33,
	0x2e, 0x39, 0x36, 0x61, 0x2e, 0x36, 0x35, 0x2e, 0x36, 0x35, 0x20, 0x30,
	0x20, 0x31, 0x20, 0x30, 0x2d, 0x2e, 0x39, 0x32, 0x2d, 0x2e, 0x39, 0x32,
	0x4c, 0x35, 0x20, 0x31, 0x36, 0x2e, 0x35, 0x38, 0x6c, 0x2d, 0x33, 0x2e,
	0x35, 0x34, 0x2d, 0x33, 0x2e, 0x35, 0x34, 0x7a, 0x22, 0x3e, 0x3c, 0x2f,
	0x70, 0x61, 0x74, 0x68, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x2f,
	0x67, 0x3e, 0x0a, 0x3c, 0x2f, 0x73, 0x76, 0x67, 0x3e,
}

// /icons/skipped.svg
var file5 = []byte{
	0x20, 0x20, 0x3c, 0x73, 0x76, 0x67, 0x20, 0x76, 0x69, 0x65, 0x77, 0x42,
	0x6f, 0x78, 0x3d, 0x22, 0x30, 0x20, 0x30, 0x20, 0x32, 0x30, 0x20, 0x32,
	0x30, 0x22, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73, 0x3d, 0x22, 0x68, 0x74,
	0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x33, 0x2e,
	0x6f, 0x72, 0x67, 0x2f, 0x32, 0x30, 0x30, 0x30, 0x2f, 0x73, 0x76, 0x67,
	0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x67, 0x20, 0x66, 0x69,
	0x6c, 0x6c, 0x3d, 0x22, 0x6e, 0x6f, 0x6e, 0x65, 0x22, 0x20, 0x66, 0x69,
	0x6c, 0x6c, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d, 0x22, 0x65, 0x76, 0x65,
	0x6e, 0x6f, 0x64, 0x64, 0x22, 0x20, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x66,
	0x6f, 0x72, 0x6d, 0x3d, 0x22, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x6c, 0x61,
	0x74, 0x65, 0x28, 0x33, 0x2c, 0x33, 0x29, 0x22, 0x3e, 0x0a, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x3c, 0x67, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d,
	0x22, 0x23, 0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x22, 0x20, 0x66, 0x69,
	0x6c, 0x6c, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d, 0x22, 0x6e, 0x6f, 0x6e,
	0x7a, 0x65, 0x72, 0x6f, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x20, 0x64, 0x3d, 0x22,
	0x4d, 0x37, 0x20, 0x31, 0x34, 0x41, 0x37, 0x20, 0x37, 0x20, 0x30, 0x20,
	0x31, 0x20, 0x31, 0x20, 0x37, 0x20, 0x30, 0x61, 0x37, 0x20, 0x37, 0x20,
	0x30, 0x20, 0x30, 0x20, 0x31, 0x20, 0x30, 0x20, 0x31, 0x34, 0x7a, 0x6d,
	0x30, 0x2d, 0x31, 0x2e, 0x34, 0x36, 0x33, 0x41, 0x35, 0x2e, 0x35, 0x33,
	0x37, 0x20, 0x35, 0x2e, 0x35, 0x33, 0x37, 0x20, 0x30, 0x20, 0x31, 0x20,
	0x30, 0x20, 0x37, 0x20, 0x31, 0x2e, 0x34, 0x36, 0x33, 0x61, 0x35, 0x2e,
	0x35, 0x33, 0x37, 0x20, 0x35, 0x2e, 0x35, 0x33, 0x37, 0x20, 0x30, 0x20,
	0x30, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x31, 0x2e, 0x30, 0x37, 0x34,
	0x7a, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x32,
	0x2e, 0x32, 0x35, 0x20, 0x33, 0x2e, 0x34, 0x35, 0x36, 0x41, 0x2e, 0x38,
	0x35, 0x33, 0x2e, 0x38, 0x35, 0x33, 0x20, 0x30, 0x20, 0x31, 0x20, 0x31,
	0x20, 0x33, 0x2e, 0x34, 0x35, 0x36, 0x20, 0x32, 0x2e, 0x32, 0x35, 0x6c,
	0x38, 0x2e, 0x32, 0x39, 0x34, 0x20, 0x38, 0x2e, 0x32, 0x39, 0x34, 0x61,
	0x2e, 0x38, 0x35, 0x33, 0x2e, 0x38, 0x35, 0x33, 0x20, 0x30, 0x20, 0x30,
	0x20, 0x31, 0x2d, 0x31, 0x2e, 0x32, 0x30, 0x36, 0x20, 0x31, 0x2e, 0x32,
	0x30, 0x36, 0x4c, 0x32, 0x2e, 0x32, 0x35, 0x20, 0x33, 0x2e, 0x34, 0x35,
	0x36, 0x7a, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x3c, 0x2f, 0x67, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x2f, 0x67,
	0x3e, 0x0a, 0x20, 0x20, 0x3c, 0x2f, 0x73, 0x76, 0x67, 0x3e,
}

// /icons/pending.svg
var file6 = []byte{
	0x20, 0x20, 0x20, 0x20, 0x3c, 0x73, 0x76, 0x67, 0x20, 0x76, 0x69, 0x65,
	0x77, 0x42, 0x6f, 0x78, 0x3d, 0x22, 0x30, 0x20, 0x30, 0x20, 0x32, 0x30,
	0x20, 0x32, 0x30, 0x22, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73, 0x3d, 0x22,
	0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x77,
	0x33, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x32, 0x30, 0x30, 0x30, 0x2f, 0x73,
	0x76, 0x67, 0x22, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x2d, 0x72, 0x75, 0x6c,
	0x65, 0x3d, 0x22, 0x65, 0x76, 0x65, 0x6e, 0x6f, 0x64, 0x64, 0x22, 0x20,
	0x63, 0x6c, 0x69, 0x70, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d, 0x22, 0x65,
	0x76, 0x65, 0x6e, 0x6f, 0x64, 0x64, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x73, 0x74, 0x72, 0x6f, 0x6b, 0x65, 0x2d,
	0x6c, 0x69, 0x6e, 0x65, 0x6a, 0x6f, 0x69, 0x6e, 0x3d, 0x22, 0x72, 0x6f,
	0x75, 0x6e, 0x64, 0x22, 0x20, 0x73, 0x74, 0x72, 0x6f, 0x6b, 0x65, 0x2d,
	0x6d, 0x69, 0x74, 0x65, 0x72, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x3d, 0x22,
	0x31, 0x2e, 0x34, 0x31, 0x34, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x20, 0x66, 0x69,
	0x6c, 0x6c, 0x3d, 0x22, 0x6e, 0x6f, 0x6e, 0x65, 0x22, 0x20, 0x64, 0x3d,
	0x22, 0x4d, 0x30, 0x20, 0x30, 0x68, 0x32, 0x30, 0x76, 0x32, 0x30, 0x48,
	0x30, 0x7a, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x0a, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x37, 0x2e,
	0x36, 0x38, 0x34, 0x20, 0x35, 0x2e, 0x36, 0x35, 0x35, 0x76, 0x31, 0x2e,
	0x31, 0x34, 0x32, 0x63, 0x2e, 0x30, 0x30, 0x31, 0x2d, 0x2e, 0x30, 0x32,
	0x31, 0x2e, 0x30, 0x30, 0x34, 0x2d, 0x2e, 0x30, 0x31, 0x38, 0x2e, 0x30,
	0x31, 0x33, 0x2d, 0x2e, 0x30, 0x30, 0x32, 0x2e, 0x30, 0x36, 0x33, 0x2e,
	0x31, 0x30, 0x35, 0x2e, 0x31, 0x37, 0x2e, 0x32, 0x35, 0x33, 0x2e, 0x33,
	0x31, 0x38, 0x2e, 0x34, 0x33, 0x37, 0x2e, 0x33, 0x32, 0x33, 0x2e, 0x33,
	0x39, 0x39, 0x2e, 0x38, 0x32, 0x31, 0x2e, 0x39, 0x34, 0x20, 0x31, 0x2e,
	0x34, 0x38, 0x39, 0x20, 0x31, 0x2e, 0x36, 0x31, 0x37, 0x61, 0x2e, 0x39,
	0x32, 0x2e, 0x39, 0x32, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x20, 0x2e,
	0x32, 0x36, 0x37, 0x2e, 0x36, 0x33, 0x32, 0x6c, 0x2e, 0x30, 0x31, 0x38,
	0x2e, 0x39, 0x37, 0x35, 0x61, 0x2e, 0x39, 0x32, 0x38, 0x2e, 0x39, 0x32,
	0x38, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x2d, 0x2e, 0x32, 0x38, 0x37,
	0x2e, 0x36, 0x38, 0x34, 0x63, 0x2d, 0x2e, 0x36, 0x37, 0x2e, 0x36, 0x33,
	0x39, 0x2d, 0x31, 0x2e, 0x31, 0x36, 0x37, 0x20, 0x31, 0x2e, 0x31, 0x35,
	0x36, 0x2d, 0x31, 0x2e, 0x34, 0x38, 0x39, 0x20, 0x31, 0x2e, 0x35, 0x34,
	0x34, 0x61, 0x33, 0x2e, 0x35, 0x35, 0x20, 0x33, 0x2e, 0x35, 0x35, 0x20,
	0x30, 0x20, 0x30, 0x20, 0x30, 0x2d, 0x2e, 0x33, 0x31, 0x31, 0x2e, 0x34,
	0x32, 0x34, 0x63, 0x2d, 0x2e, 0x30, 0x31, 0x35, 0x2e, 0x30, 0x32, 0x36,
	0x2d, 0x2e, 0x30, 0x31, 0x37, 0x2e, 0x30, 0x32, 0x39, 0x2d, 0x2e, 0x30,
	0x31, 0x38, 0x2d, 0x2e, 0x30, 0x30, 0x39, 0x76, 0x31, 0x2e, 0x32, 0x34,
	0x36, 0x68, 0x34, 0x2e, 0x36, 0x33, 0x32, 0x76, 0x2d, 0x31, 0x2e, 0x32,
	0x34, 0x36, 0x63, 0x2d, 0x2e, 0x30, 0x30, 0x31, 0x2e, 0x30, 0x33, 0x38,
	0x2d, 0x2e, 0x30, 0x30, 0x33, 0x2e, 0x30, 0x33, 0x35, 0x2d, 0x2e, 0x30,
	0x31, 0x38, 0x2e, 0x30, 0x30, 0x39, 0x61, 0x33, 0x2e, 0x35, 0x35, 0x20,
	0x33, 0x2e, 0x35, 0x35, 0x20, 0x30, 0x20, 0x30, 0x20, 0x30, 0x2d, 0x2e,
	0x33, 0x31, 0x31, 0x2d, 0x2e, 0x34, 0x32, 0x34, 0x63, 0x2d, 0x2e, 0x33,
	0x32, 0x32, 0x2d, 0x2e, 0x33, 0x38, 0x38, 0x2d, 0x2e, 0x38, 0x31, 0x39,
	0x2d, 0x2e, 0x39, 0x30, 0x35, 0x2d, 0x31, 0x2e, 0x34, 0x38, 0x39, 0x2d,
	0x31, 0x2e, 0x35, 0x34, 0x34, 0x61, 0x2e, 0x39, 0x32, 0x38, 0x2e, 0x39,
	0x32, 0x38, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x2d, 0x2e, 0x32, 0x38,
	0x37, 0x2d, 0x2e, 0x36, 0x38, 0x34, 0x6c, 0x2e, 0x30, 0x31, 0x38, 0x2d,
	0x2e, 0x39, 0x37, 0x35, 0x61, 0x2e, 0x39, 0x32, 0x2e, 0x39, 0x32, 0x20,
	0x30, 0x20, 0x30, 0x20, 0x31, 0x20, 0x2e, 0x32, 0x36, 0x37, 0x2d, 0x2e,
	0x36, 0x33, 0x32, 0x63, 0x2e, 0x36, 0x36, 0x38, 0x2d, 0x2e, 0x36, 0x37,
	0x37, 0x20, 0x31, 0x2e, 0x31, 0x36, 0x36, 0x2d, 0x31, 0x2e, 0x32, 0x31,
	0x38, 0x20, 0x31, 0x2e, 0x34, 0x38, 0x39, 0x2d, 0x31, 0x2e, 0x36, 0x31,
	0x37, 0x2e, 0x31, 0x34, 0x38, 0x2d, 0x2e, 0x31, 0x38, 0x34, 0x2e, 0x32,
	0x35, 0x35, 0x2d, 0x2e, 0x33, 0x33, 0x32, 0x2e, 0x33, 0x31, 0x38, 0x2d,
	0x2e, 0x34, 0x33, 0x37, 0x2e, 0x30, 0x30, 0x39, 0x2d, 0x2e, 0x30, 0x31,
	0x36, 0x2e, 0x30, 0x31, 0x32, 0x2d, 0x2e, 0x30, 0x31, 0x39, 0x2e, 0x30,
	0x31, 0x33, 0x2e, 0x30, 0x30, 0x32, 0x56, 0x35, 0x2e, 0x36, 0x35, 0x35,
	0x48, 0x37, 0x2e, 0x36, 0x38, 0x34, 0x7a, 0x4d, 0x36, 0x20, 0x36, 0x2e,
	0x38, 0x38, 0x34, 0x56, 0x35, 0x2e, 0x31, 0x33, 0x38, 0x43, 0x36, 0x20,
	0x34, 0x2e, 0x35, 0x30, 0x39, 0x20, 0x36, 0x2e, 0x35, 0x31, 0x38, 0x20,
	0x34, 0x20, 0x37, 0x2e, 0x31, 0x35, 0x38, 0x20, 0x34, 0x68, 0x35, 0x2e,
	0x36, 0x38, 0x34, 0x43, 0x31, 0x33, 0x2e, 0x34, 0x38, 0x32, 0x20, 0x34,
	0x20, 0x31, 0x34, 0x20, 0x34, 0x2e, 0x35, 0x30, 0x39, 0x20, 0x31, 0x34,
	0x20, 0x35, 0x2e, 0x31, 0x33, 0x38, 0x76, 0x31, 0x2e, 0x37, 0x34, 0x36,
	0x63, 0x30, 0x20, 0x2e, 0x36, 0x31, 0x35, 0x2d, 0x2e, 0x36, 0x31, 0x36,
	0x20, 0x31, 0x2e, 0x34, 0x30, 0x31, 0x2d, 0x32, 0x2e, 0x30, 0x39, 0x32,
	0x20, 0x32, 0x2e, 0x39, 0x31, 0x31, 0x6c, 0x2d, 0x2e, 0x30, 0x30, 0x37,
	0x2e, 0x33, 0x37, 0x38, 0x43, 0x31, 0x33, 0x2e, 0x33, 0x35, 0x37, 0x20,
	0x31, 0x31, 0x2e, 0x35, 0x38, 0x20, 0x31, 0x34, 0x20, 0x31, 0x32, 0x2e,
	0x33, 0x38, 0x39, 0x20, 0x31, 0x34, 0x20, 0x31, 0x33, 0x2e, 0x30, 0x34,
	0x38, 0x76, 0x31, 0x2e, 0x38, 0x31, 0x34, 0x63, 0x30, 0x20, 0x2e, 0x36,
	0x32, 0x39, 0x2d, 0x2e, 0x35, 0x31, 0x38, 0x20, 0x31, 0x2e, 0x31, 0x33,
	0x38, 0x2d, 0x31, 0x2e, 0x31, 0x35, 0x38, 0x20, 0x31, 0x2e, 0x31, 0x33,
	0x38, 0x48, 0x37, 0x2e, 0x31, 0x35, 0x38, 0x43, 0x36, 0x2e, 0x35, 0x31,
	0x38, 0x20, 0x31, 0x36, 0x20, 0x36, 0x20, 0x31, 0x35, 0x2e, 0x34, 0x39,
	0x31, 0x20, 0x36, 0x20, 0x31, 0x34, 0x2e, 0x38, 0x36, 0x32, 0x76, 0x2d,
	0x31, 0x2e, 0x38, 0x31, 0x34, 0x63, 0x30, 0x2d, 0x2e, 0x36, 0x35, 0x39,
	0x2e, 0x36, 0x34, 0x33, 0x2d, 0x31, 0x2e, 0x34, 0x36, 0x38, 0x20, 0x32,
	0x2e, 0x30, 0x39, 0x39, 0x2d, 0x32, 0x2e, 0x38, 0x37, 0x35, 0x6c, 0x2d,
	0x2e, 0x30, 0x30, 0x37, 0x2d, 0x2e, 0x33, 0x37, 0x38, 0x43, 0x36, 0x2e,
	0x36, 0x31, 0x36, 0x20, 0x38, 0x2e, 0x32, 0x38, 0x35, 0x20, 0x36, 0x20,
	0x37, 0x2e, 0x34, 0x39, 0x39, 0x20, 0x36, 0x20, 0x36, 0x2e, 0x38, 0x38,
	0x34, 0x7a, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d, 0x22, 0x23, 0x46, 0x46, 0x46,
	0x46, 0x46, 0x46, 0x22, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x2d, 0x72, 0x75,
	0x6c, 0x65, 0x3d, 0x22, 0x6e, 0x6f, 0x6e, 0x7a, 0x65, 0x72, 0x6f, 0x22,
	0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x2f, 0x73, 0x76, 0x67,
	0x3e,
}

// /icons/sleeping.svg
var file7 = []byte{
	0x20, 0x20, 0x3c, 0x73, 0x76, 0x67, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73,
	0x3d, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77,
	0x2e, 0x77, 0x33, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x32, 0x30, 0x30, 0x30,
	0x2f, 0x73, 0x76, 0x67, 0x22, 0x20, 0x76, 0x69, 0x65, 0x77, 0x42, 0x6f,
	0x78, 0x3d, 0x22, 0x30, 0x20, 0x30, 0x20, 0x38, 0x30, 0x20, 0x38, 0x30,
	0x22, 0x20, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x3d, 0x22, 0x69, 0x63, 0x6f,
	0x6e, 0x2d, 0x64, 0x72, 0x6f, 0x6e, 0x65, 0x2d, 0x73, 0x6c, 0x65, 0x65,
	0x70, 0x22, 0x20, 0x77, 0x69, 0x64, 0x74, 0x68, 0x3d, 0x22, 0x38, 0x30,
	0x70, 0x78, 0x22, 0x20, 0x68, 0x65, 0x69, 0x67, 0x68, 0x74, 0x3d, 0x22,
	0x38, 0x30, 0x70, 0x78, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c,
	0x6c, 0x69, 0x6e, 0x65, 0x61, 0x72, 0x47, 0x72, 0x61, 0x64, 0x69, 0x65,
	0x6e, 0x74, 0x20, 0x69, 0x64, 0x3d, 0x22, 0x53, 0x56, 0x47, 0x49, 0x44,
	0x5f, 0x31, 0x5f, 0x22, 0x20, 0x67, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e,
	0x74, 0x55, 0x6e, 0x69, 0x74, 0x73, 0x3d, 0x22, 0x75, 0x73, 0x65, 0x72,
	0x53, 0x70, 0x61, 0x63, 0x65, 0x4f, 0x6e, 0x55, 0x73, 0x65, 0x22, 0x20,
	0x78, 0x31, 0x3d, 0x22, 0x31, 0x32, 0x2e, 0x36, 0x30, 0x34, 0x22, 0x20,
	0x79, 0x31, 0x3d, 0x22, 0x36, 0x39, 0x2e, 0x38, 0x34, 0x35, 0x22, 0x20,
	0x78, 0x32, 0x3d, 0x22, 0x36, 0x39, 0x2e, 0x33, 0x39, 0x37, 0x22, 0x20,
	0x79, 0x32, 0x3d, 0x22, 0x31, 0x33, 0x2e, 0x30, 0x35, 0x32, 0x22, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x67, 0x72, 0x61, 0x64,
	0x69, 0x65, 0x6e, 0x74, 0x54, 0x72, 0x61, 0x6e, 0x73, 0x66, 0x6f, 0x72,
	0x6d, 0x3d, 0x22, 0x6d, 0x61, 0x74, 0x72, 0x69, 0x78, 0x28, 0x31, 0x20,
	0x30, 0x20, 0x30, 0x20, 0x2d, 0x31, 0x20, 0x30, 0x20, 0x38, 0x32, 0x2e,
	0x32, 0x29, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c,
	0x73, 0x74, 0x6f, 0x70, 0x20, 0x6f, 0x66, 0x66, 0x73, 0x65, 0x74, 0x3d,
	0x22, 0x30, 0x22, 0x20, 0x73, 0x74, 0x6f, 0x70, 0x2d, 0x63, 0x6f, 0x6c,
	0x6f, 0x72, 0x3d, 0x22, 0x23, 0x62, 0x34, 0x64, 0x33, 0x66, 0x64, 0x22,
	0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x73, 0x74,
	0x6f, 0x70, 0x20, 0x6f, 0x66, 0x66, 0x73, 0x65, 0x74, 0x3d, 0x22, 0x31,
	0x22, 0x20, 0x73, 0x74, 0x6f, 0x70, 0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72,
	0x3d, 0x22, 0x23, 0x38, 0x34, 0x62, 0x30, 0x66, 0x33, 0x22, 0x2f, 0x3e,
	0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x2f, 0x6c, 0x69, 0x6e, 0x65, 0x61,
	0x72, 0x47, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x74, 0x3e, 0x0a, 0x20,
	0x20, 0x20, 0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x0a, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x38, 0x30, 0x2e, 0x32, 0x20,
	0x33, 0x39, 0x2e, 0x39, 0x76, 0x2e, 0x38, 0x63, 0x30, 0x20, 0x31, 0x2e,
	0x34, 0x2d, 0x2e, 0x31, 0x20, 0x32, 0x2e, 0x39, 0x2d, 0x2e, 0x33, 0x20,
	0x34, 0x2e, 0x31, 0x2d, 0x2e, 0x34, 0x20, 0x33, 0x2e, 0x34, 0x2d, 0x31,
	0x2e, 0x32, 0x20, 0x36, 0x2e, 0x36, 0x2d, 0x32, 0x2e, 0x34, 0x20, 0x39,
	0x2e, 0x37, 0x43, 0x37, 0x31, 0x2e, 0x37, 0x20, 0x36, 0x39, 0x2e, 0x34,
	0x20, 0x35, 0x37, 0x2e, 0x32, 0x20, 0x38, 0x30, 0x20, 0x34, 0x30, 0x2e,
	0x32, 0x20, 0x38, 0x30, 0x20, 0x32, 0x32, 0x2e, 0x38, 0x20, 0x38, 0x30,
	0x20, 0x38, 0x2e, 0x31, 0x20, 0x36, 0x39, 0x20, 0x32, 0x2e, 0x36, 0x20,
	0x35, 0x33, 0x2e, 0x36, 0x63, 0x2d, 0x31, 0x2d, 0x32, 0x2e, 0x37, 0x2d,
	0x31, 0x2e, 0x36, 0x2d, 0x35, 0x2e, 0x37, 0x2d, 0x32, 0x2e, 0x31, 0x2d,
	0x38, 0x2e, 0x37, 0x2d, 0x2e, 0x31, 0x2d, 0x31, 0x2e, 0x31, 0x2d, 0x2e,
	0x32, 0x2d, 0x32, 0x2e, 0x33, 0x2d, 0x2e, 0x33, 0x2d, 0x33, 0x2e, 0x35,
	0x56, 0x34, 0x30, 0x63, 0x30, 0x2d, 0x32, 0x2e, 0x31, 0x2e, 0x31, 0x2d,
	0x34, 0x2e, 0x32, 0x2e, 0x34, 0x2d, 0x36, 0x2e, 0x32, 0x43, 0x33, 0x2e,
	0x36, 0x20, 0x31, 0x34, 0x2e, 0x37, 0x20, 0x32, 0x30, 0x2e, 0x32, 0x20,
	0x30, 0x20, 0x34, 0x30, 0x2e, 0x32, 0x20, 0x30, 0x73, 0x33, 0x36, 0x2e,
	0x36, 0x20, 0x31, 0x34, 0x2e, 0x36, 0x20, 0x33, 0x39, 0x2e, 0x35, 0x20,
	0x33, 0x33, 0x2e, 0x38, 0x63, 0x2e, 0x32, 0x20, 0x31, 0x2e, 0x39, 0x2e,
	0x35, 0x20, 0x34, 0x20, 0x2e, 0x35, 0x20, 0x36, 0x2e, 0x31, 0x7a, 0x22,
	0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d,
	0x22, 0x75, 0x72, 0x6c, 0x28, 0x23, 0x53, 0x56, 0x47, 0x49, 0x44, 0x5f,
	0x31, 0x5f, 0x29, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c,
	0x70, 0x61, 0x74, 0x68, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d, 0x22, 0x23,
	0x66, 0x66, 0x66, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x38, 0x30, 0x2e, 0x32, 0x20,
	0x34, 0x30, 0x2e, 0x37, 0x63, 0x30, 0x20, 0x31, 0x2e, 0x34, 0x2d, 0x2e,
	0x31, 0x20, 0x32, 0x2e, 0x39, 0x2d, 0x2e, 0x33, 0x20, 0x34, 0x2e, 0x31,
	0x2d, 0x2e, 0x34, 0x20, 0x33, 0x2e, 0x34, 0x2d, 0x31, 0x2e, 0x32, 0x20,
	0x36, 0x2e, 0x36, 0x2d, 0x32, 0x2e, 0x34, 0x20, 0x39, 0x2e, 0x37, 0x2d,
	0x33, 0x2e, 0x33, 0x20, 0x32, 0x2e, 0x31, 0x2d, 0x37, 0x2e, 0x33, 0x20,
	0x34, 0x2d, 0x31, 0x30, 0x2e, 0x39, 0x20, 0x35, 0x2e, 0x33, 0x2d, 0x38,
	0x2e, 0x36, 0x20, 0x33, 0x2d, 0x31, 0x37, 0x2e, 0x35, 0x20, 0x34, 0x2e,
	0x32, 0x2d, 0x32, 0x36, 0x2e, 0x33, 0x20, 0x34, 0x2e, 0x32, 0x2d, 0x38,
	0x2e, 0x39, 0x2d, 0x2e, 0x31, 0x2d, 0x31, 0x37, 0x2e, 0x38, 0x2d, 0x31,
	0x2e, 0x35, 0x2d, 0x32, 0x36, 0x2e, 0x33, 0x2d, 0x34, 0x2e, 0x37, 0x2d,
	0x34, 0x2d, 0x31, 0x2e, 0x34, 0x2d, 0x37, 0x2e, 0x37, 0x2d, 0x33, 0x2e,
	0x34, 0x2d, 0x31, 0x31, 0x2e, 0x34, 0x2d, 0x35, 0x2e, 0x37, 0x68, 0x2d,
	0x2e, 0x31, 0x63, 0x2d, 0x31, 0x2d, 0x32, 0x2e, 0x37, 0x2d, 0x31, 0x2e,
	0x36, 0x2d, 0x35, 0x2e, 0x37, 0x2d, 0x32, 0x2e, 0x31, 0x2d, 0x38, 0x2e,
	0x37, 0x2d, 0x2e, 0x31, 0x2d, 0x31, 0x2e, 0x31, 0x2d, 0x2e, 0x32, 0x2d,
	0x32, 0x2e, 0x32, 0x2d, 0x2e, 0x32, 0x2d, 0x33, 0x2e, 0x34, 0x76, 0x2d,
	0x31, 0x2e, 0x32, 0x43, 0x35, 0x20, 0x34, 0x34, 0x2e, 0x36, 0x20, 0x31,
	0x31, 0x20, 0x34, 0x37, 0x2e, 0x38, 0x20, 0x31, 0x37, 0x2e, 0x34, 0x20,
	0x34, 0x39, 0x2e, 0x39, 0x63, 0x37, 0x2e, 0x33, 0x20, 0x32, 0x2e, 0x34,
	0x20, 0x31, 0x35, 0x2e, 0x32, 0x20, 0x33, 0x2e, 0x36, 0x20, 0x32, 0x33,
	0x20, 0x33, 0x2e, 0x36, 0x20, 0x37, 0x2e, 0x38, 0x2e, 0x31, 0x20, 0x31,
	0x35, 0x2e, 0x37, 0x2d, 0x2e, 0x39, 0x20, 0x32, 0x33, 0x2d, 0x33, 0x2e,
	0x33, 0x20, 0x36, 0x2e, 0x32, 0x2d, 0x31, 0x2e, 0x39, 0x20, 0x31, 0x32,
	0x2e, 0x35, 0x2d, 0x35, 0x2e, 0x32, 0x20, 0x31, 0x36, 0x2e, 0x38, 0x2d,
	0x39, 0x2e, 0x35, 0x7a, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20,
	0x3c, 0x65, 0x6c, 0x6c, 0x69, 0x70, 0x73, 0x65, 0x20, 0x66, 0x69, 0x6c,
	0x6c, 0x3d, 0x22, 0x23, 0x66, 0x66, 0x66, 0x22, 0x20, 0x63, 0x78, 0x3d,
	0x22, 0x34, 0x30, 0x2e, 0x32, 0x22, 0x20, 0x63, 0x79, 0x3d, 0x22, 0x35,
	0x36, 0x2e, 0x38, 0x22, 0x20, 0x72, 0x78, 0x3d, 0x22, 0x32, 0x31, 0x2e,
	0x33, 0x22, 0x20, 0x72, 0x79, 0x3d, 0x22, 0x31, 0x35, 0x2e, 0x34, 0x22,
	0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x6c, 0x69, 0x6e, 0x65,
	0x61, 0x72, 0x47, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x69,
	0x64, 0x3d, 0x22, 0x53, 0x56, 0x47, 0x49, 0x44, 0x5f, 0x32, 0x5f, 0x22,
	0x20, 0x67, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x74, 0x55, 0x6e, 0x69,
	0x74, 0x73, 0x3d, 0x22, 0x75, 0x73, 0x65, 0x72, 0x53, 0x70, 0x61, 0x63,
	0x65, 0x4f, 0x6e, 0x55, 0x73, 0x65, 0x22, 0x20, 0x78, 0x31, 0x3d, 0x22,
	0x33, 0x32, 0x2e, 0x32, 0x37, 0x35, 0x22, 0x20, 0x79, 0x31, 0x3d, 0x22,
	0x37, 0x33, 0x35, 0x2e, 0x36, 0x30, 0x37, 0x22, 0x20, 0x78, 0x32, 0x3d,
	0x22, 0x34, 0x37, 0x2e, 0x35, 0x33, 0x36, 0x22, 0x20, 0x79, 0x32, 0x3d,
	0x22, 0x37, 0x35, 0x30, 0x2e, 0x38, 0x36, 0x38, 0x22, 0x0a, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x67, 0x72, 0x61, 0x64, 0x69, 0x65,
	0x6e, 0x74, 0x54, 0x72, 0x61, 0x6e, 0x73, 0x66, 0x6f, 0x72, 0x6d, 0x3d,
	0x22, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x6c, 0x61, 0x74, 0x65, 0x28, 0x30,
	0x20, 0x2d, 0x36, 0x38, 0x36, 0x29, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x3c, 0x73, 0x74, 0x6f, 0x70, 0x20, 0x6f, 0x66, 0x66,
	0x73, 0x65, 0x74, 0x3d, 0x22, 0x30, 0x22, 0x20, 0x73, 0x74, 0x6f, 0x70,
	0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x3d, 0x22, 0x23, 0x37, 0x34, 0x61,
	0x32, 0x65, 0x33, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x3c, 0x73, 0x74, 0x6f, 0x70, 0x20, 0x6f, 0x66, 0x66, 0x73, 0x65,
	0x74, 0x3d, 0x22, 0x31, 0x22, 0x20, 0x73, 0x74, 0x6f, 0x70, 0x2d, 0x63,
	0x6f, 0x6c, 0x6f, 0x72, 0x3d, 0x22, 0x23, 0x35, 0x63, 0x38, 0x62, 0x64,
	0x62, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x2f, 0x6c,
	0x69, 0x6e, 0x65, 0x61, 0x72, 0x47, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e,
	0x74, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x65, 0x6c, 0x6c, 0x69,
	0x70, 0x73, 0x65, 0x20, 0x63, 0x78, 0x3d, 0x22, 0x34, 0x30, 0x2e, 0x32,
	0x22, 0x20, 0x63, 0x79, 0x3d, 0x22, 0x35, 0x37, 0x2e, 0x35, 0x22, 0x20,
	0x72, 0x78, 0x3d, 0x22, 0x31, 0x32, 0x2e, 0x34, 0x22, 0x20, 0x72, 0x79,
	0x3d, 0x22, 0x38, 0x2e, 0x39, 0x22, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d,
	0x22, 0x75, 0x72, 0x6c, 0x28, 0x23, 0x53, 0x56, 0x47, 0x49, 0x44, 0x5f,
	0x32, 0x5f, 0x29, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c,
	0x6c, 0x69, 0x6e, 0x65, 0x61, 0x72, 0x47, 0x72, 0x61, 0x64, 0x69, 0x65,
	0x6e, 0x74, 0x20, 0x69, 0x64, 0x3d, 0x22, 0x53, 0x56, 0x47, 0x49, 0x44,
	0x5f, 0x33, 0x5f, 0x22, 0x20, 0x67, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e,
	0x74, 0x55, 0x6e, 0x69, 0x74, 0x73, 0x3d, 0x22, 0x75, 0x73, 0x65, 0x72,
	0x53, 0x70, 0x61, 0x63, 0x65, 0x4f, 0x6e, 0x55, 0x73, 0x65, 0x22, 0x20,
	0x78, 0x31, 0x3d, 0x22, 0x33, 0x31, 0x2e, 0x30, 0x34, 0x22, 0x20, 0x79,
	0x31, 0x3d, 0x22, 0x37, 0x30, 0x30, 0x2e, 0x36, 0x33, 0x36, 0x22, 0x20,
	0x78, 0x32, 0x3d, 0x22, 0x35, 0x38, 0x2e, 0x38, 0x39, 0x34, 0x22, 0x20,
	0x79, 0x32, 0x3d, 0x22, 0x37, 0x32, 0x38, 0x2e, 0x34, 0x39, 0x22, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x67, 0x72, 0x61, 0x64,
	0x69, 0x65, 0x6e, 0x74, 0x54, 0x72, 0x61, 0x6e, 0x73, 0x66, 0x6f, 0x72,
	0x6d, 0x3d, 0x22, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x6c, 0x61, 0x74, 0x65,
	0x28, 0x30, 0x20, 0x2d, 0x36, 0x38, 0x36, 0x29, 0x22, 0x3e, 0x0a, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x73, 0x74, 0x6f, 0x70, 0x20, 0x6f,
	0x66, 0x66, 0x73, 0x65, 0x74, 0x3d, 0x22, 0x30, 0x22, 0x20, 0x73, 0x74,
	0x6f, 0x70, 0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x3d, 0x22, 0x23, 0x37,
	0x34, 0x61, 0x32, 0x65, 0x33, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x3c, 0x73, 0x74, 0x6f, 0x70, 0x20, 0x6f, 0x66, 0x66,
	0x73, 0x65, 0x74, 0x3d, 0x22, 0x31, 0x22, 0x20, 0x73, 0x74, 0x6f, 0x70,
	0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x3d, 0x22, 0x23, 0x35, 0x63, 0x38,
	0x62, 0x64, 0x62, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c,
	0x2f, 0x6c, 0x69, 0x6e, 0x65, 0x61, 0x72, 0x47, 0x72, 0x61, 0x64, 0x69,
	0x65, 0x6e, 0x74, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x70, 0x61,
	0x74, 0x68, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x3d, 0x22,
	0x4d, 0x35, 0x35, 0x2e, 0x38, 0x20, 0x34, 0x30, 0x2e, 0x39, 0x48, 0x35,
	0x30, 0x63, 0x2d, 0x2e, 0x33, 0x20, 0x30, 0x2d, 0x2e, 0x37, 0x2d, 0x2e,
	0x32, 0x2d, 0x2e, 0x38, 0x2d, 0x2e, 0x34, 0x2d, 0x2e, 0x31, 0x2d, 0x2e,
	0x33, 0x2d, 0x2e, 0x31, 0x2d, 0x2e, 0x37, 0x2e, 0x31, 0x2d, 0x2e, 0x39,
	0x6c, 0x34, 0x2e, 0x37, 0x2d, 0x35, 0x2e, 0x38, 0x68, 0x2d, 0x34, 0x2e,
	0x31, 0x63, 0x2d, 0x2e, 0x34, 0x20, 0x30, 0x2d, 0x2e, 0x39, 0x2d, 0x2e,
	0x33, 0x2d, 0x2e, 0x39, 0x2d, 0x2e, 0x39, 0x20, 0x30, 0x2d, 0x2e, 0x35,
	0x2e, 0x33, 0x2d, 0x2e, 0x39, 0x2e, 0x39, 0x2d, 0x2e, 0x39, 0x68, 0x35,
	0x2e, 0x38, 0x63, 0x2e, 0x33, 0x20, 0x30, 0x20, 0x2e, 0x37, 0x2e, 0x32,
	0x2e, 0x38, 0x2e, 0x34, 0x2e, 0x31, 0x2e, 0x33, 0x2e, 0x31, 0x2e, 0x37,
	0x2d, 0x2e, 0x31, 0x2e, 0x39, 0x6c, 0x2d, 0x34, 0x2e, 0x37, 0x20, 0x35,
	0x2e, 0x38, 0x68, 0x34, 0x2e, 0x31, 0x63, 0x2e, 0x34, 0x20, 0x30, 0x20,
	0x2e, 0x39, 0x2e, 0x33, 0x2e, 0x39, 0x2e, 0x39, 0x2d, 0x2e, 0x31, 0x2e,
	0x36, 0x2d, 0x2e, 0x35, 0x2e, 0x39, 0x2d, 0x2e, 0x39, 0x2e, 0x39, 0x7a,
	0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x69, 0x6c, 0x6c,
	0x3d, 0x22, 0x75, 0x72, 0x6c, 0x28, 0x23, 0x53, 0x56, 0x47, 0x49, 0x44,
	0x5f, 0x33, 0x5f, 0x29, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20,
	0x3c, 0x67, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x6c,
	0x69, 0x6e, 0x65, 0x61, 0x72, 0x47, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e,
	0x74, 0x20, 0x69, 0x64, 0x3d, 0x22, 0x53, 0x56, 0x47, 0x49, 0x44, 0x5f,
	0x34, 0x5f, 0x22, 0x20, 0x67, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x74,
	0x55, 0x6e, 0x69, 0x74, 0x73, 0x3d, 0x22, 0x75, 0x73, 0x65, 0x72, 0x53,
	0x70, 0x61, 0x63, 0x65, 0x4f, 0x6e, 0x55, 0x73, 0x65, 0x22, 0x20, 0x78,
	0x31, 0x3d, 0x22, 0x34, 0x33, 0x2e, 0x33, 0x32, 0x22, 0x20, 0x79, 0x31,
	0x3d, 0x22, 0x36, 0x38, 0x38, 0x2e, 0x33, 0x35, 0x35, 0x22, 0x20, 0x78,
	0x32, 0x3d, 0x22, 0x37, 0x31, 0x2e, 0x31, 0x37, 0x34, 0x22, 0x20, 0x79,
	0x32, 0x3d, 0x22, 0x37, 0x31, 0x36, 0x2e, 0x32, 0x31, 0x22, 0x0a, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x67, 0x72, 0x61,
	0x64, 0x69, 0x65, 0x6e, 0x74, 0x54, 0x72, 0x61, 0x6e, 0x73, 0x66, 0x6f,
	0x72, 0x6d, 0x3d, 0x22, 0x74, 0x72, 0x61, 0x6e, 0x73, 0x6c, 0x61, 0x74,
	0x65, 0x28, 0x30, 0x20, 0x2d, 0x36, 0x38, 0x36, 0x29, 0x22, 0x3e, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x73, 0x74, 0x6f,
	0x70, 0x20, 0x6f, 0x66, 0x66, 0x73, 0x65, 0x74, 0x3d, 0x22, 0x30, 0x22,
	0x20, 0x73, 0x74, 0x6f, 0x70, 0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x3d,
	0x22, 0x23, 0x37, 0x34, 0x61, 0x32, 0x65, 0x33, 0x22, 0x2f, 0x3e, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x73, 0x74, 0x6f,
	0x70, 0x20, 0x6f, 0x66, 0x66, 0x73, 0x65, 0x74, 0x3d, 0x22, 0x31, 0x22,
	0x20, 0x73, 0x74, 0x6f, 0x70, 0x2d, 0x63, 0x6f, 0x6c, 0x6f, 0x72, 0x3d,
	0x22, 0x23, 0x35, 0x63, 0x38, 0x62, 0x64, 0x62, 0x22, 0x2f, 0x3e, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x2f, 0x6c, 0x69, 0x6e, 0x65,
	0x61, 0x72, 0x47, 0x72, 0x61, 0x64, 0x69, 0x65, 0x6e, 0x74, 0x3e, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x3d, 0x22, 0x4d,
	0x36, 0x37, 0x2e, 0x34, 0x20, 0x32, 0x38, 0x2e, 0x37, 0x68, 0x2d, 0x38,
	0x2e, 0x38, 0x63, 0x2d, 0x2e, 0x34, 0x20, 0x30, 0x2d, 0x2e, 0x39, 0x2d,
	0x2e, 0x32, 0x2d, 0x31, 0x2e, 0x31, 0x2d, 0x2e, 0x37, 0x2d, 0x2e, 0x32,
	0x2d, 0x2e, 0x34, 0x2d, 0x2e, 0x31, 0x2d, 0x2e, 0x39, 0x2e, 0x31, 0x2d,
	0x31, 0x2e, 0x33, 0x6c, 0x37, 0x2e, 0x33, 0x2d, 0x39, 0x68, 0x2d, 0x36,
	0x2e, 0x34, 0x63, 0x2d, 0x2e, 0x37, 0x20, 0x30, 0x2d, 0x31, 0x2e, 0x32,
	0x2d, 0x2e, 0x35, 0x2d, 0x31, 0x2e, 0x32, 0x2d, 0x31, 0x2e, 0x32, 0x73,
	0x2e, 0x35, 0x2d, 0x31, 0x2e, 0x32, 0x20, 0x31, 0x2e, 0x32, 0x2d, 0x31,
	0x2e, 0x32, 0x68, 0x38, 0x2e, 0x38, 0x63, 0x2e, 0x34, 0x20, 0x30, 0x20,
	0x2e, 0x39, 0x2e, 0x32, 0x20, 0x31, 0x2e, 0x31, 0x2e, 0x37, 0x73, 0x2e,
	0x31, 0x2e, 0x39, 0x2d, 0x2e, 0x31, 0x20, 0x31, 0x2e, 0x33, 0x6c, 0x2d,
	0x37, 0x2e, 0x33, 0x20, 0x39, 0x68, 0x36, 0x2e, 0x33, 0x63, 0x2e, 0x37,
	0x20, 0x30, 0x20, 0x31, 0x2e, 0x32, 0x2e, 0x35, 0x20, 0x31, 0x2e, 0x32,
	0x20, 0x31, 0x2e, 0x32, 0x20, 0x30, 0x20, 0x2e, 0x36, 0x2d, 0x2e, 0x34,
	0x20, 0x31, 0x2e, 0x32, 0x2d, 0x31, 0x2e, 0x31, 0x20, 0x31, 0x2e, 0x32,
	0x7a, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66,
	0x69, 0x6c, 0x6c, 0x3d, 0x22, 0x75, 0x72, 0x6c, 0x28, 0x23, 0x53, 0x56,
	0x47, 0x49, 0x44, 0x5f, 0x34, 0x5f, 0x29, 0x22, 0x2f, 0x3e, 0x0a, 0x20,
	0x20, 0x20, 0x20, 0x3c, 0x2f, 0x67, 0x3e, 0x0a, 0x20, 0x20, 0x3c, 0x2f,
	0x73, 0x76, 0x67, 0x3e,
}

// /icons/running.svg
var file8 = []byte{
	0x20, 0x20, 0x3c, 0x73, 0x76, 0x67, 0x20, 0x76, 0x69, 0x65, 0x77, 0x42,
	0x6f, 0x78, 0x3d, 0x22, 0x30, 0x20, 0x30, 0x20, 0x32, 0x30, 0x20, 0x32,
	0x30, 0x22, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73, 0x3d, 0x22, 0x68, 0x74,
	0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x33, 0x2e,
	0x6f, 0x72, 0x67, 0x2f, 0x32, 0x30, 0x30, 0x30, 0x2f, 0x73, 0x76, 0x67,
	0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x67, 0x20, 0x66, 0x69,
	0x6c, 0x6c, 0x3d, 0x22, 0x6e, 0x6f, 0x6e, 0x65, 0x22, 0x20, 0x66, 0x69,
	0x6c, 0x6c, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d, 0x22, 0x65, 0x76, 0x65,
	0x6e, 0x6f, 0x64, 0x64, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x3c, 0x63, 0x69, 0x72, 0x63, 0x6c, 0x65, 0x20, 0x63, 0x78, 0x3d,
	0x22, 0x31, 0x30, 0x22, 0x20, 0x63, 0x79, 0x3d, 0x22, 0x31, 0x30, 0x22,
	0x20, 0x72, 0x3d, 0x22, 0x35, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x0a, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x35, 0x2e, 0x39,
	0x37, 0x34, 0x20, 0x32, 0x2e, 0x34, 0x33, 0x31, 0x63, 0x2e, 0x34, 0x38,
	0x31, 0x2d, 0x2e, 0x32, 0x35, 0x37, 0x2e, 0x39, 0x39, 0x2d, 0x2e, 0x34,
	0x36, 0x39, 0x20, 0x31, 0x2e, 0x35, 0x32, 0x31, 0x2d, 0x2e, 0x36, 0x33,
	0x4c, 0x38, 0x2e, 0x30, 0x39, 0x35, 0x20, 0x30, 0x68, 0x33, 0x2e, 0x38,
	0x31, 0x6c, 0x2e, 0x36, 0x20, 0x31, 0x2e, 0x38, 0x63, 0x2e, 0x35, 0x33,
	0x2e, 0x31, 0x36, 0x32, 0x20, 0x31, 0x2e, 0x30, 0x34, 0x2e, 0x33, 0x37,
	0x34, 0x20, 0x31, 0x2e, 0x35, 0x32, 0x31, 0x2e, 0x36, 0x33, 0x31, 0x6c,
	0x31, 0x2e, 0x36, 0x39, 0x38, 0x2d, 0x2e, 0x38, 0x34, 0x39, 0x20, 0x32,
	0x2e, 0x36, 0x39, 0x34, 0x20, 0x32, 0x2e, 0x36, 0x39, 0x34, 0x2d, 0x2e,
	0x38, 0x35, 0x20, 0x31, 0x2e, 0x36, 0x39, 0x38, 0x63, 0x2e, 0x32, 0x35,
	0x38, 0x2e, 0x34, 0x38, 0x31, 0x2e, 0x34, 0x37, 0x2e, 0x39, 0x39, 0x2e,
	0x36, 0x33, 0x32, 0x20, 0x31, 0x2e, 0x35, 0x32, 0x31, 0x6c, 0x31, 0x2e,
	0x38, 0x2e, 0x36, 0x76, 0x33, 0x2e, 0x38, 0x31, 0x6c, 0x2d, 0x31, 0x2e,
	0x38, 0x2e, 0x36, 0x61, 0x38, 0x2e, 0x35, 0x31, 0x38, 0x20, 0x38, 0x2e,
	0x35, 0x31, 0x38, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x2d, 0x2e, 0x36,
	0x33, 0x31, 0x20, 0x31, 0x2e, 0x35, 0x32, 0x31, 0x6c, 0x2e, 0x38, 0x34,
	0x39, 0x20, 0x31, 0x2e, 0x36, 0x39, 0x38, 0x2d, 0x32, 0x2e, 0x36, 0x39,
	0x34, 0x20, 0x32, 0x2e, 0x36, 0x39, 0x34, 0x2d, 0x31, 0x2e, 0x36, 0x39,
	0x38, 0x2d, 0x2e, 0x38, 0x35, 0x63, 0x2d, 0x2e, 0x34, 0x38, 0x31, 0x2e,
	0x32, 0x35, 0x38, 0x2d, 0x2e, 0x39, 0x39, 0x2e, 0x34, 0x37, 0x2d, 0x31,
	0x2e, 0x35, 0x32, 0x31, 0x2e, 0x36, 0x33, 0x32, 0x6c, 0x2d, 0x2e, 0x36,
	0x20, 0x31, 0x2e, 0x38, 0x68, 0x2d, 0x33, 0x2e, 0x38, 0x31, 0x6c, 0x2d,
	0x2e, 0x36, 0x2d, 0x31, 0x2e, 0x38, 0x61, 0x38, 0x2e, 0x35, 0x31, 0x38,
	0x20, 0x38, 0x2e, 0x35, 0x31, 0x38, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31,
	0x2d, 0x31, 0x2e, 0x35, 0x32, 0x31, 0x2d, 0x2e, 0x36, 0x33, 0x31, 0x6c,
	0x2d, 0x31, 0x2e, 0x36, 0x39, 0x38, 0x2e, 0x38, 0x34, 0x39, 0x2d, 0x32,
	0x2e, 0x36, 0x39, 0x34, 0x2d, 0x32, 0x2e, 0x36, 0x39, 0x34, 0x2e, 0x38,
	0x35, 0x2d, 0x31, 0x2e, 0x36, 0x39, 0x38, 0x61, 0x38, 0x2e, 0x35, 0x31,
	0x38, 0x20, 0x38, 0x2e, 0x35, 0x31, 0x38, 0x20, 0x30, 0x20, 0x30, 0x20,
	0x31, 0x2d, 0x2e, 0x36, 0x33, 0x32, 0x2d, 0x31, 0x2e, 0x35, 0x32, 0x31,
	0x6c, 0x2d, 0x31, 0x2e, 0x38, 0x2d, 0x2e, 0x36, 0x76, 0x2d, 0x33, 0x2e,
	0x38, 0x31, 0x6c, 0x31, 0x2e, 0x38, 0x2d, 0x2e, 0x36, 0x63, 0x2e, 0x31,
	0x36, 0x32, 0x2d, 0x2e, 0x35, 0x33, 0x2e, 0x33, 0x37, 0x34, 0x2d, 0x31,
	0x2e, 0x30, 0x34, 0x2e, 0x36, 0x33, 0x31, 0x2d, 0x31, 0x2e, 0x35, 0x32,
	0x31, 0x6c, 0x2d, 0x2e, 0x38, 0x34, 0x39, 0x2d, 0x31, 0x2e, 0x36, 0x39,
	0x38, 0x20, 0x32, 0x2e, 0x36, 0x39, 0x34, 0x2d, 0x32, 0x2e, 0x36, 0x39,
	0x34, 0x20, 0x31, 0x2e, 0x36, 0x39, 0x38, 0x2e, 0x38, 0x35, 0x7a, 0x4d,
	0x31, 0x30, 0x20, 0x31, 0x34, 0x61, 0x34, 0x20, 0x34, 0x20, 0x30, 0x20,
	0x31, 0x20, 0x30, 0x20, 0x30, 0x2d, 0x38, 0x20, 0x34, 0x20, 0x34, 0x20,
	0x30, 0x20, 0x30, 0x20, 0x30, 0x20, 0x30, 0x20, 0x38, 0x7a, 0x22, 0x0a,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x69, 0x6c, 0x6c,
	0x3d, 0x22, 0x23, 0x66, 0x66, 0x62, 0x65, 0x30, 0x30, 0x22, 0x2f, 0x3e,
	0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x2f, 0x67, 0x3e, 0x0a, 0x20, 0x20,
	0x3c, 0x2f, 0x73, 0x76, 0x67, 0x3e,
}

// /icons/failure.svg
var file9 = []byte{
	0x20, 0x20, 0x3c, 0x73, 0x76, 0x67, 0x20, 0x76, 0x69, 0x65, 0x77, 0x42,
	0x6f, 0x78, 0x3d, 0x22, 0x30, 0x20, 0x30, 0x20, 0x32, 0x30, 0x20, 0x32,
	0x30, 0x22, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73, 0x3d, 0x22, 0x68, 0x74,
	0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x33, 0x2e,
	0x6f, 0x72, 0x67, 0x2f, 0x32, 0x30, 0x30, 0x30, 0x2f, 0x73, 0x76, 0x67,
	0x22, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73, 0x3a, 0x73, 0x65, 0x72, 0x69,
	0x66, 0x3d, 0x22, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77,
	0x77, 0x2e, 0x73, 0x65, 0x72, 0x69, 0x66, 0x2e, 0x63, 0x6f, 0x6d, 0x2f,
	0x22, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d,
	0x22, 0x65, 0x76, 0x65, 0x6e, 0x6f, 0x64, 0x64, 0x22, 0x0a, 0x20, 0x20,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x63, 0x6c, 0x69, 0x70, 0x2d, 0x72, 0x75,
	0x6c, 0x65, 0x3d, 0x22, 0x65, 0x76, 0x65, 0x6e, 0x6f, 0x64, 0x64, 0x22,
	0x20, 0x73, 0x74, 0x72, 0x6f, 0x6b, 0x65, 0x2d, 0x6c, 0x69, 0x6e, 0x65,
	0x6a, 0x6f, 0x69, 0x6e, 0x3d, 0x22, 0x72, 0x6f, 0x75, 0x6e, 0x64, 0x22,
	0x20, 0x73, 0x74, 0x72, 0x6f, 0x6b, 0x65, 0x2d, 0x6d, 0x69, 0x74, 0x65,
	0x72, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x3d, 0x22, 0x31, 0x2e, 0x34, 0x31,
	0x34, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x70, 0x61, 0x74,
	0x68, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d, 0x22, 0x6e, 0x6f, 0x6e, 0x65,
	0x22, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x30, 0x20, 0x30, 0x68, 0x32, 0x30,
	0x76, 0x32, 0x30, 0x48, 0x30, 0x7a, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20,
	0x20, 0x20, 0x3c, 0x67, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d, 0x22, 0x23,
	0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x22, 0x20, 0x66, 0x69, 0x6c, 0x6c,
	0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d, 0x22, 0x6e, 0x6f, 0x6e, 0x7a, 0x65,
	0x72, 0x6f, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x3c,
	0x70, 0x61, 0x74, 0x68, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x35, 0x2e, 0x34,
	0x33, 0x34, 0x20, 0x36, 0x2e, 0x35, 0x36, 0x36, 0x61, 0x2e, 0x38, 0x30,
	0x32, 0x2e, 0x38, 0x30, 0x32, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x20,
	0x31, 0x2e, 0x31, 0x33, 0x32, 0x2d, 0x31, 0x2e, 0x31, 0x33, 0x32, 0x6c,
	0x37, 0x2e, 0x37, 0x37, 0x38, 0x20, 0x37, 0x2e, 0x37, 0x37, 0x38, 0x61,
	0x2e, 0x38, 0x30, 0x32, 0x2e, 0x38, 0x30, 0x32, 0x20, 0x30, 0x20, 0x30,
	0x20, 0x31, 0x2d, 0x31, 0x2e, 0x31, 0x33, 0x32, 0x20, 0x31, 0x2e, 0x31,
	0x33, 0x32, 0x4c, 0x35, 0x2e, 0x34, 0x33, 0x34, 0x20, 0x36, 0x2e, 0x35,
	0x36, 0x36, 0x7a, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x20, 0x73, 0x65, 0x72, 0x69, 0x66,
	0x3a, 0x69, 0x64, 0x3d, 0x22, 0x4c, 0x69, 0x6e, 0x65, 0x2d, 0x31, 0x33,
	0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x31, 0x33, 0x2e, 0x32, 0x31, 0x32,
	0x20, 0x35, 0x2e, 0x34, 0x33, 0x34, 0x61, 0x2e, 0x38, 0x30, 0x32, 0x2e,
	0x38, 0x30, 0x32, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x20, 0x31, 0x2e,
	0x31, 0x33, 0x32, 0x20, 0x31, 0x2e, 0x31, 0x33, 0x32, 0x6c, 0x2d, 0x37,
	0x2e, 0x37, 0x37, 0x38, 0x20, 0x37, 0x2e, 0x37, 0x37, 0x38, 0x61, 0x2e,
	0x38, 0x30, 0x32, 0x2e, 0x38, 0x30, 0x32, 0x20, 0x30, 0x20, 0x30, 0x20,
	0x31, 0x2d, 0x31, 0x2e, 0x31, 0x33, 0x32, 0x2d, 0x31, 0x2e, 0x31, 0x33,
	0x32, 0x6c, 0x37, 0x2e, 0x37, 0x37, 0x38, 0x2d, 0x37, 0x2e, 0x37, 0x37,
	0x38, 0x7a, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x2f,
	0x67, 0x3e, 0x0a, 0x20, 0x20, 0x3c, 0x2f, 0x73, 0x76, 0x67, 0x3e,
}

// /icons/success.svg
var file10 = []byte{
	0x20, 0x20, 0x3c, 0x73, 0x76, 0x67, 0x20, 0x76, 0x69, 0x65, 0x77, 0x42,
	0x6f, 0x78, 0x3d, 0x22, 0x30, 0x20, 0x30, 0x20, 0x32, 0x30, 0x20, 0x32,
	0x30, 0x22, 0x20, 0x78, 0x6d, 0x6c, 0x6e, 0x73, 0x3d, 0x22, 0x68, 0x74,
	0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x77, 0x33, 0x2e,
	0x6f, 0x72, 0x67, 0x2f, 0x32, 0x30, 0x30, 0x30, 0x2f, 0x73, 0x76, 0x67,
	0x22, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d,
	0x22, 0x65, 0x76, 0x65, 0x6e, 0x6f, 0x64, 0x64, 0x22, 0x20, 0x63, 0x6c,
	0x69, 0x70, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d, 0x22, 0x65, 0x76, 0x65,
	0x6e, 0x6f, 0x64, 0x64, 0x22, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x73, 0x74, 0x72, 0x6f, 0x6b, 0x65, 0x2d, 0x6c, 0x69, 0x6e, 0x65,
	0x6a, 0x6f, 0x69, 0x6e, 0x3d, 0x22, 0x72, 0x6f, 0x75, 0x6e, 0x64, 0x22,
	0x20, 0x73, 0x74, 0x72, 0x6f, 0x6b, 0x65, 0x2d, 0x6d, 0x69, 0x74, 0x65,
	0x72, 0x6c, 0x69, 0x6d, 0x69, 0x74, 0x3d, 0x22, 0x31, 0x2e, 0x34, 0x31,
	0x34, 0x22, 0x3e, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x3c, 0x70, 0x61, 0x74,
	0x68, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d, 0x22, 0x6e, 0x6f, 0x6e, 0x65,
	0x22, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x30, 0x20, 0x30, 0x68, 0x32, 0x30,
	0x76, 0x32, 0x30, 0x48, 0x30, 0x7a, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20,
	0x20, 0x20, 0x3c, 0x70, 0x61, 0x74, 0x68, 0x0a, 0x20, 0x20, 0x20, 0x20,
	0x20, 0x20, 0x64, 0x3d, 0x22, 0x4d, 0x31, 0x34, 0x2e, 0x35, 0x37, 0x37,
	0x20, 0x36, 0x2e, 0x32, 0x33, 0x61, 0x2e, 0x38, 0x38, 0x37, 0x2e, 0x38,
	0x38, 0x37, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x20, 0x31, 0x2e, 0x31,
	0x37, 0x2d, 0x2e, 0x30, 0x31, 0x39, 0x2e, 0x37, 0x30, 0x34, 0x2e, 0x37,
	0x30, 0x34, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x20, 0x2e, 0x30, 0x32,
	0x31, 0x20, 0x31, 0x2e, 0x30, 0x36, 0x33, 0x6c, 0x2d, 0x36, 0x2e, 0x38,
	0x34, 0x34, 0x20, 0x36, 0x2e, 0x34, 0x33, 0x39, 0x2d, 0x2e, 0x30, 0x32,
	0x35, 0x2e, 0x30, 0x32, 0x33, 0x61, 0x31, 0x2e, 0x31, 0x31, 0x20, 0x31,
	0x2e, 0x31, 0x31, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31, 0x2d, 0x31, 0x2e,
	0x34, 0x36, 0x33, 0x2d, 0x2e, 0x30, 0x32, 0x33, 0x6c, 0x2d, 0x33, 0x2e,
	0x32, 0x30, 0x34, 0x2d, 0x33, 0x2e, 0x30, 0x31, 0x35, 0x61, 0x2e, 0x37,
	0x30, 0x34, 0x2e, 0x37, 0x30, 0x34, 0x20, 0x30, 0x20, 0x30, 0x20, 0x31,
	0x20, 0x2e, 0x30, 0x32, 0x31, 0x2d, 0x31, 0x2e, 0x30, 0x36, 0x33, 0x2e,
	0x38, 0x38, 0x37, 0x2e, 0x38, 0x38, 0x37, 0x20, 0x30, 0x20, 0x30, 0x20,
	0x31, 0x20, 0x31, 0x2e, 0x31, 0x37, 0x2e, 0x30, 0x31, 0x39, 0x6c, 0x32,
	0x2e, 0x37, 0x35, 0x37, 0x20, 0x32, 0x2e, 0x35, 0x39, 0x34, 0x20, 0x36,
	0x2e, 0x33, 0x39, 0x37, 0x2d, 0x36, 0x2e, 0x30, 0x31, 0x38, 0x7a, 0x22,
	0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x66, 0x69, 0x6c, 0x6c, 0x3d,
	0x22, 0x23, 0x46, 0x46, 0x46, 0x46, 0x46, 0x46, 0x22, 0x20, 0x66, 0x69,
	0x6c, 0x6c, 0x2d, 0x72, 0x75, 0x6c, 0x65, 0x3d, 0x22, 0x6e, 0x6f, 0x6e,
	0x7a, 0x65, 0x72, 0x6f, 0x22, 0x2f, 0x3e, 0x0a, 0x20, 0x20, 0x3c, 0x2f,
	0x73, 0x76, 0x67, 0x3e,
}

// /style.css
var file11 = []byte(`:root {
    --font-sans: -apple-system,BlinkMacSystemFont,"Segoe UI","Roboto","Oxygen","Ubuntu","Cantarell","Fira Sans","Droid Sans","Helvetica Neue",sans-serif;
    --font-mono: Menlo,Monaco,Lucida Console,Liberation Mono,DejaVu Sans Mono,Bitstream Vera Sans Mono,Courier New,monospace;
}

html, body {
    background: #f7f8fa;
    color: #1e375a;
    width: 100%;
    height: 100%;
    font-family: var(--font-sans);
    font-size: 15px;
}

main {
    box-sizing: border-box;
    max-width: 800px;
    margin: 0px auto;
    margin-bottom: 30px;
}

main section > header h1 {
    height: 41px;
    font-size: 30px;
    font-weight: 400;
    font-style: normal;
    font-stretch: normal;
    line-height: normal;
    letter-spacing: normal;
    color: #1e375a;
    margin: 30px 0;
}

body > header {
    height: 56px;
    box-shadow: 0 2px 4px 0 rgba(30,55,90,.05);
    box-sizing: border-box;
    background-color: rgba(30,55,90,.97);
    padding: 0 15px;
    display: flex;
    align-items: center;
}

header .logo {
    width: 30px;
    height: 30px;
}

.navbar .inline-nav {
    display: flex;
    flex: 1 1 auto;
    justify-content: flex-end;
}

.navbar .inline-nav li {
    display: inline-block;
    margin-left: 15px;
}

.navbar .inline-nav a,
.navbar .inline-nav a:active,
.navbar .inline-nav a:visited {
    color: #FFF;
    opacity: 0.75;
    text-decoration: none;
}

.navbar .inline-nav a.active,
.navbar .inline-nav a:hover {
    opacity: 1;
}

.card {
    box-shadow: 0 2px 4px 0 rgba(30,55,90,.1);
    box-sizing: border-box;
    border: 1px solid rgba(30,55,90,.05);
    border-radius: 4px;
    background-color: #fff;
    margin-bottom: 10px;
}

a.card {
    text-decoration: none;
}

/*
 * stage card component
 */

.stage {
    padding: 15px;
    display: grid;
    grid-gap: 10px 0px;
    grid-template-columns: 30px 30px auto 150px;
}

.stage img {
    border-radius: 50%;
    width: 20px;
    height: 20px;
    grid-row: 2;
    grid-column: 2;
}

.stage h2 {
    color: #1e375a;
    font-size: 18px;
    grid-row: 1;
    grid-column: 2 / span 3;
    overflow: hidden;
    white-space: nowrap;
    text-overflow: ellipsis;
}

.stage .desc {
    color: rgba(30,55,90,.6);
    font-size: 14px;
    grid-row: 2;
    grid-column: 3;
    align-items: center;
    display: flex;
}

.stage time,
.stage .time {
    color: rgba(30,55,90,.6);
    font-size: 14px;
    grid-row: 2;
    grid-column: 4;
    text-align: right;
    display: flex;
    align-items: center;
    justify-content: flex-end;
}

.stage em {
    background-color: rgba(35,100,210,.07);
    border-radius: 2px;
    color: #2364d2;
    padding: 0 4px;
    margin: 0 4px;
    line-height: 17px;
}

.stage .connector {
    width: 15px;
    height: 15px;
    opacity: .2;
    border-bottom-left-radius: 8px;
    border-left: 1px solid #1e375a;
    border-bottom: 1px solid #1e375a;
    display: block;
    grid-row: 2;
    grid-column: 1;
    margin-top: -4px;
    margin-left: 9px;
}

/*
 * step components
 */

.steps header {
    border-bottom: 1px solid rgba(30,55,90,.1);
    display: grid;
    grid-gap: 10px 0px;
    grid-template-columns: 30px auto;
    padding: 10px 15px;
}

.steps header .name {
    align-items: center;
    display: flex;
}

.steps .step {
    display: grid;
    grid-gap: 10px 0px;
    grid-template-columns: 30px auto 100px;
    padding: 10px 15px;
}

.steps .step:hover {
    background: #f8f8fa;
}

.steps .step .name {
    align-items: center;
    display: flex;
}

.steps .body {
    padding: 10px 0px;
}

.steps .step {
    position: relative;
}

.steps .step:before,
.steps .step:after {
    border-left: 1px solid #1e375a;
    content: " ";
    height: 10px;
    left: 25px;
    opacity: 0.15;
    position: absolute;
    width: 1px;
}

.steps .step:after {
    bottom: 0px;
}

.steps .step:before {
    top: 0px;
}

.steps .step:last-of-type:after,
.steps .step:first-of-type:before {
    display: none;
}

.steps .status-name {
    align-items: center;
    color: rgba(30,55,90,.6);
    display: flex;
    font-size: 14px;
    font-style: italic;
    justify-content: flex-end;
    text-transform: capitalize;
}

/**
 * breadcrumb component
 */

.breadcrumb {
    margin-top: 35px;
}

.breadcrumb ul {
    display: flex;
}

.breadcrumb ul li {
    display: flex;
    align-items: center;
}

.breadcrumb ul li.separator {
    margin: 0px 15px;
    width: 19px;
    height: 19px;
    display: inline-block;
    background: url(/static/icons/arrow-right.svg) no-repeat center center;
    transform: rotate(270deg);
}

.breadcrumb a,
.breadcrumb a:active,
.breadcrumb a:visited {
    text-decoration: none;
    color: #2364d2;
    display: block;
    line-height: 19px;
}

.breadcrumb a:hover {
    text-decoration: underline;
}

/**
 * alert components
 */

.alert.sleeping {
    box-shadow: 0 2px 4px 0 rgba(30,55,90,.1);
    box-sizing: border-box;
    border: 1px solid rgba(30,55,90,.05);
    border-radius: 4px;
    margin-bottom: 10px;
    padding: 30px;
    padding-top: 130px;
    text-align: center;
    background-color: #FFFFFF;
    background-image: url(icons/sleeping.svg);
    background-size: 80px;
    background-position-x: center;
    background-position-y: 30px;
    background-repeat: no-repeat;
}

/**
 * status component.
 */

.status {
    background-color: #ff4164;
    border-radius: 50%;
    display: block;
    width: 20px;
    height: 20px;
    background-image: url('icons/failure.svg');
}

.status.skipped {
    background-color: #96a5be;
    background-image: url('icons/skipped.svg');
}

.status.pending {
    background-color: #96a5be;
    background-image: url('icons/pending.svg');
    animation: wrench 2s ease infinite;
}

.status.running {
    background-image: url('icons/running.svg');
    background-color: transparent;
    animation: spin 2s linear infinite;
}

.status.success,
.status.passing {
    background-color: #19d78c;
    background-image: url('icons/success.svg');
}

/*
 * log entry components
 */

.logs {
    box-shadow: 0 2px 4px 0 rgba(30,55,90,.1);
    box-sizing: border-box;
    border: 1px solid rgba(30,55,90,.05);
    border-radius: 4px;
    background-color: #fff;
    margin-bottom: 10px;   
}

.entry {
    padding: 15px;
    display: grid;
    grid-gap: 5px 10px;
    grid-template-columns: 70px auto 150px;
    border-bottom: 1px solid rgba(30,55,90,.05);
}

.entry:last-of-type {
    border-bottom: none;
}

.entry .level {
    grid-row: 1;
    grid-column: 1;
    max-height: 18px;
}

.entry .message {
    align-items: center;
    background: #eaedf2;
    border-radius: 3px;
    display: flex;
    line-height: 18px;
    padding: 0px 5px;
    font-family: var(--font-mono);
    font-size: 13px;
    grid-row: 1;
    grid-column: 2;
}

.entry .fields {
    font-family: var(--font-mono);
    font-size: 13px;
    grid-row: 2;
    grid-column: 2;
    display: inline-grid;
    grid-row-gap: 5px;
}

.entry .time,
.entry time {
    color: rgba(30,55,90,.6);
    font-size: 14px;
    grid-row: 1;
    grid-column: 3;
    text-align: right;
    display: flex;
    align-items: center;
    justify-content: flex-end;
}

.level {
    border-radius: 3px;
    display: block;
    font-size: 11px;
    font-family: var(--font-mono);
    line-height: 18px;
    min-width: 50px;
    text-align: center;
    text-transform: uppercase;
}

.level.panic,
.level.error {
    background: #ff4164;
    color: #FFF;
}

.level.warn {
    background: #ffbe00;
    color: #FFF;
}

.level.info {
    background: #2364d2;
    color: #fff;
}

.level.debug {
    background: #96a5be;
    color: #fff;
}

.level.trace {
    background: #96a5be;
    color: #fff;
}

.fields span {
    background: #eaedf2;
    border-radius: 3px;
    padding: 3px 5px;
    font-size: 11px;
    color: #8d96a8;
}

.fields span em:after {
    content: "=";
    opacity: 0.5;
    margin: 0px 3px;
}

/*
 * animations
 */

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(359deg); }
}

@keyframes wrench {
    0%  { transform: rotate(-12deg); }
    8%  { transform: rotate(12deg); }
    10% { transform: rotate(24deg); }
    18% { transform: rotate(-24deg); }
    20% { transform: rotate(-24deg); }
    28% { transform: rotate(24deg); }
    30% { transform: rotate(24deg); }
    38% { transform: rotate(-24deg); }
    40% { transform: rotate(-24deg); }
    48% { transform: rotate(24deg); }
    50% { transform: rotate(24deg); }
    58% { transform: rotate(-24deg); }
    60% { transform: rotate(-24deg); }
    68% { transform: rotate(24deg); }
    75%, 100% { transform: rotate(0deg); }
}`)

// /logs.html
var file12 = []byte(`<!DOCTYPE html>
<html>
<head>
<meta charset="UTF-8">
<title>Drone Runner Dashboard</title>
<link rel="stylesheet" type="text/css" href="reset.css">
<link rel="stylesheet" type="text/css" href="style.css">
<script src="timeago.js" type="text/javascript"></script>
</head>
<body>

<header class="navbar">
    <div class="logo">
        <svg viewBox="0 0 60 60" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><defs><path d="M12.086 5.814l-.257.258 10.514 10.514C20.856 18.906 20 21.757 20 25c0 9.014 6.618 15 15 15 3.132 0 6.018-.836 8.404-2.353l10.568 10.568C48.497 55.447 39.796 60 30 60 13.434 60 0 46.978 0 30 0 19.903 4.751 11.206 12.086 5.814zm5.002-2.97C20.998 1.015 25.378 0 30 0c16.566 0 30 13.022 30 30 0 4.67-1.016 9.04-2.835 12.923l-9.508-9.509C49.144 31.094 50 28.243 50 25c0-9.014-6.618-15-15-15-3.132 0-6.018.836-8.404 2.353l-9.508-9.508zM35 34c-5.03 0-9-3.591-9-9s3.97-9 9-9c5.03 0 9 3.591 9 9s-3.97 9-9 9z" id="a"></path></defs><use fill="#FFF" xlink:href="#a" fill-rule="evenodd"></use></svg>
    </div>
    <nav class="inline-nav">
        <ul>
            <li><a href="/">Dashboard</a></li>
            <li><a href="/logs">Logging</a></li>
        </ul>
    </nav>
</header>

<main>
    <section>
        <header>
            <h1>Recent Logs</h1>
        </header>
        <div class="alert sleeping">
            <p>There is no recent log activity to display.</p>
        </div>
        <article class="logs">
            <div class="entry">
                <span class="level trace">trace</span>
                <span class="message">received stage</span>
                <span class="fields">
                    <span><em>stage.id</em>95</span>
                    <span><em>stage.name</em>test</span>
                    <span><em>stage.number</em>1</span>
                </span>
                <span class="time" datetime="0"></span>
            </div>
            <div class="entry">
                <span class="level debug">debug</span>
                <span class="message">received stage</span>
                <span class="fields">
                    <span><em>stage.id</em>95</span>
                    <span><em>stage.name</em>test</span>
                    <span><em>stage.number</em>1</span>
                </span>
                <span class="time" datetime="0"></span>
            </div>
            <div class="entry">
                <span class="level warn">warn</span>
                <span class="message">received stage</span>
                <span class="fields">
                    <span><em>stage.id</em>95</span>
                    <span><em>stage.name</em>test</span>
                    <span><em>stage.number</em>1</span>
                </span>
                <span class="time" datetime="0"></span>
            </div>
            <div class="entry">
                <span class="level error">error</span>
                <span class="message">received stage</span>
                <span class="fields">
                    <span><em>stage.id</em>95</span>
                    <span><em>stage.name</em>test</span>
                    <span><em>stage.number</em>1</span>
                </span>
                <span class="time" datetime="0"></span>
            </div>
            <div class="entry">
                <span class="level info">info</span>
                <span class="message">received stage</span>
                <span class="fields">
                    <span><em>stage.id</em>95</span>
                    <span><em>stage.name</em>test</span>
                    <span><em>stage.number</em>1</span>
                </span>
                <span class="time" datetime="0"></span>
            </div>
        </article>
    </section>
</main>

<footer></footer>

<script>
timeago.render(document.querySelectorAll('.time'));
</script>
</body>
</html>`)
