// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief Charged particle spectra for a range of energies
  class TASSO_1982_I177174 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(TASSO_1982_I177174);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(ChargedFinalState(), "FS");

      size_t ih = 1;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en+"x2"], 2, 1, ih);
        book(_h[en+"x3"], 3, 1, ih);
        if (en == "14"s || en == "22"s) {
          book(_h[en+"x1"], 1, 1, ih-1);
        }
        else if (en == "30"s) {
          book(_h["x1"], 1, 1, 3);
        }
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // First, veto on leptonic events by requiring at least 4 charged FS particles
      const ChargedFinalState& fs = apply<ChargedFinalState>(event, "FS");
      const size_t numParticles = fs.particles().size();

      // Even if we only generate hadronic events, we still need a cut on numCharged >= 2.
      if (numParticles < 2)  vetoEvent;

      // Get beams and average beam momentum
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      const double meanBeamMom = 0.5*(beams.first.p3().mod() + beams.second.p3().mod());
      MSG_DEBUG("Avg beam momentum = " << meanBeamMom);

      for (const Particle& p : fs.particles()) {
        double xp = p.p3().mod()/meanBeamMom;
        if (_sqs == "14"s || _sqs == "22"s)  _h[_sqs+"x1"]->fill(xp);
        else if (sqrtS() > 29*GeV)  _h["x1"]->fill(xp);
        _h[_sqs+"x2"]->fill(xp);
        _h[_sqs+"x3"]->fill(xp);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for (auto& item : _h) {
        if (item.first.find("x3") != string::npos) {
          scale(item.second, 1.0/sumOfWeights());
        }
        else {
          // The sqrtS-dependence is not reentrant safe
          scale(item.second, crossSection()/microbarn/sumOfWeights()*sqr(sqrtS()));
        }
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    string _sqs = "";
    /// @}


  };


  RIVET_DECLARE_PLUGIN(TASSO_1982_I177174);
}
